/*
    This file is part of wmpload - a program to monitor ppp activity for X
    Copyright (C) 1999  Matt Smith <mdsmith@engr.utk.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* wmpload.c */
#include <stdlib.h>		/* exit */
#include <stdio.h>		/* printf */
#include <string.h>		/* strlen */
#include <unistd.h>		/* sleep, select, gettimeofday */
#include <math.h>		/* log10 */
#include <ctype.h>		/* toupper */
#include <sys/time.h>		/* select, gettimeofday */
#include <sys/types.h>		/* select */
#include <sys/select.h>		/* select ? */
#include <errno.h>
#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>
#include "wmgeneral.h"
#include "wmpload-master.xpm"
#include "wmpload.h"

/* a few defines */
enum { WMPLOAD_TOTAL, WMPLOAD_RATE, WMPLOAD_DEVICE, WMPLOAD_MAX, WMPLOAD_NONE};
enum { GREEN_COLOR, RED_COLOR };
#define DEFAULT_DEVICE		"ppp0"
#define DEFAULT_AVERAGE		10
#define DEFAULT_DIV		2048.0F
#define DEFAULT_UPDATE_MSEC	1000
#define DEFAULT_GREEN_IS_IN	1
#define DEFAULT_LOGSCALE	0
#ifdef LINUXPROC
#define DEFAULT_NOPROC		0
#endif
#define DEFAULT_FIELD1		WMPLOAD_TOTAL
#define DEFAULT_FIELD2		WMPLOAD_RATE

#define BlitGreenString(a,b,c)	BlitString(GREEN_COLOR, a, b, c)
#define BlitRedString(a,b,c)	BlitString(RED_COLOR, a, b, c)
#define HIST_SIZE		57

/* globals */
static char *Progname;
static char *Progver;
static char wmpload_mask_bits[64*64];
static int wmpload_mask_width = 64;
static int wmpload_mask_height = 64;
static if_data ifd;

typedef struct ScreenDat
{
	char  greenField1[16];
	char  greenField2[16];
	char  redField1[16];
	char  redField2[16];
	float greenHistory[HIST_SIZE];
	float redHistory[HIST_SIZE];
	float max;
	int   online;
} ScreenData;
static ScreenData *sd;

typedef struct Opt
{
	char  *device;
	int   average;
	float div;
	int   update_msec;
	int   green_is_in;
	int   logscale;
#ifdef LINUXPROC
	int   noproc;
#endif
	int   field1;
	int   field2;
} Options;
static Options opt;

/* function prototypes */
static void DrawScreen(ScreenData *s);
static void UpdateChartHistory( ScreenData *s, float gpercent, float rpercent);
static void DrawDivLines(int div);
static void do_total(char *buf, unsigned long t);
static void do_rate(char *buf, float r);
static void Displayfunc(void);
static void parse_cmdline(int argc, char *argv[]);
static void wmpload_routine(int argc, char *argv[]);
static int  NextEventOrTimeout(XEvent *e, unsigned long msec);
static void ProcessEvent(XEvent *e);
static void BlitString(int color, char *name, int x, int y);
static void ChartGreenPercent(float percent, int column);
static void ChartRedPercent(float percent, int column);
static void ChartFrontGreenPercent(float percent, int column);
static void ChartFrontRedPercent(float percent, int column);
static void usage(void);

void DrawScreen(ScreenData *s)
{
	int i;
	
	for(i=0;i<HIST_SIZE;i++)
	{
		/* put history on chart */
		if (s->greenHistory[i] > s->redHistory[i])
		{
			/* green in back */
			ChartGreenPercent(
				s->greenHistory[i]/(float)((int)s->max+1), i+3);
			ChartFrontRedPercent(
				s->redHistory[i]/(float)((int)s->max+1), i+3);
		}
		else
		{
			/* red in back */
			ChartRedPercent(
				s->redHistory[i]/(float)((int)s->max+1), i+3);
			ChartFrontGreenPercent(
				s->greenHistory[i]/(float)((int)s->max+1), i+3);
		}
	}
	
	/* BlankTextArea() */
	BlitGreenString("                             ", 2, 4);
	
	/* give preference to download data */
	if (opt.green_is_in)
	{
		if (s->redHistory[HIST_SIZE-1] > s->greenHistory[HIST_SIZE-1])
		{
			BlitRedString(s->redField1, 2, 4);
			if(s->online)
			{
				BlitRedString(s->redField2, 28, 4);
				BlitRedString("+", 52, 4);
			}
		}
		else
		{
			BlitGreenString(s->greenField1, 2, 4);
			if(s->online)
			{
				BlitGreenString(s->greenField2, 28, 4);
				BlitGreenString("-", 52, 4);
			}
		}
	}
	else
	{
		if (s->redHistory[HIST_SIZE-1] < s->greenHistory[HIST_SIZE-1])
		{
			BlitGreenString(s->greenField1, 2, 4);
			if(s->online)
			{
				BlitGreenString(s->greenField2, 28, 4);
				BlitGreenString("+", 52, 4);
			}
		}
		else
		{
			BlitRedString(s->redField1, 2, 4);
			if(s->online)
			{
				BlitRedString(s->redField2, 28, 4);
				BlitRedString("-", 52, 4);
			}
		}
	}
	
	DrawDivLines((int)(s->max+1));
	return;
}

void UpdateChartHistory( ScreenData *s, float gpercent, float rpercent)
{
	int i;
	s->max = 0.0F;
	for(i=0;i<HIST_SIZE;i++)
	{
		if (i!=(HIST_SIZE-1))
		{
			/* slide all back one */
			s->greenHistory[i] = s->greenHistory[i+1];
			s->redHistory[i] = s->redHistory[i+1];
		}
		else
		{
			/* newest at leading edge */
			s->greenHistory[i] = gpercent;
			s->redHistory[i] = rpercent;
		}
		
		if (s->greenHistory[i] > s->max) s->max = s->greenHistory[i];
		if (s->redHistory[i] > s->max) s->max = s->redHistory[i];
	}
	return;
}


void DrawDivLines(int div)
{
	int top = 18;
	int bottom = 59;
	int height = bottom-top;
	int i;
	
	if (div < 2) return;
	
	for(i=1;i<div;i++)
		copyXPMArea(66, 14, 57, 1, 3, i*height/div+top);
	return;
}

void do_total(char *buf, unsigned long t)
{
	if (t == 0UL)
		sprintf(buf, "offline");
	else if(t < 1000UL)			/* 999 max */
		sprintf(buf, "%ld", t);
	else if (t < 10189UL)			/* 9.9k max */
		sprintf(buf, "%1.1fk", (float)t/1024.0F);
	else if (t < 1024UL*1000UL)		/* 999k max */
		sprintf(buf, "%1.0fk", (float)t/1024.0F);
	else if (t < 10189UL*1024UL)		/* 9.9m max */
		sprintf(buf, "%1.1fm", (float)t/1024.0F/1024.0F);
	else if (t < 1024UL*1024UL*1000UL)	/* 999m max */
		sprintf(buf, "%1.0fm", (float)t/1024.0F/1024.0F);
	else
		sprintf(buf, "%1.1fg", (float)t/1024.0F/1024.0F/1024.0F);
	return;
}

void do_rate(char *buf, float r)
{
	if (r < 1000.0F)			/* 999 max */
		sprintf(buf, "%1.0f", r);
	else if (r < 10189.0F)			/* 9.9 k/s max */
		sprintf(buf, "%1.1fk", r/1024.0F);
	else if (r < 1024.0F*1000.0F)		/* 999 k/s max */
		sprintf(buf, "%1.0fk", r/1024.0F);
	else if (r < 10189.0F*1024.0F)		/* 9.9 M/s max */
		sprintf(buf, "%1.1fm", r/1024.0F/1024.0F);
	else if (r < 1024.0F*1024.0F*1000.0F)	/* 999 M/s max */
		sprintf(buf, "%1.0fm", r/1024.0F/1024.0F);
	else
		sprintf(buf, "%1.1fg", r/1024.0F/1024.0F/1024.0F);
	return;
}

void Displayfunc(void)
{
	unsigned long g, r;
	float gr, rr, gm, rm, gpoint, rpoint;
	
	get_stat(&ifd);
	
	if (opt.green_is_in)
	{
		g = ifd.in_bytes;
		r = ifd.out_bytes;
		gr = (float)ifd.in_rate;
		rr = (float)ifd.out_rate;
		gm = (float)ifd.in_max;
		rm = (float)ifd.out_max;
	}
	else
	{
		g = ifd.out_bytes;
		r = ifd.in_bytes;
		gr = (float)ifd.out_rate;
		rr = (float)ifd.in_rate;
		gm = (float)ifd.out_max;
		rm = (float)ifd.in_max;
	}
	
	gpoint = gr / opt.div;
	rpoint = rr / opt.div;
	if (opt.logscale)
	{
		gpoint = (gpoint < 1.0F) ? 0.0F : log10(gpoint);
		rpoint = (rpoint < 1.0F) ? 0.0F : log10(rpoint);
	}
	
	UpdateChartHistory(sd, gpoint, rpoint);
	
	switch(opt.field1)
	{
		case WMPLOAD_TOTAL:
			do_total(sd->greenField1, g);
			do_total(sd->redField1, r);
			break;
		case WMPLOAD_RATE:
			if (g) do_rate(sd->greenField1, gr);
			if (r) do_rate(sd->redField1, rr);
			break;
		case WMPLOAD_DEVICE:
			strncpy(sd->greenField1, opt.device,
				sizeof(sd->greenField1));
			strncpy(sd->redField1, opt.device,
				sizeof(sd->redField1));
			break;
		case WMPLOAD_MAX:
			if (g) do_rate(sd->greenField1, gm);
			if (r) do_rate(sd->redField1, rm);
			break;
		default:
			break;
	}
	
	switch(opt.field2)
	{
		case WMPLOAD_TOTAL:
			do_total(sd->greenField2, g);
			do_total(sd->redField2, r);
			break;
		case WMPLOAD_RATE:
			if (g) do_rate(sd->greenField2, gr);
			if (r) do_rate(sd->redField2, rr);
			break;
		case WMPLOAD_DEVICE:
			strncpy(sd->greenField2, opt.device,
				sizeof(sd->greenField2));
			strncpy(sd->redField2, opt.device,
				sizeof(sd->redField2));
			break;
		case WMPLOAD_MAX:
			if (g) do_rate(sd->greenField2, gm);
			if (r) do_rate(sd->redField2, rm);
			break;
		default:
			break;
	}
	
	if( g || r)
		sd->online = 1;
	else
		sd->online = 0;
	
	DrawScreen(sd);
	return;
}

void parse_cmdline(int argc, char *argv[])
{
	int i;
	
	for (i=1; i<argc; i++)
	{
		char *arg = argv[i];
		
		if (*arg=='-')
		{
			switch (arg[1])
			{
			case 'a':
				if (strcmp(arg+1, "average"))
					usage();
				
				if( argc > (i+1))
				{
					opt.average =
						atoi(argv[i+1]);
					i++;
				}
				break;
			case 'd' :
				if (!strcmp(arg+1, "div") && argc > (i+1))
				{
					opt.div = atof(argv[i+1]);
					i++;
					break;
				}else
				if (!strcmp(arg+1, "device") && argc > (i+1))
				{
					opt.device = strdup(argv[i+1]);
					i++;
					break;
				}

				if (strcmp(arg+1, "display"))
					usage();

				break;
			case 'f':
				if (!strcmp(arg+1, "field1") && argc>(i+1))
				{
					if (!strcmp(argv[i+1], "total"))
					{
						opt.field1 = WMPLOAD_TOTAL;
						i++;
						break;
					}
					else if(!strcmp(argv[i+1], "rate"))
					{
						opt.field1 = WMPLOAD_RATE;
						i++;
						break;
					}
					else if(!strcmp(argv[i+1], "device"))
					{
						opt.field1 = WMPLOAD_DEVICE;
						i++;
						break;
					}
					else if(!strcmp(argv[i+1], "max"))
					{
						opt.field1 = WMPLOAD_MAX;
						i++;
						break;
					}
					else if(!strcmp(argv[i+1], "none"))
					{
						opt.field1 = WMPLOAD_NONE;
						i++;
						break;
					}
					else
						usage();
				}
				else if (!strcmp(arg+1, "field2") && argc>(i+1))
				{
					if (!strcmp(argv[i+1], "total"))
					{
						opt.field2 = WMPLOAD_TOTAL;
						i++;
						break;
					}
					else if(!strcmp(argv[i+1], "rate"))
					{
						opt.field2 = WMPLOAD_RATE;
						i++;
						break;
					}
					else if(!strcmp(argv[i+1], "device"))
					{
						opt.field2 = WMPLOAD_DEVICE;
						i++;
						break;
					}
					else if(!strcmp(argv[i+1], "max"))
					{
						opt.field2 = WMPLOAD_MAX;
						i++;
						break;
					}
					else if(!strcmp(argv[i+1], "none"))
					{
						opt.field2 = WMPLOAD_NONE;
						i++;
						break;
					}
					else
						usage();
				}
				else
					usage();
				break;
			case 'g':
				if (!strcmp(arg+1, "greenin"))
				{
					opt.green_is_in = 1;
					break;
				}else
				if (!strcmp(arg+1, "greenout"))
				{
					opt.green_is_in = 0;
					break;
				}
				else
				if (strcmp(arg+1, "geometry"))
					usage();

				break;
			case 'l':
				if (!strcmp(arg+1, "logscale"))
				{
					opt.logscale = 1;
					break;
				}
				else
					usage();
				break;
#ifdef LINUXPROC				
			case 'n':
				if (!strcmp(arg+1, "noproc"))
				{
					opt.noproc = 1;
					break;
				}
				else
					usage();
				break;
#endif				
			case 'r':
				if (!strcmp(arg+1, "redin"))
				{
					opt.green_is_in = 0;
					break;
				}
				else	
				if (!strcmp(arg+1, "redout"))
				{
					opt.green_is_in = 1;
					break;
				}
				else
					usage();
				break;
			case 'u':
				if (strcmp(arg+1, "update"))
					usage();

				if (argc > (i+1))
				{
					opt.update_msec =
						atoi(argv[i+1]);
					i++;
				}
				break;
			case 'h':
			case 'v':
			default:
				usage();
				break;
			}
		}
	}
	return;
}


int main(int argc, char *argv[])
{
	Progname = argv[0];
	Progver = VERSION;
	
	opt.device = DEFAULT_DEVICE;
	opt.average = DEFAULT_AVERAGE;
	opt.div = DEFAULT_DIV;
	opt.update_msec = DEFAULT_UPDATE_MSEC;
	opt.green_is_in = DEFAULT_GREEN_IS_IN;
	opt.logscale = DEFAULT_LOGSCALE;
#ifdef LINUXPROC
	opt.noproc = DEFAULT_NOPROC;
#endif
	opt.field1 = DEFAULT_FIELD1;
	opt.field2 = DEFAULT_FIELD2;
	
	parse_cmdline(argc, argv);
	
	sd = (ScreenData*)calloc(sizeof(ScreenData), 1);
	if (sd == NULL) die("calloc");
	
	/**************** setup device **************************/
#ifdef LINUXPROC
	ifd_initialize(&ifd, opt.device, opt.average, !opt.noproc);
#else
	ifd_initialize(&ifd, opt.device, opt.average, 0);
#endif	
		
	wmpload_routine(argc, argv);
	return 0;
}

void wmpload_routine(int argc, char *argv[])
{
	int delay_left, actual;
	struct timeval tv;
	XEvent Event;
	
	createXBMfromXPM(wmpload_mask_bits,
			wmpload_master_xpm,
			wmpload_mask_width,
			wmpload_mask_height);
	openXwindow(	argc, argv,
			wmpload_master_xpm,
			wmpload_mask_bits,
			wmpload_mask_width,
			wmpload_mask_height,
			"wmPload");
	
	Displayfunc();
	
	delay_left = opt.update_msec;
	while (1)
	{
		int start, stop;
		
		RedrawWindow();
		
		if(delay_left >= 0)
		{
			gettimeofday(&tv, NULL);
			start = tv.tv_sec*1000 + tv.tv_usec/1000;
			
			if (!NextEventOrTimeout(&Event, delay_left))
			{
				Displayfunc();
				/* timeout - no X events */
				delay_left = opt.update_msec;
				continue;
			}
			else
			{
				/* an X event occured */
				gettimeofday(&tv, NULL);
				stop = tv.tv_sec*1000 + tv.tv_usec/1000;
				actual = stop-start;
				/* decrease delay_left by time actually taken */
				delay_left -= actual;
				if (delay_left < 0) delay_left = 0;
			}
		}
		else
			XNextEvent(display, &Event);
		
		ProcessEvent(&Event);
	}
	return;
}

int NextEventOrTimeout(XEvent *e, unsigned long msec)
{
	struct timeval timeout;
	fd_set rset;
	
	XSync(display, False);
	if (XPending(display))
	{
		XNextEvent(display, e);
		return True;
	}
	
	timeout.tv_sec = msec/1000;
	timeout.tv_usec = (msec%1000)*1000;
	
	FD_ZERO(&rset);
	FD_SET(ConnectionNumber(display), &rset);
	
	if (select(ConnectionNumber(display)+1, &rset, NULL, NULL, &timeout) > 0)
	{
		XNextEvent(display, e);
		return True;
    	}
	return False;
}

void ProcessEvent(XEvent *e)
{
	switch (e->type)
	{
		case Expose:
			RedrawWindow();
			break;
		case DestroyNotify:
			ifd_stop(&ifd);
			XCloseDisplay(display);
			exit(0);
		case ButtonPress:
			break;
		case VisibilityNotify:
			RedrawWindow();
			break;
		default:
			break;
	}
	return;
}

/* Blits a string at given coordinates */
void BlitString(int color, char *name, int x, int y)
{
	int i, c, k;
	int a, b, u, d;
	
	if (color == GREEN_COLOR)
	{
		a = 0;
		b = 0;
		u = 80;
		d = 86;
	}
	else
	{
		a = 35;
		b = 8;
		u = 92;
		d = 98;
	}
	
	k = x;
	for (i=0; name[i]; i++)
	{
		c = toupper(name[i]); 
		if (c >= 'A' && c <= 'Z')
		{
			/* its a letter */
			c -= 'A';
			copyXPMArea(c * 6, 74+a, 6, 8, k, y);
			k += 6;
		}
		else if (c>='0' && c<='9')
		{
			/* its a number or symbol */
			c -= '0';
			copyXPMArea(c * 6, 64+a, 6, 8, k, y);
			k += 6;
		}
		else if (c == ' ')
		{
			copyXPMArea(72, 20, 2, 8, k, y);
			k += 2;
		}
		else if (c == '.')
		{
			copyXPMArea(2+b, 96, 1, 1, k+1, y+7);
			copyXPMArea(1+b, 96, 1, 1, k+1, y+6);
			k+=2;
		}
		else if (c == '+')	/* up arrow */
		{
			copyXPMArea(u, 64+35, 6, 8, k, y);
			k+= 6;
		}
		else if (c == '-')	/* down arrow */
		{
			copyXPMArea(d, 64+35, 6, 8, k, y);
			k+= 6;
		}
	}
	return;
}


void ChartGreenPercent(float percent, int column)
{
	int pix_height = 43;
	int pix_width = 42;
	int pix_loc_x = 66;
	int pix_loc_y = 17;
	int col;
	
	if (percent > 1.0) percent = 1.0;
	if (percent < 0.0) percent = 0.0;
	
	col = (int)((float)pix_width * percent + 0.5) + pix_loc_x;
	
	copyXPMArea(col, pix_loc_y, 1, pix_height, column, 17);
	return;
}

void ChartRedPercent(float percent, int column)
{
	int pix_height = 43;
	int pix_width = 42;
	int pix_loc_x = 66+44;
	int pix_loc_y = 17;
	int col;
	
	if (percent > 1.0) percent = 1.0;
	if (percent < 0.0) percent = 0.0;
	
	col = (int)((float)pix_width * percent + 0.5) + pix_loc_x;
	
	copyXPMArea(col, pix_loc_y, 1, pix_height, column, 17); 
	return;
}

void ChartFrontGreenPercent(float percent, int column)
{
	int pix_height = 43;
	int pix_width = 42;
	int pix_loc_x = 66;
	int pix_loc_y = 17;
	int col;
	int top;
	int height;
	
	if (percent > 1.0) percent = 1.0;
	if (percent < 0.0) percent = 0.0;
	
	height = (int)((float)pix_width * percent + 0.5);
	
	col = height + pix_loc_x;
	
	top = pix_loc_y + (pix_height-height);
	
	copyXPMArea(col, top, 1, height+0, column, 17+43-height);
	return;
}

void ChartFrontRedPercent(float percent, int column)
{
	int pix_height = 43;
	int pix_width = 42;
	int pix_loc_x = 66+44;
	int pix_loc_y = 17;
	int col;
	int top;
	int height;
	
	if (percent > 1.0) percent = 1.0;
	if (percent < 0.0) percent = 0.0;
	
	height = (int)((float)pix_width * percent + 0.5);
	
	col = height + pix_loc_x;
	
	top = pix_loc_y + (pix_height-height);
	
	copyXPMArea(col, top, 1, height+0, column, 17+43-height);
	return;
}

void usage(void)
{
	fprintf(stderr,
"%s version %s : show data for a network interface (ppp)\n"
"Usage : %s [options]\n"
" where options includes\n"
" option         default     description\n"
" -average <n> : 10        : average data over n points, 1 for no average\n"
" -device <X>  : ppp0      : monitor device X\n"
" -display <d> : default   : use X11 display d\n"
" -div <n>     : 2048      : show dividing lines every n bytes/s\n"
" -field1 <X>  : total     : show X where X is one of\n"
"                            [total|rate|max|device|none] in first text label\n"
" -field2 <X>  : rate      : show X where X is one of\n"
"                            [total|rate|max|device|none] in second text label\n"
" -geometry <g>: default   : use X11 geometry spec g\n"
" -greenin     : True      : use green color for inbound data\n"
" -logscale    : False     : use base 10 logarithmic scale for charts\n"
#ifdef LINUXPROC
" -noproc      : False     : don't use proc interface, use ioctl\n"
#endif
" -redin       : False     : use red color for inbound data\n"
" -update <n>  : 1000      : update display every n milliseconds\n\n"

"Note that -redout works the same as -greenin, similarly -greenout\n"
"has the same effect as -redin\n\n"

"See the manual page for more detail\n",
	
	Progname, Progver, Progname);
	 
	exit(1);
}
