/*

 wminet.c
 Multi-function system monitor
 by Dave Clark (clarkd@skynet.ca) (http://www.neotokyo.org/illusion)
 Martijn Pieterse (pieterse@xs4all.nl) (http://windowmaker.mezaway.org)
 and Antoine Nulle (warp@xs4all.nl) (http://windowmaker.mezaway.org)

 This software is licensed through the GNU General Public Lisence.

 ProFTPD support by Mike Kershaw aka Dragorn (dragorn@melchior.nerv-un.ml.org)
 ProFTPD support was made 64bit clean by Martijn Pieterse (pieterse@xs4all.nl)
 
 see http://windowmaker.mezaway.org for more awesome wm dock apps :)

*/


#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>

#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <utmp.h>
#include <dirent.h>

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include "../wmgeneral/wmgeneral.h"
#include "../wmgeneral/misc.h"

#include "wminet-master.xpm"
char wminet_mask_bits[64*64];
int  wminet_mask_width = 64;
int  wminet_mask_height = 64;

#include "config.h"
#include "proftpd.h"

#define WMINET_VERSION "2.0.3"

#define CHAR_WIDTH 5
#define CHAR_HEIGHT 7

// Lame work-around... Sigh... when will they standardize the headers!?!?
#define TCP_ESTABLISHED 1

extern	char **environ;

char	*ProgName;

int nUsers=0;
int nFtp=0;
int nHttp=0;
int nNfs=0;
int nProc=0;
int nlpd=0;

int loopinterval=1;

typedef struct
{
    int pos;
    int port;
    char label[10];
    int value;
} portwatch_t;

portwatch_t portwatch[6];

int numftpclasses=0;
char ftpclasses[MAX_FTP_CLASSES][64];

int monitor_proc=0;
int monitor_users=0;
int monitor_ftp=0;
int monitor_http=0;
int monitor_nfs=0;
int monitor_lpd=0;

char action1[256];
char action2[256];
char action3[256];
char action4[256];
char action5[256];

char ftp_pid_path[256];
int use_proftpd=0;

char uconfig_file[256];

void usage(void);
void printversion(void);
void BlitString(char *name, int x, int y);
void BlitNum(int num, int x, int y);
void GetStats( void );
void wminet_routine(int, char **);
int PortWatch( short port );
int ReadConfigInt(FILE *fp, char *setting, int *value);
int ReadConfigString(FILE *fp, char *setting, char *value);
int Read_Config_File( char *filename );


int main(int argc, char *argv[]) {

	int		i;

    uconfig_file[0] = 0;
    
	/* Parse Command Line */

	ProgName = argv[0];
	if (strlen(ProgName) >= 5)
		ProgName += (strlen(ProgName) - 5);
	
	for (i=1; i<argc; i++) {
		char *arg = argv[i];

		if (*arg=='-') {
			switch (arg[1]) {
			case 'd' :
				if (strcmp(arg+1, "display")) {
					usage();
					exit(1);
				}
				break;
			case 'g' :
				if (strcmp(arg+1, "geometry")) {
					usage();
					exit(1);
				}
				break;
			case 'v' :
				printversion();
				exit(0);
				break;
            case 'c' :
                if (argc > (i+1))
                {
                    strcpy(uconfig_file, argv[i+1]);
                    i++;
                }
                break;
            default:
				usage();
				exit(0);
				break;
			}
		}
	}

	wminet_routine(argc, argv);

	return 0;
}

/*******************************************************************************\
|* wminet_routine														   *|
\*******************************************************************************/

void wminet_routine(int argc, char **argv)
{
    int			i,j,k;
	XEvent		Event;
	int			but_stat = -1;

    time_t		curtime;
	time_t		prevtime;

    DIR         *dir;
    struct dirent *dent;

    char config_file[512];
    char buf[64];

    memset(&ftpclasses, 0, sizeof(ftpclasses));
    action1[0]=0;
    action2[0]=0;
    action3[0]=0;
    action4[0]=0;
    action5[0]=0;

    bzero(&portwatch, sizeof(portwatch));
    
	createXBMfromXPM(wminet_mask_bits, wminet_master_xpm, wminet_mask_width, wminet_mask_height);
    
	openXwindow(argc, argv, wminet_master_xpm, wminet_mask_bits, wminet_mask_width, wminet_mask_height);

	AddMouseRegion(0, 5, 6, 58, 16);
	AddMouseRegion(1, 5, 16, 58, 26);
	AddMouseRegion(2, 5, 26, 58, 36);
	AddMouseRegion(3, 5, 36, 58, 46);
	AddMouseRegion(4, 5, 46, 58, 56);

    // Read config file

    if (uconfig_file[0] != 0)
    {
        // user-specified config file
        fprintf(stderr, "Using user-specified config file '%s'.\n", uconfig_file);
        Read_Config_File(uconfig_file);
    }
    else
    {
        sprintf(config_file, "%s/.wminetrc", getenv("HOME"));

        if (!Read_Config_File(config_file))
        {
            // Fall back to /etc/wminetrc
            sprintf(config_file, "/etc/wminetrc");
        
            Read_Config_File(config_file);
        }
    }

    
    if ( monitor_ftp )
    {
        // Determine FTP PID files to search
        dir = opendir(ftp_pid_path);
        if (dir)
        {
            while ((dent = readdir(dir)))
            {
                //printf("scanning: %s\n", dent->d_name);

                /* If we're configged for proftpd, get that file  */
                if (use_proftpd)
                {
                    if (strstr(dent->d_name, "proftpd-") != NULL)
                        strcpy(ftpclasses[0], dent->d_name);
                }
                else
                {
                    if (strstr(dent->d_name, "ftp.pids-") != NULL)
                    {
                        strcpy(ftpclasses[numftpclasses++], dent->d_name);
                        //printf("ftppidfile: %s\n", dent->d_name);
                    }
                }
            }

            closedir(dir);
        }
    }

//    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, 5);
//    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, 16);
//    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, 27);
//    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, 38);
//    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, 49);
                
//    BlitString("XX", 45, 5);
//    BlitString("XX", 45, 16);
//    BlitString("XX", 45, 27);
//    BlitString("XX", 45, 38);
//    BlitString("XX", 45, 49);

    
    RedrawWindow();

    prevtime = time(0) - 1;
    
    while (1)
    {
		curtime = time(0);
		waitpid(0, NULL, WNOHANG);

        
        if ( curtime >= prevtime + loopinterval)
        {
            prevtime = curtime;

            // Get data to display
            GetStats();
    
            for (i=1; i<6; i++)
            {

                if ( i == monitor_proc )
                {
                    BlitString("procs:", 5, (11*(i-1)) + 5);
                    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, (11*(i-1)) + 5);
                    BlitNum(nProc, 45, (11*(i-1)) + 5);
                }

                if ( i == monitor_users )
                {
                    BlitString("users:", 5, (11*(i-1)) + 5);
                    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, (11*(i-1)) + 5);
                    BlitNum(nUsers, 45, (11*(i-1)) + 5);
                }

                if ( i == monitor_ftp )
                {
                    BlitString("ftp  :", 5, (11*(i-1)) + 5);
                    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, (11*(i-1)) + 5);
                    BlitNum(nFtp, 45, (11*(i-1)) + 5);
                }

                if ( i == monitor_http )
                {
                    BlitString("http :", 5, (11*(i-1)) + 5);
                    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, (11*(i-1)) + 5);
                    BlitNum(nHttp, 45, (11*(i-1)) + 5);
                }

                if ( i == monitor_nfs )
                {
                    BlitString("nfs  :", 5, (11*(i-1)) + 5);
                    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, (11*(i-1)) + 5);
                    BlitNum(nNfs, 45, (11*(i-1)) + 5);
                }

                if ( i == monitor_lpd )
                {
                    BlitString("lpd  :", 5, (11*(i-1)) + 5);
                    copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, (11*(i-1)) + 5);
                    BlitNum(nlpd, 45, (11*(i-1)) + 5);
                }

                for (j=1; j<6; j++)
                {
                    if ( i == portwatch[j].pos )
                    {
                        strncpy(buf, portwatch[j].label, 5);

                        for (k=0; k<5; k++)
                        {
                            if (buf[k] == 0)
                                buf[k] = ' ';
                        }
                        
                        buf[5] = ':';
                        buf[6] = 0;

                        BlitString(buf, 5, (11*(i-1)) + 5);
                        copyXPMArea(39, 84, (3*CHAR_WIDTH), 8, 39, (11*(i-1)) + 5);
                        BlitNum(portwatch[j].value, 45, (11*(i-1)) + 5);
                    }
                }
            }
                    
            RedrawWindow();
        }
        
        // X Events
        while (XPending(display))
        {
			XNextEvent(display, &Event);
            switch (Event.type)
            {
			case Expose:
				RedrawWindow();
				break;
			case DestroyNotify:
				XCloseDisplay(display);
				exit(0);
                break;
			case ButtonPress:
				i = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);

				but_stat = i;
				break;
			case ButtonRelease:
				i = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);

                if (but_stat == i && but_stat >= 0)
                {
                    switch (but_stat)
                    {
                    case 0 :
                        execCommand(action1);
                        break;
                    case 1 :
                        execCommand(action2);
                        break;
                    case 2:
                        execCommand(action3);
                        break;
                    case 3:
                        execCommand(action4);
                        break;
                    case 4:
                        execCommand(action5);
                        break;

					}
				}
				but_stat = -1;
//				RedrawWindow();
				break;
			}
		}

		usleep(100000L);
	}
}


void GetStats( void )
{
    struct utmp *ut;
    int 	fd;
	FILE	*fp;
    pid_t pid;
    char buf[1024];
    char *tok,*tok1;
    int i,j;
    char seps[]={"/"};
    char sep2[]={":"};
    char sep3[]={" "};

#ifdef HTTP_MONITOR_PROC
    DIR *dir;
    struct dirent *dent;
#endif
    
    // get statistics for display

    // logged in users
    nUsers=0;

    if ( monitor_users )
    {
        
        setutent();
        while ((ut = getutent()))
        {
            if ((ut->ut_type == USER_PROCESS) && 
		(ut->ut_name[0] != '\0'))
            {
                nUsers++;
            }
        }
        endutent();
    }

    // ftp connects
    nFtp = 0;

    if ( monitor_ftp )
    {
        /* hacked in...  Dragorn */
        if (use_proftpd)
        {
            logrun_t runent;
            logrun_header_t head;

            sprintf(buf, "%s/%s", ftp_pid_path, ftpclasses[0]);

            if (( fd = open(buf, O_RDONLY, 0644)) == -1)
            {
                printf("Unable to open %s - Disabling FTP monitor\n", buf);
                monitor_ftp = 0;
            }

            // Eat the header
            read(fd, &head, sizeof(logrun_header_t));
            if (head.r_magic != PROFTPD_MAGIC)
            {
                printf("Your proftpd runfile is either corrupt or you are running\n"
                       "a version of proftpd that wminet cannot support.\n"
                       "Disabling FTP monitor.\n");
                monitor_ftp = 0;
            }
            lseek(fd, head.r_size, SEEK_SET);

            while (fd != -1 && read(fd, (char *)&runent, sizeof(logrun_t)) == sizeof(logrun_t) )
                if (runent.pid > 0) nFtp++;

            close(fd);

        } else {
            for (i=0; i!= numftpclasses; i++)
            {
                sprintf(buf, "%s/%s", ftp_pid_path, ftpclasses[i]);
                //printf("opening '%s'\n", buf);
                fp = fopen(buf, "r");
                if (fp)
                {
                    while ( fread(&pid, sizeof(pid), 1, fp) == 1 )
                    {
                        if (pid)
                        {
                            nFtp++;
                        }
                    }
                    fclose(fp);
                }
            }
        }
    }


    // httpd processes
    nHttp = 0;

#ifdef HTTP_MONITOR_PROC

    if ( monitor_http )
    {

        dir = opendir("/proc");
        if (dir)
        {
            while ( (dent = readdir(dir)) )
            {
                if (!isalpha(dent->d_name[0]))
                {
                    sprintf(buf, "/proc/%s/stat", dent->d_name);
                    //printf("opening '%s'\n", buf);
                    fp=fopen(buf, "r");
                    if (fp)
                    {
                        fgets(buf, 128, fp);
                        tok = buf + strlen(dent->d_name) + 1;
                        //printf("checking '%s'\n", tok);
                        if (strncmp(tok, "(httpd)", 7) == 0)
                        {
                            nHttp++;
                        }
                        fclose(fp);
                    }
                }
            }
            closedir(dir);
        }
    }
#endif

#ifdef HTTP_MONITOR_NET

    if ( monitor_http )
    {

        fp = fopen("/proc/net/tcp", "r");
        if (fp)
        {
            fgets(buf, 512, fp); // get rid of text header
    
            while ( (fgets(buf, 512, fp)) )
            {
                tok = strtok(buf, sep2);
                tok = strtok(NULL, sep2);
                tok = strtok(NULL, sep2);
    
                tok[4]=0;
				tok1 = strtok(NULL, sep2);
				tok1 += 5;
				tok1[2] = 0;
    
                // printf("port: %i\n", strtol(tok, NULL, 16));
				// printf("state: %i\n", strtol(tok1, NULL, 16));
    
                i = strtol(tok, NULL, 16);
				j = strtol(tok1, NULL, 16);
    
                // should make this configurable
                if (( i == 80  || i == 8080) && (j == TCP_ESTABLISHED))
                {
                    nHttp++;
                }
    
            }
    
            fclose(fp);
        }
    }
#endif


    // NFS connects
    nNfs = 0;

    if ( monitor_nfs )
    {
        fp = popen("/usr/sbin/showmount -a", "r");
        if (fp)
        {
            while ( (fgets(buf, 128, fp)) )
            {
                nNfs++;
            }
    
            nNfs--;
            
            pclose(fp);
        }
    }

    // Total Processes
    nProc = 0;

    if ( monitor_proc )
    {
        
        fp = fopen("/proc/loadavg", "r");
        if (fp)
        {
            fgets((char *) &buf, 50, fp);
            tok = strtok((char *)&buf, (char *)&seps); // mmmmmmmm... strtok
            tok = strtok(NULL, (char *)&seps);
            nProc=atoi(tok);
            
            fclose(fp);
        }
    }

    // lpd
    if ( monitor_lpd )
    {
        fp = popen( "lpq -s",  "r");

        if (fp)
        {

            fgets(buf, 128, fp);
            
            // lp@defiant  0 jobs
            tok = strtok(buf, sep3);
            tok = strtok(NULL, sep3);

            nlpd = atoi(tok);

            pclose(fp);
        }
    }

    // Port Watchers
    for (i=1; i<6; i++)
    {

        if ( portwatch[i].pos != 0)
        {
            portwatch[i].value = PortWatch( portwatch[i].port );
        }
    }

}


int PortWatch( short port )
{
    FILE *fp;
    char buf[1024];
    char *tok,*tok1;
    int i,j;
    char sep2[]={":"};

    int count=0;

    fp = fopen("/proc/net/tcp", "r");
    if (fp)
    {
        fgets(buf, 512, fp); // get rid of text header

        while ( (fgets(buf, 512, fp)) )
        {
            tok = strtok(buf, sep2);
            tok = strtok(NULL, sep2);
            tok = strtok(NULL, sep2);

            tok[4]=0;
            tok1 = strtok(NULL, sep2);
            tok1 += 5;
            tok1[2] = 0;

            // printf("port: %i\n", strtol(tok, NULL, 16));
            // printf("state: %i\n", strtol(tok1, NULL, 16));

            i = strtol(tok, NULL, 16);
            j = strtol(tok1, NULL, 16);

            if (( i == port) && (j == TCP_ESTABLISHED))
            {
                count++;
            }

        }

        fclose(fp);
    }

    return count;
}



// Blits a string at given co-ordinates
void BlitString(char *name, int x, int y)
{
    int		i;
	int		c;
    int		k;

	k = x;
    for (i=0; name[i]; i++)
    {

        c = toupper(name[i]); 
        if (c >= 'A' && c <= 'Z')
        {   // its a letter
			c -= 'A';
			copyXPMArea(c * 6, 74, 6, 8, k, y);
			k += 6;
        }
        else
        {   // its a number or symbol
			c -= '0';
			copyXPMArea(c * 6, 64, 6, 8, k, y);
			k += 6;
		}
	}

}


// Blits number to give coordinates.. two 0's, right justified

void BlitNum(int num, int x, int y)
{
    char buf[1024];
    int newx=x;

    if (num > 99)
    {
        newx -= CHAR_WIDTH;
    }

    if (num > 999)
    {
        newx -= CHAR_WIDTH;
    }

    sprintf(buf, "%02i", num);

    BlitString(buf, newx, y);
}
    

// ReadConfigSetting
int ReadConfigString(FILE *fp, char *setting, char *value)
{
    char str[1024];
    char buf[1024];
    int i;
    int len;
    int slen;
    char *p=NULL;


    if (!fp)
    {
        return 0;
    }

    sprintf(str, "%s=", setting);
    slen = strlen(str);
    
    fseek(fp, 0, SEEK_SET);

    while ( !feof(fp) )
    {
        
        if (!fgets(buf, 512, fp))
            break;
        
        len = strlen(buf);

        // strip linefeed
        for (i=0; i!=len; i++)
        {
            if (buf[i] == '\n')
            {
                buf[i] = 0;
            }
        }

        //printf("Scanning '%s'...\n", buf);
        if ( strncmp(buf, str, strlen(str)) == 0)
        {
            // found our setting
            
            for(i=0; i!=slen; i++)
            {
                if ( buf[i] == '=' )
                {
                    p=buf+i+1;
                    strcpy(value, p);
                    return 1;
                }
            }
    
        }
    }
    
        return 0;
}

int ReadConfigInt(FILE *fp, char *setting, int *value)
{
    char buf[1024];

    if (ReadConfigString(fp, setting, (char *) &buf))
    {
        *value = atoi(buf);
        return 1;
    }

    return 0;
}

int Read_Config_File( char *filename )
{
    FILE *fp;

    fp = fopen(filename, "r");
    if (fp)
    {
        ReadConfigInt(fp, "interval", &loopinterval);
        ReadConfigString(fp, "action1", action1);
        ReadConfigString(fp, "action2", action2);
        ReadConfigString(fp, "action3", action3);
        ReadConfigString(fp, "action4", action4);
        ReadConfigString(fp, "action5", action5);
        ReadConfigInt(fp, "monitor_proc", &monitor_proc);
        ReadConfigInt(fp, "monitor_users", &monitor_users);
        ReadConfigInt(fp, "monitor_ftp", &monitor_ftp);
        ReadConfigInt(fp, "monitor_http", &monitor_http);
        ReadConfigInt(fp, "monitor_nfs", &monitor_nfs);
        ReadConfigInt(fp, "monitor_lpd", &monitor_lpd);
        ReadConfigInt(fp, "use_proftpd", &use_proftpd);
        ReadConfigString(fp, "ftp_pid_path", ftp_pid_path);

        ReadConfigInt(fp, "portwatch1.pos", &portwatch[1].pos);
        ReadConfigInt(fp, "portwatch1.port", &portwatch[1].port);
        ReadConfigString(fp, "portwatch1.label", portwatch[1].label);
        ReadConfigInt(fp, "portwatch2.pos", &portwatch[2].pos);
        ReadConfigInt(fp, "portwatch2.port", &portwatch[2].port);
        ReadConfigString(fp, "portwatch2.label", portwatch[2].label);
        ReadConfigInt(fp, "portwatch3.pos", &portwatch[3].pos);
        ReadConfigInt(fp, "portwatch3.port", &portwatch[3].port);
        ReadConfigString(fp, "portwatch3.label", portwatch[3].label);
        ReadConfigInt(fp, "portwatch4.pos", &portwatch[4].pos);
        ReadConfigInt(fp, "portwatch4.port", &portwatch[4].port);
        ReadConfigString(fp, "portwatch4.label", portwatch[4].label);
        ReadConfigInt(fp, "portwatch5.pos", &portwatch[5].pos);
        ReadConfigInt(fp, "portwatch5.port", &portwatch[5].port);
        ReadConfigString(fp, "portwatch5.label", portwatch[5].label);
        
        fclose(fp);
        return 1;
    }
    else
    {
        perror("Read_Config_File");
        fprintf(stderr, "Unable to open %s, no settings read.\n", filename);
        return 0;
    }

}

    
            
            



/*******************************************************************************\
|* usage																	   *|
\*******************************************************************************/

void usage(void)
{
    fprintf(stderr, "\nWMiNET - illusion <clarkd@skyia.com>, warp <warp@xs4all.nl>\n         & tijno <pieterse@xs4all.nl>\n\n");
	fprintf(stderr, "usage:\n");
	fprintf(stderr, "    -display <display name>\n");
	fprintf(stderr, "    -geometry +XPOS+YPOS      initial window position\n");
    fprintf(stderr, "    -c <filename>             use specified config file\n");
    fprintf(stderr, "    -h                        this help screen\n");
	fprintf(stderr, "    -v                        print the version number\n");
	fprintf(stderr, "\n");
}

/*******************************************************************************\
|* printversion																   *|
\*******************************************************************************/

void printversion(void)
{
	fprintf(stderr, "wminet v%s\n", WMINET_VERSION);
}
