#ifndef DEFAULT_TIME_WARP_EVENT_SET_H
#define DEFAULT_TIME_WARP_EVENT_SET_H

// Copyright (c) 2003 Clifton Labs, Inc.  All rights reserved.

// CLIFTON LABS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE
// SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT
// NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  CLIFTON LABS SHALL NOT BE
// LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, RESULT
// OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Dale E. Martin              dmartin@cliftonlabs.com

#include "warped.h"
#include "TimeWarpEventSet.h"
#include "Event.h"
#include "EventFunctors.h"
#include <deque>
using std::deque;

class SimulationObject;
class TimeWarpEventSetFactory;
class SimulationConfiguration;
class TimeWarpSimulationManager;




/** The default implementation of TimeWarpEventSet.  This implementation
    works in the following manner.  Events are kept in two groups,
    unprocessed and processed.  Processed events are not explicitly sorted
    (although they maybe be by virtue of the order of their insertion),
    unprocessed events are sorted on demand.  That is, they are inserted in
    arbitrary order but when events are requested they are sorted if
    needed.
*/
class DefaultTimeWarpEventSet : public TimeWarpEventSet {

  /**
     This class manages the set of events for a single SimulationObject.
     It is the heart of the DefaultTimeWarpEventSet implementation, but it
     is invisible to the interface so it is a private part of that class
     and not contained elsewhere.
  */
  class EventContainer {
  public:
    EventContainer() : sortStatus( SORTED ){}
    ~EventContainer(){}

    void insert( const Event *newEvent ){ 
      unprocessedEvents.push_back( newEvent );
      sortStatus = UNSORTED;
    }

    void sortByReceiveTimes(){
      std::sort( unprocessedEvents.begin(),
		 unprocessedEvents.end(),
		 receiveTimeLessThan() );
      sortStatus = SORTED;
    }

    void remove( SimulationObject *reclaimer,
		 const EventId toRemoveId );

    /**
       Return the next event (with respect to receive time), leaves it in
       the set.
    */
    const Event *nextEvent();

    /**
       Return the next event (with respect to receive time), and removes it
       in the set.
    */
    const Event *getNextEvent();
    
    void rollback( const VTime & );

    /**
       For debugging purposes.
    */
    void debugDump( ostream & ) const;

    /**
       Clears the processedEvents vector of events at or before
       collectTime.
       @param object The object that will be used to reclaim the events.
       @param collectTime Events in the processed queue at or before this
       time will be reclaimed.
    */
    void garbageCollect( SimulationObject *object, const VTime &collectTime );

  private:
    deque<const Event *> unprocessedEvents;
    vector<const Event *> processedEvents;
    enum SortStatus { SORTED, UNSORTED } sortStatus;
  };

public:
  DefaultTimeWarpEventSet( TimeWarpSimulationManager *initSimManager );

  ~DefaultTimeWarpEventSet(){};
   
  bool insert( const Event *event );

  void handleAntiMessage( SimulationObject *eventFor,
			  const EventId cancelEventId );

  void remove( const Event *, findMode );

  const Event *getEvent(SimulationObject *);

  const Event *getEvent(SimulationObject *, const VTime&);

  const Event *peekEvent(SimulationObject *);

  const Event *peekEvent(SimulationObject *, const VTime&);

  void garbageCollect( SimulationObject *, const VTime & );

  void configure( SimulationConfiguration & ){}

  void rollback( SimulationObject *, const VTime & );

  void debugDump( const string &objectName, ostream &os ){ 
    os << "\n---------------\n" << objectName << "\n";
    getEventContainer( objectName ).debugDump( os );
  }

private:
  EventContainer &getEventContainer( const string &objectName );

  StringHashMap<EventContainer *> events;
  TimeWarpSimulationManager *mySimulationManager;
};

#endif
