#ifndef VHDLKERNEL_CC
#define VHDLKERNEL_CC
// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.


//
// $Id: VHDLKernel.cc,v 1.1 2003/12/04 19:13:18 dmartin Exp $
//
//---------------------------------------------------------------------------

#include <clutils/StringUtilities.h>
#include <clutils/Debug.h>
#include <warped/SimulationStream.h>
#include "VHDLKernel.hh"
#include "FileType.hh"
#include "AccessVariable.hh"
#include "Block.hh"
#include "RecordType.hh"
#include "ArrayType.hh"
#include "AccessType.hh"
#include "SharedFileType.hh"
#include "SigEvent.hh"
#include "WaitEvent.hh"
#include "SharedFileEvent.hh"
#include "STDTypes.hh"

using std::streampos;
using std::cin;
using std::flush;

vector<VHDLKernel*> listOfProcesses;

extern SharedFileType *output;
extern SharedFileType *input;

/* Implements the global object identifier algorithm. Will produce an integer
   number which is ensured to be globally unique and thus may be used as an
   identifier of the object. */

int 
VHDLKernel::getGlobalObjectId(){
  static int globalId = 0;
  return globalId++;
}

const string &
VHDLKernel::getName( const OBJECT_ID& id ){ 
  return listOfProcesses[id.getSimulationObjectID()]->getName(); 
}

void
VHDLKernel::initializeGlobalObjectId() {
  static bool initFlag = false;
  if (initFlag == false) {
    initFlag = true;
    output = 
      new SharedFileType("stdout", _savant_file_open_kind(_savant_file_open_kind::WRITE_MODE));
    // input = new SharedFileType("stdin", _savant_file_open_kind(_savant_file_open_kind::READ_MODE));
    
  }
}

VHDLKernel::VHDLKernel( const string &name, _savant_entity_elab *ptr) : proc(ptr){
  initializeGlobalObjectId();
  vhdlProcessId = getGlobalObjectId();
  processName = string( name + "." + intToString( vhdlProcessId )).c_str();

  cout << "Constructing " << processName << endl;

  listOfProcesses.push_back(this);
  postponed = false;
  numFiles  = 0;
}

VHDLKernel::~VHDLKernel(){}

void
VHDLKernel::initialize() {
  getVHDLState()->initState(proc, this);
  // set GUARD signals to result of their expression
  if( getGuard() != NULL ){
    updateGuard(getGuard());
  }
  executeVHDL( *(VHDLKernel_state *)getState() );
}

void
VHDLKernel::finalize() {
  getVHDLState()->cleanState();
  for( vector<SimulationStream*>::iterator i = fileHandles.begin();
       i != fileHandles.end();
       i++ ){
//     (*i)->flush();
  }
}

Block *
VHDLKernel::getEventsToExecute(){
  // A transaction that has been posted to a process could be cancelled
  // during a marking process The events to be cancelled are received are
  // informed by CANCELTRANSACTION event eventsToCancel is tha array that
  // holds the CANCELTRANSACTION events
  Block eventsToCancel;
  
  // In a simulation cycle, we could get different events that correspond
  // to several transactions posted for this cycle eventsInCurrentCycle
  // collects all these events from the simulation
  // kernel(TimeWarp/Sequential/...)
  Block *eventsInCurrentCycle = new Block();

  
  //Collect all the events to be processed in this
  //simulation time cycle
  //Define an expandable array to hold the events in the current
  //simulation cycle
  while ( haveMoreEvents() ){
    //get the elements from the inputQ 
    //add them to the expandable array of pointers
    const VHDLEvent *currentEvent = dynamic_cast<const VHDLEvent *>(getEvent());
    ASSERT( currentEvent != 0 );
    //Check if the currentEvent is a Cancellation Event 
    //that has been sent during the Marking process
    if( currentEvent->isCancelTransactionEvent() ) { //CANCELTRANSACTION
      eventsToCancel.addElement( const_cast<VHDLEvent *>(currentEvent) );
    }
    else {
      eventsInCurrentCycle->addElement( const_cast<VHDLEvent *>(currentEvent) );
    }
  }
    
  // Only test time, not time & delta, for a postponed process.
  // Have to introduce this change in the WARPED API to handle this case...

  //CANCEL out the messages by looking at the events to be processed
  //in this cycle and the remove them
  for( int i = 0; i < eventsToCancel.getNumberOfElements(); i++ ){
    bool matchFound = false;
    SigEvent *toCancel = static_cast<SigEvent *>(eventsToCancel.getElement(i));
    for( int j = 0; j < eventsInCurrentCycle->getNumberOfElements(); j++ ){
      if ( static_cast<VHDLEvent *>(eventsInCurrentCycle->getElement(j))->isSigEvent() ){
        SigEvent *postedEvent = static_cast<SigEvent *>(eventsInCurrentCycle->getElement(j));
	if ( postedEvent->getSigId() == toCancel->getSigId() &&
	     postedEvent->getSequenceNumber() == toCancel->getSequenceNumber() ){
	  ASSERT (postedEvent->getSigSrc() == toCancel->getSigSrc());
	  ASSERT (postedEvent->getSenderSigId() == toCancel->getSenderSigId());
	  eventsInCurrentCycle->removeElement(j);
	  matchFound = true;
	  break;
	}
      }
    }
    
    if( matchFound == false ){
      cerr << getProcessName()
	   << " -- Error: can't find Transaction to cancel after marking\n";
      cerr << "Proceeding with the simulation\n";
      cerr << "Simulation may not be functionally correct\n";
      abort();
    }
  }
  
  return eventsInCurrentCycle;
}

void
VHDLKernel::executeProcess() {
  Block *eventsInCurrentCycle = getEventsToExecute();
  // process all events at the current time
  for( int i = 0; i < eventsInCurrentCycle->getNumberOfElements(); i++ ){
    VHDLEvent *event = static_cast<VHDLEvent *>(eventsInCurrentCycle->getElement(i));
    clutils::debug << getName() << " executing " << *event << " at time " << getTimeNow() << endl;
    if (event != NULL) {
      event->execute( this );
    }
  }
  delete eventsInCurrentCycle;

  executeVHDL( *getVHDLState() );
}

EnumerationType
VHDLKernel::savantendfile( VHDLKernel*, int fileidx ){
  return savantendfile_boolean(NULL, fileidx);
}

EnumerationType
VHDLKernel::savantendfile_boolean(VHDLKernel*, int fileidx) {
  if (fileHandles[fileidx]->eof())  {
    return SAVANT_BOOLEAN_TRUE;
  }
  else{
    return SAVANT_BOOLEAN_FALSE;
  }
}

EnumerationType
VHDLKernel::savantendfile(VHDLKernel* base, FileType& file) {
  return savantendfile_boolean(base, file.get_fileHandle());
}

EnumerationType
VHDLKernel::savantendfile_boolean(VHDLKernel* base, FileType &file ){
  return savantendfile_boolean(base, file.get_fileHandle());
}

int 
VHDLKernel::savantreadline(VHDLKernel*, 
			   int fileidx, 
			   AccessVariable &line) {
  int linesize;
  const char *tempLine = getline(*(fileHandles[fileidx]), linesize);
  line.setVal( string( tempLine ) );
  
  return NORMAL_RETURN;
}


int 
VHDLKernel::savantwriteline( VHDLKernel*, 
			     int fileidx,
			     AccessVariable &line ){
  ostringstream outbuf;
  
  outbuf << line.getVal();
  outbuf << "\n";

#ifdef VHDLDBG
  cout << "Writing data to file " << fileidx << ":\n|" << outbuf.str() << "| "
       << "at VTime = " << getSimulationTime() << endl;
#endif
  
  *fileHandles[fileidx] << outbuf.str() << "\n";
  
  line.setVal("");

  return NORMAL_RETURN;
}

int 
VHDLKernel::savantwriteline( VHDLKernel *base, int fileidx, AccessType &line ){
  ostringstream outbuf;
  int returnValue;
  
  outbuf << line;
  const string outString = outbuf.str();

  AccessVariable tempStr;

  tempStr.setVal( outString );
  
  returnValue = savantwriteline( base, fileidx, tempStr );

  delete line.val;
  line.val = NULL;

  return returnValue;
}

int
VHDLKernel::savantreadline( VHDLKernel *base, int fileidx, AccessType& line ){
  int returnValue;
  AccessVariable tempStr;
  ostringstream outbuf;
  
  if (line.val != NULL) {
    delete line.val;
    line.val = NULL;
  }
  
  returnValue = savantreadline(base, fileidx, tempStr);
  if ( returnValue == NORMAL_RETURN) {
    outbuf << tempStr.getVal();
    line.val = new ArrayType( ObjectBase::VARIABLE, 
			      SavantstringType_info, -1, 
			      outbuf.str().c_str() );
  }
  
  return returnValue;
}

int
VHDLKernel::savantreadline( VHDLKernel*base, 
			    int fileidx, 
			    AccessType& line,
			    IntegerType &count ){
  int returnValue, dataRead = 0;
  AccessVariable tempStr;
  ostringstream outbuf;
  if (line.val != NULL) {
    delete line.val;
    line.val = NULL;
  }
  
  returnValue = savantreadline(base, fileidx, tempStr);
  if ( returnValue == NORMAL_RETURN) {
    for( int length = 0; length < tempStr.length(); length++ ){
      outbuf << tempStr.getVal()[length];
      if( tempStr.getVal()[length] == FILE_DELIMITER) {
	dataRead++;
      }
    }
    
    line.val = new ArrayType(ObjectBase::VARIABLE, SavantstringType_info, -1, outbuf.str().c_str());
  }
  
  count = IntegerType(ObjectBase::VARIABLE, UniversalInteger(dataRead), SavantintegerType_info);
  
  return returnValue;
}

int
VHDLKernel::savantreadline( VHDLKernel *base, 
			    const FileType &file,
			   AccessType& line ){
  return savantreadline(base, file.get_fileHandle(), line);
}

int
VHDLKernel::savantwriteline(VHDLKernel*base, 
			    FileType &file,
			    AccessType& line ){
  return savantwriteline(base, file.get_fileHandle(), line);
}

int
VHDLKernel::savantwriteline( VHDLKernel *, 
			     SharedFileType *sf,
			     AccessType &line ){
  ostringstream tempStream;
  tempStream << line;
  string outputData = tempStream.str();
		
  SharedFileEvent *sfe = new SharedFileEvent( getTimeNow(),
					      getTimeNow(),
					      getName(),
					      sf->getName(),
					      outputData );

  getObjectHandle( sfe->getReceiver() )->receiveEvent( sfe );

  // Have to reset the line type variable...
  delete line.val;
  line.val = NULL;
  
  return NORMAL_RETURN;
}

int
VHDLKernel::savantreadline( VHDLKernel *, 
			    SharedFileType *,
			    AccessType &line ){
  // Check if this is the second call and if we have the data...
  if( getVHDLState()->sharedReadPending == true ){
    line.val = new ArrayType( ObjectBase::VARIABLE, 
			      SavantstringType_info, 
			      -1, 
			      sharedData.c_str() );
    sharedData.erase( sharedData.begin(), sharedData.end() );
    getVHDLState()->sharedReadPending = false;
    return NORMAL_RETURN;
  }

  // Okay...let's sechdule a shared read here...
  SharedFileEvent *sfe = new SharedFileEvent( getTimeNow(),
					      getTimeNow(),
					      getName(),
					      sfe->getReceiver(),
					      "" );
					      
  getObjectHandle( sfe->getReceiver() )->receiveEvent( sfe );
  getVHDLState()->sharedReadPending = true;
  
  return EXECUTE_WAIT_RETURN;
}

char*
VHDLKernel::getline(istream& file, int& linesize) {
  unsigned int bytesread = 0;
  unsigned int chunksize = 1024;
  char buf[1024];
  char* retstr;
  streampos linestart;

  if (!file.good()) {
    return NULL;
  }

  if (file.eof()) {
    return NULL;
  }
  
  linestart = file.tellg(); // save position of beginning of line
  file.get(buf, chunksize, '\n'); // get the line and the # of chars read
  bytesread = file.gcount();

  if (bytesread < (chunksize - 1)) {
    // line fit into original buffer
 
    linesize = bytesread;
    retstr = new char[bytesread + 1];
    memcpy(retstr, buf, bytesread);
    *(retstr + bytesread) = '\0';

    // read newline .ie. next character also...
    file.get(buf[0]);
    // Check for EOF...this seems necessary...don't know why exactly...
    // should check it and fix it later on...
    if (!file.get(buf[0]).eof())  {
      file.seekg(-1, ios::cur);
    }
    return retstr;
  }
  else {
    // we filled the static buffer--see if we're really done
    if (file.peek() == '\n') {
      // we've got the whole line, so skip that pesky newline!
      file.seekg(1, ios::cur);
      linesize = bytesread;
      retstr = new char[bytesread + 1];
      memcpy(retstr, buf, bytesread);
      *(retstr + bytesread) = '\0';
      return retstr;
    }
    else {
      // the line's longer than the initial value of chunksize, so dynamic
      // memory is used, repeatedly growing exponentially, until we do get
      // the line in one piece.  Trying static memory first, then using
      // dynamic is ugly, but it prevents having to use any dynamic memory
      // in all but the most massive input file lines.
      char* tmpptr = NULL;
      do {
	if (tmpptr != NULL) delete [] tmpptr;
	file.seekg(linestart); // rewind back to beginning of the line
	chunksize *= 2;  // get more chars this time around
	tmpptr = new char[chunksize];
	file.get(tmpptr, chunksize, '\n');
	bytesread = file.gcount();
      } while (bytesread == (chunksize - 1));

      file.seekg(1, ios::cur);
      linesize = bytesread;
      retstr = new char[bytesread + 1];
      memcpy(retstr, tmpptr, bytesread);
      *(retstr + bytesread) = '\0';
      delete [] tmpptr;
      return retstr;
    }
  }
}

bool 
VHDLKernel::eatwhite(AccessVariable &line) {
  // skip any whitespace
  return line.eatwhite();
}

// Opens a file for i/o.

int
VHDLKernel::openFile( const string &fileName, 
		      const _savant_file_open_kind &mode, 
		      ios::openmode opening_mode ) {
  SimulationStream *newHandle = 0;
  if (mode.val == _savant_file_open_kind::READ_MODE) {
    // We want to read
    if ( fileName == "stdin" ) {
      // We want to read from stdin

      // XXX FIXME! Cannot read from stdin. Allready fixed in WARPED?
      cerr << "Cannot read input from stdin!  Need modifications to WARPED API\n";
      abort();
    }
    else {
      // We want to read from a file
      newHandle = getIFStream(fileName.c_str());
    }
  }
  else {
    // We want to write
    if( fileName == "stdout" ) {
      // We want to write to stdout
      newHandle = (SimulationStream *)wout;
    }
    else {
      // We want to write to a file
      newHandle = getOFStream(fileName, opening_mode);
    }
  }

  fileHandles.push_back( newHandle );

  return numFiles++;
}

void
VHDLKernel::closeFile( const _savant_file_open_kind &mode,
		       int ){
  // XXX FIXME! Does not close files!
  if (mode.val == _savant_file_open_kind::READ_MODE) {
    // fileHandles[fileHandle]->close();
  }
  else {
    // fileHandles[fileHandle]->close();
  }
}

void
VHDLKernel::reportError(const string& msg, const SEVERITY level)  {
  char option = 'n';
  
  switch (level) {
  case NOTE:
    cerr << "Note: " << msg << endl;
    break;

  case WARNING:
    cerr << "Warning: " << msg << endl;
    break;

  case ERROR:
    cerr << "Error: " << msg << endl;
    cerr << "Do you wish to continue? (y/n) " << flush;
    cin  >> option;
    if ((option != 'y') && (option != 'Y')) {
      abort();
    }
    
    break;

  case ABORT:
  default:
    cerr << "Abort: " << msg << endl;
    abort();
    break;
  }
}

const VHDLVTime &
VHDLKernel::getTimeNow() const {
  return dynamic_cast<const VHDLVTime &>(getSimulationTime());
}

void
VHDLKernel::deallocateState(State *trash) {
  delete trash;
}

void
VHDLKernel::reclaimEvent( const Event *e ) {
  delete e;
}

void
VHDLKernel::reclaimSerializedInstance(SerializedInstance *si) {
  delete si;
}

void
VHDLKernel::disconnectDriver( SigEvent *event ){
  SignalBase *signal = getVHDLState()->locateSig(event->getSigId()); 
  int processId = event->getSigSrc();
  
  disconnectDriver( signal, processId );
  
}

void
VHDLKernel::updateDriver( SigEvent *event ){
  //  int sigId = event->getSigId();
  int sigSrc = event->getSigSrc();
  int senderSigId = event->getSenderSigId();

  const VHDLData &data = event->getData();
  ArrayInfo dInfo = event->getDestBounds();
  ArrayInfo sInfo = event->getSourceBounds();

  SignalBase *signal = getVHDLState()->locateSig(event->getSigId()); 
  if( signal == 0 ){
    cerr << getName() << " asked to update signal id |" << 
      event->getSigId() << "|, but can't locate it in my state." << endl;
    abort();
  }

  updateDriver( signal, sigSrc, senderSigId, &data, &sInfo, &dInfo );
}

void 
VHDLKernel::updateDriver( SignalBase *, 
			  int,
			  int,
			  const VHDLData *,
			  ArrayInfo *,
			  ArrayInfo * ){
  abort();
}


void
VHDLKernel::updateWait( WaitEvent* event ){
  if( event->getWaitId() != getVHDLState()->waitLabel ){
    // This case is NOT an error case.  It could happen that I was waiting
    // with the following VHDL statement:
    //    wait on sig until <cond> for <time> ns;
    // If the signal has an event on it, I will no longer wait here, but
    // the wait-resume event that I sent out earlier WILL arrive here.  I
    // should not abort on that.  I have to check if I was waiting on this
    // wait stmt.  If not, just return without doing anything.
    return;
  }

  updateWait( dynamic_cast<const VHDLVTime &>(event->getSendTime()) );
}

void
VHDLKernel::updateWait( const VHDLVTime & ){ abort(); }

bool
VHDLKernel::resumeWait(int, const EnumerationType & ){ abort(); }

void
VHDLKernel::executeWait(int, const PhysicalType & ){ abort(); }

void
VHDLKernel::setProcedureWait( Wait * ){ abort(); }

void
VHDLKernel::assignSignal( VHDLType &, 
			  VHDLKernel *, 
			  const VHDLType &,
			  const PhysicalType &,
			  const PhysicalType &,
			  const ArrayInfo &,
			  const ArrayInfo & ){ 
  abort(); 
}

void
VHDLKernel::assignSignal( SignalBase &, 
			  VHDLKernel *, 
			  const VHDLData &,
			  const VHDLVTime &, 
			  const VHDLVTime &,
			  const ArrayInfo &, 
			  const ArrayInfo & ){ 
  abort(); 
}


void 
VHDLKernel::updateSignal(VHDLType*){ abort(); }

void
VHDLKernel::updateSignal( SignalBase*, bool ){ abort(); }

EnumerationType 
VHDLKernel::locateQuietAttribute( const SignalBase *, 
				  const VHDLVTime ){ 
  abort();
}

EnumerationType
VHDLKernel::locateQuietAttribute( const VHDLType *, 
				  const VHDLVTime ){
  abort();
}

EnumerationType
VHDLKernel::locateQuietAttribute( const ScalarType *, 
				  const VHDLVTime ){
  abort();
}

EnumerationType
VHDLKernel::locateQuietAttribute( const RecordType *,
				  const VHDLVTime ){
  abort();
}

EnumerationType
VHDLKernel::locateQuietAttribute( const ArrayType *,
				  const VHDLVTime ){
  abort();
}

EnumerationType
VHDLKernel::locateEventAttribute( const VHDLType* ){ abort(); }

EnumerationType 
VHDLKernel::locateEventAttribute(SignalBase *){ abort(); }

EnumerationType
VHDLKernel::locateEventAttribute( const ScalarType* ){ abort(); }

EnumerationType
VHDLKernel::locateEventAttribute( const RecordType* ){ abort(); }

EnumerationType
VHDLKernel::locateEventAttribute( const ArrayType* ){ abort(); }
  
EnumerationType
VHDLKernel::locateActiveAttribute( const VHDLType* ){ abort(); }

EnumerationType
VHDLKernel::locateActiveAttribute( const SignalBase *){ abort(); }

EnumerationType
VHDLKernel::locateActiveAttribute( const ScalarType* ){ abort(); }

EnumerationType
VHDLKernel::locateActiveAttribute( const RecordType* ){ abort(); }

EnumerationType 
VHDLKernel::locateActiveAttribute( const ArrayType* ){ abort(); }

EnumerationType 
VHDLKernel::locateStableAttribute( const SignalBase *, 
				   const VHDLVTime ){
    abort(); 
}

EnumerationType
VHDLKernel::locateStableAttribute( const VHDLType *, 
				   const VHDLVTime ){
  abort(); 
}

EnumerationType
VHDLKernel::locateStableAttribute( const ScalarType *, 
				   const VHDLVTime ){
  abort(); 
}

EnumerationType
VHDLKernel::locateStableAttribute( const RecordType *, 
				   const VHDLVTime ){
  abort(); 
}

EnumerationType
VHDLKernel::locateStableAttribute( const ArrayType *, 
				   const VHDLVTime ){
  abort(); 
}

EnumerationType *
VHDLKernel::locateTransactionAttribute( const VHDLType *){ 
  abort(); 
}

EnumerationType *
VHDLKernel::locateTransactionAttribute( const SignalBase *){
  abort(); 
}

EnumerationType *
VHDLKernel::locateTransactionAttribute( const ScalarType *){
  abort(); 
}

EnumerationType *
VHDLKernel::locateTransactionAttribute( const RecordType *){
  abort(); 
}

EnumerationType *
VHDLKernel::locateTransactionAttribute( const ArrayType *){
  abort(); 
}

PhysicalType
VHDLKernel::locateLastActiveAttribute( const VHDLType *){ 
  abort(); 
}

PhysicalType 
VHDLKernel::locateLastActiveAttribute( const SignalBase *){ 
  abort(); 
}

PhysicalType 
VHDLKernel::locateLastActiveAttribute( const ScalarType *){ 
  abort(); 
}

PhysicalType
VHDLKernel::locateLastActiveAttribute( const RecordType *){ 
  abort(); 
}

PhysicalType
VHDLKernel::locateLastActiveAttribute( const ArrayType *){
  abort(); 
}

const PhysicalType
VHDLKernel::locateLastEventAttribute( const VHDLType *){ 
  abort(); 
}

const PhysicalType 
VHDLKernel::locateLastEventAttribute( const SignalBase *){ 
  abort(); 
}

const PhysicalType
VHDLKernel::locateLastEventAttribute( const ScalarType *){
  abort(); 
}

const PhysicalType
VHDLKernel::locateLastEventAttribute( const RecordType *){
  abort(); 
}

const PhysicalType
VHDLKernel::locateLastEventAttribute( const ArrayType *){
  abort(); 
}

VHDLType *
VHDLKernel::locateLastValueAttribute( const VHDLType *){ 
  abort(); 
}

VHDLType *
VHDLKernel::locateLastValueAttribute( const SignalBase *){
  abort(); 
}

VHDLType *
VHDLKernel::locateLastValueAttribute( const ScalarType *){
  abort(); 
}

VHDLType *
VHDLKernel::locateLastValueAttribute( const RecordType *){
  abort(); 
}

VHDLType *
VHDLKernel::locateLastValueAttribute( const ArrayType *){ 
  abort(); 
}


void
VHDLKernel::scheduleWait( int waitId, const VHDLVTime &waitTime ){
  // schedule resumption from this wait by sending event to self
  WaitEvent *waitEv = new WaitEvent( getTimeNow(),
				     waitTime, 
				     getName(),
				     getName(),
				     waitId );
  receiveEvent( waitEv );
}

#endif

