#include "UniversalInteger.hh"
#include "UniversalReal.hh"
#include "UniversalLongLongInteger.hh"
#include "AccessVariable.hh"
#include "AccessType.hh"
#include "ArrayType.hh"
#include "std_standardPkg.hh"
#include <sstream>
#include <cmath>

using std::stringstream;
using std::ostringstream;

const UniversalInteger UNIVERSAL_TRUE(1);
const UniversalInteger UNIVERSAL_FALSE(0);

UniversalInteger
UniversalInteger::typeCast(VHDLData &toCast){
  switch(toCast.getUniversalKind()){
  case UNIVERSAL_REAL:
    return UniversalInteger( dynamic_cast<UniversalReal &>(toCast).val );
    break;
  case UNIVERSAL_LONG_LONG_INTEGER:
    return UniversalInteger( dynamic_cast<UniversalLongLongInteger &>(toCast).val );
    break;
  case UNIVERSAL_INTEGER:
    return toCast.clone();
    break;
  default:
    cerr << toCast.getUniversalKind() 
	 << " cannot be type casted into an UniversalInteger ! " << endl;
    return UniversalInteger(0);
  }
}

UniversalInteger::UniversalInteger(const VHDLData &v) {
  this->operator=(v);
}

// Reducing the range by one so that we are able to detect when we
// overflow.
const UniversalInteger &
UniversalInteger::getMax(){
  static UniversalInteger max( INT_MAX );
  return max;
}

const UniversalInteger &
UniversalInteger::getMin(){
  static UniversalInteger min( INT_MIN );
  return min;
}

UniversalInteger::UniversalInteger(const UniversalReal &r ){
  val = int(r.val);
  overflow_flag = false;
}

VHDLData &
UniversalInteger::operator=( const VHDLData &d  ){
  val = dynamic_cast<const UniversalInteger &>(d).val;
  return *this;
}

UniversalInteger&
UniversalInteger::operator=( const UniversalInteger &i ){
  val = i.val;
  return *this;
}

int
UniversalInteger::savantwrite(AccessVariable  &line) const {
  ostringstream tempStr;
  
  tempStr << line.getVal() << *this << FILE_DELIMITER;
  line.setVal( tempStr.str() );

  return NORMAL_RETURN;
}

int
UniversalInteger::savantwrite(AccessType &line) const {
  ostringstream tempStr;

  if (line.val != NULL ){
    tempStr << line << *this << FILE_DELIMITER << '\0';
  }
  else {
    tempStr << *this << FILE_DELIMITER << '\0';
  }
  
  if (line.val != NULL ){
    delete line.val;
  }
  
  line.val = new ArrayType(ObjectBase::VARIABLE, SavantstringType_info, -1, tempStr.str().c_str());

  return NORMAL_RETURN;
}

int
UniversalInteger::savantread(AccessVariable  & ){
  return NORMAL_RETURN;
}

int
UniversalInteger::savantread(AccessType &line ){
  char *workString = line.val->getString();
  char *input      = new char[strlen(workString) + 1];
  stringstream input_stream;
  
  extractToDelimiter(workString, FILE_DELIMITER, input);

  input_stream << input;
  input_stream >> this->val;
  //  delete [] input_stream.str();
  //  sscanf(input, "%d", &(this->val));

  delete line.val;

  if (strlen(workString) > 0 ){
    line.val = new ArrayType(ObjectBase::VARIABLE, SavantstringType_info, -1, workString);
  }
  else {
    line.val = NULL;
  }

  delete [] workString;
  delete [] input;
  
  return NORMAL_RETURN;
}

UniversalInteger
UniversalInteger::vhdlUnaryPlus() const  {
  return *this;
}

UniversalInteger
UniversalInteger::vhdlUnaryMinus() const {
  return UniversalInteger( -1*getIntValue() );
}


UniversalInteger
UniversalInteger::vhdlAbs() const {
  return UniversalInteger( ::abs(getIntValue()));
}

UniversalInteger
UniversalInteger::vhdlPlus( const RValue &rhs ) const {
  return UniversalInteger( getIntValue() + rhs.getIntValue() );
}

UniversalInteger
UniversalInteger::vhdlMinus( const RValue &rhs ) const {
  return UniversalInteger( getIntValue() - rhs.getIntValue() );
}

UniversalInteger
UniversalInteger::vhdlMultiply( const RValue &rhs ) const {
  return UniversalInteger( getIntValue() * rhs.getIntValue() );
}

UniversalInteger
UniversalInteger::vhdlDivide( const RValue &rhs ) const {
  return UniversalInteger( getIntValue() / rhs.getIntValue() );
}
// VHDL LRM Section 7.2.6: 
// The result of the modulus operation is such that (A mod B) has the sign
// of B and an absolute value less than the absolute vaue of B; in
// addition, for some integer value N, this result must satisfy the
// relation:
//     A = B*N + (A mod B)
UniversalInteger
UniversalInteger::vhdlMod( const RValue &rhs ) const {
  int lval = getIntValue();
  int rval = rhs.getIntValue();
  return UniversalInteger((lval*rval < 0) ? (rval + lval%rval) : (lval%rval));
}
// VHDL LRM Section 7.2.6: 
// Integer division and remainder are defined by the following relation:
//     A = (A/B)*B + (A rem B)
// where (A rem B) has the sign of A and an absolute value less than the
// absolute value of B.
UniversalInteger
UniversalInteger::vhdlRem( const RValue &rhs ) const {
  return UniversalInteger( getIntValue() % rhs.getIntValue() );
}

UniversalInteger
UniversalInteger::vhdlPow( const RValue &rhs ) const {
  return UniversalInteger( ::pow(getDoubleValue(), 
				 rhs.getDoubleValue() ));
}

// The following functions are necessary since bit is an enum type and all
// enums are implemented as integers within the kernel.
//The following functons are only for bit types

UniversalInteger
UniversalInteger::vhdlAnd( const UniversalInteger &rhs ) const {
  return UniversalInteger( getIntValue() & rhs.getIntValue() );
}

UniversalInteger
UniversalInteger::vhdlOr( const UniversalInteger &rhs ) const {
  return( getIntValue() | rhs.getIntValue() );
}

UniversalInteger
UniversalInteger::vhdlNand( const UniversalInteger &rhs ) const {
  return UniversalInteger( 1 - ( getIntValue() & rhs.getIntValue() ) );
}

UniversalInteger
UniversalInteger::vhdlNor( const UniversalInteger &rhs ) const {
  return UniversalInteger( 1 - ( getIntValue() | rhs.getIntValue() ) );
}

UniversalInteger
UniversalInteger::vhdlXor( const UniversalInteger &rhs ) const {
  return( getIntValue() != rhs.getIntValue() );
}

UniversalInteger
UniversalInteger::vhdlXnor( const UniversalInteger &rhs ) const {
  return( getIntValue() == rhs.getIntValue() );
}

UniversalInteger
UniversalInteger::vhdlNot() const {
  return UniversalInteger(!getIntValue());
}
