#include "std_standardPkg.hh"
#include "Signal.hh"
#include "ImplicitSignal.hh"
#include "Variable.hh"
#include "SignalNetinfo.hh"
#include "STDTypes.hh"
#include <math.h>
#include <sstream>

using std::ostringstream;

IntegerType IntegerType::INVALID_TYPE_OBJECT;

IntegerType::IntegerType(const IntegerType& value): ScalarType( value ){
  range = value.range;
}

IntegerType::IntegerType(ObjectBase::ObjectType objType, int i) : ScalarType(objType) {
  UniversalInteger val(i);

  switch(objType) {
  case ObjectBase::SIGNAL : 
    setObject( new Signal<UniversalInteger>(val) ); 
    break;
  case ObjectBase::IMPLICIT_SIGNAL :
    setObject( new ImplicitSignal<UniversalInteger>(val) );
    break;
  case ObjectBase::VARIABLE :
    setObject( new Variable<UniversalInteger>(val) );
    break;
  case ObjectBase::SIGNAL_NETINFO :
    setObject( new SignalNetinfo ); 
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }
}

IntegerType&
IntegerType::operator = (const IntegerType& value) {
  ((ScalarType &) *this) = (const ScalarType&) value;
  range = (const IntegerTypeInfo &) value.range;
  return *this;
}

IntegerType::IntegerType(ObjectBase::ObjectType objType, const TypeInfo& typeInfo) : ScalarType(objType) {
  ASSERT ( typeInfo.getKind() == RANGE_INFO );
  
  UniversalInteger initialValue((int) ((const IntegerTypeInfo &) typeInfo).get_left());
  
  switch(objType) {
  case ObjectBase::SIGNAL: 
    setObject( new Signal<UniversalInteger>(initialValue) );
    break;
  case ObjectBase::IMPLICIT_SIGNAL:
    setObject( new ImplicitSignal<UniversalInteger>(initialValue) );
    break;
  case ObjectBase::VARIABLE:
    setObject( new Variable<UniversalInteger>(initialValue) );
    break;
  case ObjectBase::SIGNAL_NETINFO:
    setObject( new SignalNetinfo );
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }
  //Copying the range Info

  range = (IntegerTypeInfo &) typeInfo;
}

IntegerType::IntegerType(ObjectBase::ObjectType objType, 
			 const UniversalInteger& val,
			 const TypeInfo& rInfo): 
  ScalarType(objType) {
  switch(objType) {
  case ObjectBase::SIGNAL: 
    setObject( new Signal<UniversalInteger>(val) );
    break;
  case ObjectBase::IMPLICIT_SIGNAL:
    setObject( new ImplicitSignal<UniversalInteger>(val) );
    break;
  case ObjectBase::VARIABLE:
    setObject( new Variable<UniversalInteger>(val) );
    break;
  case ObjectBase::SIGNAL_NETINFO:
    setObject( new SignalNetinfo );
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }
  //Copying the range Information
  switch (rInfo.getKind()) {
  case RANGE_INFO:
    range = (IntegerTypeInfo &) rInfo;
    break;
  case ENUM_INFO: {
    EnumerationTypeInfo &eInfo = (EnumerationTypeInfo &) rInfo;
    range = IntegerTypeInfo(eInfo.get_left(), eInfo.get_direction(),
		      eInfo.get_right());
    break;
  }
  default:
    cerr << "ERROR: Unable to handle the type information passed in.\n";
    ASSERT ( false );
    exit( -1 );
  }
}

IntegerType::IntegerType(ObjectBase::ObjectType objType, 
			 const IntegerType& value,
			 const TypeInfo& rInfo): 
  ScalarType(objType) {
  UniversalInteger val = (UniversalInteger&) value.readVal();
  switch(objType) {
  case ObjectBase::SIGNAL: 
    setObject( new Signal<UniversalInteger>(val) );
    break;
  case ObjectBase::IMPLICIT_SIGNAL:
    setObject( new ImplicitSignal<UniversalInteger>(val) );
    break;
  case ObjectBase::VARIABLE:
    setObject( new Variable<UniversalInteger>(val) );
    break;
  case ObjectBase::SIGNAL_NETINFO:
    setObject( new SignalNetinfo );
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }
  //Copying the range Information
  ASSERT ( rInfo.getKind() == RANGE_INFO );
  range = (IntegerTypeInfo &) rInfo;
} 

IntegerType::IntegerType(const IntegerType& value, 
			 const TypeInfo& rInfo): ScalarType(value) {
  ASSERT ( rInfo.getKind() == RANGE_INFO );
  range = (IntegerTypeInfo &) rInfo;
}


IntegerType::IntegerType(bool alias, const IntegerType& actual,
			 const TypeInfo& rInfo) 
  : ScalarType(actual.getObject()->getKind(), alias) {
  setObject( actual.getObject() );

  ASSERT ( rInfo.getKind() == RANGE_INFO );
  range = (IntegerTypeInfo &) rInfo;
}


Type 
IntegerType::get_kind() const {
  return INTEGER_TYPE;
}

IntegerType
IntegerType::vhdlPlus( const RValue &rhs ) const {
  UniversalInteger retval( getIntValue() + rhs.getIntValue() );
  return IntegerType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

IntegerType 
IntegerType::vhdlMinus( const RValue &rhs ) const {
  UniversalInteger retval( getIntValue() - rhs.getIntValue() );
  return IntegerType( ObjectBase::VARIABLE, retval, getTypeInfo() );
}

IntegerType
IntegerType::vhdlUnaryPlus() const {
  return *this;
}

IntegerType 
IntegerType::vhdlUnaryMinus() const {
  UniversalInteger retval( -1 * getIntValue() );
  return IntegerType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

IntegerType
IntegerType::vhdlMultiply( const RValue &rhs ) const {
  UniversalInteger retval( getIntValue() * rhs.getIntValue() );
  return IntegerType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

PhysicalType 
IntegerType::vhdlMultiply( const PhysicalType &rhs ) const {
  UniversalLongLongInteger retval( getIntValue() * rhs.getInt64Value() );
  return PhysicalType(ObjectBase::VARIABLE, retval, rhs.getTypeInfo());
}

IntegerType
IntegerType::vhdlDivide( const RValue &rhs ) const {
  UniversalInteger retval( getIntValue() / rhs.getIntValue() );
  return IntegerType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

IntegerType
IntegerType::vhdlMod( const RValue &rhs ) const {
  UniversalInteger retval = dynamic_cast<const UniversalInteger &>(readVal()).vhdlMod( rhs );
  return IntegerType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

IntegerType
IntegerType::vhdlRem( const RValue &rhs ) const {
  UniversalInteger retval = dynamic_cast<const UniversalInteger &>(readVal()).vhdlRem( rhs );
  return IntegerType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

IntegerType
IntegerType::vhdlPow( const RValue &rhs ) const {
  UniversalInteger retval = dynamic_cast<const UniversalInteger &>(readVal()).vhdlPow( rhs );
  return IntegerType( ObjectBase::VARIABLE, retval, getTypeInfo() );
}

IntegerType
IntegerType::vhdlAbs() const {
  return IntegerType( ObjectBase::VARIABLE, ::abs( getIntValue() ), getTypeInfo() );
}


//The attributes of IntegerType

IntegerType 
IntegerType::LEFT(const TypeInfo& rInfo){
  ASSERT ( rInfo.getKind() == RANGE_INFO );
  
  return IntegerType(ObjectBase::VARIABLE,
		     (int) ((const IntegerTypeInfo &) rInfo).get_left(), SavantintegerType_info);
}

IntegerType
IntegerType::RIGHT(const TypeInfo& rInfo){
  ASSERT ( rInfo.getKind() == RANGE_INFO );
  return IntegerType(ObjectBase::VARIABLE,
		     (int) ((const IntegerTypeInfo &) rInfo).get_right(), SavantintegerType_info);
}

IntegerType
IntegerType::HIGH(const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == RANGE_INFO );
  const IntegerTypeInfo &rInfo = (const IntegerTypeInfo &) tInfo;
  
  if (rInfo.get_left() > rInfo.get_right())
    return IntegerType(ObjectBase::VARIABLE, (int) rInfo.get_left());
  else
    return IntegerType(ObjectBase::VARIABLE, (int) rInfo.get_right());
}

IntegerType
IntegerType::LOW(const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == RANGE_INFO );
  const IntegerTypeInfo &rInfo = (const IntegerTypeInfo &) tInfo;
  
  if (rInfo.get_left() > rInfo.get_right())
    return IntegerType(ObjectBase::VARIABLE, (int)rInfo.get_right());
  else
    return IntegerType(ObjectBase::VARIABLE, (int)rInfo.get_left());
}

EnumerationType
IntegerType::ASCENDING(const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == RANGE_INFO );
  const IntegerTypeInfo &rInfo = (const IntegerTypeInfo &) tInfo;
  
  if (rInfo.get_direction() == ArrayInfo::to){
    return SAVANT_BOOLEAN_TRUE;
  }
  else {
    return SAVANT_BOOLEAN_FALSE;
  }
}
   
IntegerType
IntegerType::SUCC(const IntegerType & x){
  int val = x.getIntValue();
  if(x.range.get_direction() == ArrayInfo::to && val >= x.range.get_left() &&
     val < x.range.get_right()){
    val = val + 1;
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else if((x.range.get_direction() == ArrayInfo::downto) &&
	  (val < x.range.get_left()) &&
	  (val >= x.range.get_right())) {
    val = val + 1;
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The parameter of `SUCC attribute not within range ";abort();
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

IntegerType
IntegerType::PRED(const IntegerType & x){
  int val = x.getIntValue();
  if(x.range.get_direction() == ArrayInfo::to && val > x.range.get_left() &&
     val <= x.range.get_right()){
    val = val - 1;
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else if((x.range.get_direction() == ArrayInfo::downto) &&
	  (val <= x.range.get_left()) &&
	  (val > x.range.get_right())) {
    val = val - 1;
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The parameter of `PRED attribute not within range ";abort();
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

IntegerType
IntegerType::LEFTOF(const IntegerType & x){
  int val = x.getIntValue();
  if(x.range.get_direction() == ArrayInfo::to && val > x.range.get_left() &&
     val <= x.range.get_right()){
    val = val - 1;
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else if(x.range.get_direction() == ArrayInfo::downto && val < x.range.get_left() &&
	  val >= x.range.get_right()){
    val = val + 1;
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The parameter of `LEFTOF attribute not within range ";abort();
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

IntegerType
IntegerType::RIGHTOF(const IntegerType & x){
  int val = x.getIntValue();
  if(x.range.get_direction() == ArrayInfo::to && val >= x.range.get_left() &&
     val < x.range.get_right()){
    val = val + 1;
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else if(x.range.get_direction() == ArrayInfo::downto && val <= x.range.get_left() &&
	  val > x.range.get_right()){
    val = val - 1;
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The parameter of `RIGHTOF attribute not within range ";abort();
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

IntegerType
IntegerType::VAL(const IntegerType & x, const TypeInfo& tInfo){
  int val = x.getIntValue();
  
  ASSERT ( tInfo.getKind() == RANGE_INFO );
  const IntegerTypeInfo &rInfo = (const IntegerTypeInfo &) tInfo;
  
  if( ((rInfo.get_direction() == ArrayInfo::to) && (val >= rInfo.get_left()) &&
      (val <= rInfo.get_right())) ||
      ((rInfo.get_direction() == ArrayInfo::downto) && (val <= rInfo.get_left()) &&
       (val >= rInfo.get_right())) )  {
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The Result of `VAL attribute not withing range";abort();
    return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

IntegerType
IntegerType::VALUE(const ArrayType& x, const TypeInfo&) {
  int length = x.get_number_of_elements();
  ostringstream value;
  for(int i = 0; (i < length); i++) {
    x.get_element(i).print(value);
  }
  int endPos;
  int val = stringToLong( value.str(), endPos );
  ASSERT( endPos > 0 );
  IntegerType retval(ObjectBase::VARIABLE, UniversalInteger(val), SavantintegerType_info);
  return retval;
}

IntegerType
IntegerType::POS(const IntegerType &x, const TypeInfo&){
  int val = x.getIntValue();
  return IntegerType(ObjectBase::VARIABLE, val, x.getTypeInfo());
}

ArrayType
IntegerType::IMAGE( const IntegerType &x ){
  int val = x.getIntValue();
  string intString = intToString( val );
  return ArrayType(ObjectBase::VARIABLE, SavantstringType_info, -1, intString.c_str() );
}

const VHDLData&
IntegerType::leftValue() {
  static UniversalInteger retValue(0);
  
  retValue = UniversalInteger(range.get_left());
  
  return retValue;
}

TypeInfo&
IntegerType::getTypeInfo() const {
  return (TypeInfo &) range;
}
