// kdf11_sys.c - KDF11 Series CPU System Support Routines
//
// Copyright (c) 2002, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "pdp11/defs.h"
#include "pdp11/kdf11.h"

// *************************************************************
// **************** F11 - KW11L Line Time Clock ****************
// *************************************************************

void f11_ResetClock(F11_CPU *f11)
{
	MAP_IO *io = &f11->ioClock;

	// Reset speedometer.
	f11->clkCount    = 0;
	f11->cpu.opMeter = 0;

	// Reset Unibus registers.
	LTC = LTC_DONE;

	io->CancelInterrupt(io, 0);
}

void f11_Tick(void *dptr)
{
	F11_CPU *f11 = (F11_CPU *)dptr;
	MAP_IO  *io  = &f11->ioClock;

	LTC |= LTC_DONE;
	if (LTC & LTC_IE)
		io->SendInterrupt(io, 0);

	// Once each second.
	if (f11->clkCount++ >= 60) {
		f11->clkCount = 0;
#ifdef DEBUG
		if (dbg_Check(DBG_IPS)) {
			dbg_Printf("%s: Speedometer: %d ips\n",
				f11->cpu.Unit.devName, f11->cpu.opMeter);
			f11->cpu.opMeter = 0;
		}
#else /* DEBUG */
		printf("%s: Speedometer: %d ips\n",
			f11->cpu.Unit.devName, f11->cpu.opMeter);
		f11->cpu.opMeter = 0;
#endif /* DEBUG */
	}
}

int f11_ReadLTC(void *dptr, uint32 pAddr, uint16 *data, uint32 acc)
{
	F11_CPU *f11 = (F11_CPU *)dptr;

	*data = LTC;

#ifdef DEBUG
	if (dbg_Check(DBG_IOREGS))
		dbg_Printf("%s: (R) LTC (%o) => %06o\n",
			f11->cpu.Unit.devName, pAddr, *data);
#endif /* DEBUG */

	return UQ_OK;
}

int f11_WriteLTC(void *dptr, uint32 pAddr, uint16 data, uint32 acc)
{
	F11_CPU *f11 = (F11_CPU *)dptr;
	MAP_IO  *io  = &f11->ioClock;

	if (pAddr & 1)
		return UQ_OK;

	// Update IE or DONE bit on Clock CSR register.
	LTC = (data & LTC_RW) | (LTC & ~LTC_RW);
	if (data & LTC_DONE)
		LTC &= ~LTC_DONE;

	// Unless DONE+IE set, clear interrupt request.
	if (((LTC & LTC_DONE) == 0) || ((LTC & LTC_IE) == 0))
		io->CancelInterrupt(io, 0);

	// Test option for XXDP to pass tests because this
	// emulator is too fast to test line time click!
	if (LTC & LTC_IE)
		io->SendInterrupt(io, 0);

#ifdef DEBUG
	if (dbg_Check(DBG_IOREGS))
		dbg_Printf("%s: (W) LTC (%o) <= %06o (Now: %06o)\n",
			f11->cpu.Unit.devName, pAddr, data, LTC);
#endif /* DEBUG */

	return UQ_OK;
}

int f11_InitTimer(P11_CPU *p11)
{
	F11_CPU   *f11 = (F11_CPU *)p11;
	UQ_IO     *uq  = (UQ_IO *)p11->uqba;
	MAP_IO    *io  = &f11->ioClock;
	CLK_QUEUE *clk = &f11->clkTimer;

	// Initialize KW11L Line Time Clock.
	io->devName      = f11->cpu.Unit.devName;
	io->keyName      = LTC_KEY;
	io->emuName      = "KDF11: Line Time Clock";
	io->emuVersion   = NULL;
	io->Device       = f11;
	io->csrAddr      = LTC_CSRADR;
	io->nRegs        = LTC_NREGS;
	io->nVectors     = LTC_NVECS;
	io->intIPL       = LTC_IPL;
	io->intVector[0] = LTC_VEC;
	io->ReadIO       = f11_ReadLTC;
	io->WriteIO      = f11_WriteLTC;
	uq->Callback->SetMap(uq, io);

	// Reset Line Time Clock.
	f11_ResetClock(f11);

	// Initialize KW11L Tick service.
	clk->Next          = NULL;
	clk->Flags         = CLK_REACTIVE;
	clk->outTimer      = LTC_TICK;
	clk->nxtTimer      = LTC_TICK;
	clk->Device        = f11;
	clk->Execute       = f11_Tick;
	ts10_SetRealTimer(clk);

	return P11_OK;
}

void f11_StartTimer(P11_CPU *p11)
{
	F11_CPU *f11 = (F11_CPU *)p11;

	f11->clkCount        = 0;
	f11->cpu.opMeter     = 0;
	f11->clkTimer.Flags |= CLK_ENABLE;
}

void f11_StopTimer(P11_CPU *p11)
{
	F11_CPU *f11 = (F11_CPU *)p11;

	f11->clkTimer.Flags &= ~CLK_ENABLE;
}

// *************************************************************
// ************** F11 Series System Configuration **************
// *************************************************************

static uint16 intVectors[] = {
	VEC_RED,   // Red Stack
	VEC_ODD,   // Odd Address
	VEC_MME,   // Memory Management Error
	VEC_NXM,   // Non-existant Memory
	VEC_PAR,   // Parity Error
	VEC_PRV,   // Privileged Instruction
	VEC_PRV,   // Illegal Instruction
	VEC_BPT,   // Breakpoint Instruction
	VEC_IOT,   // Input/Output Trap Instruction
	VEC_EMT,   // Emulator Trap Instruction
	VEC_TRAP,  // Trap Instruction
	VEC_TRC,   // Trace (T-bit)
	VEC_YEL,   // Yellow Stack
	VEC_PWRFL, // Power Failure
	VEC_FPE    // Floating Point Error
};

inline void f11_ResetCPU(F11_CPU *f11)
{
	register P11_CPU *p11 = (P11_CPU *)f11;
	int idx;

	PIRQ = 0;
	MMR0 = 0;
	MMR1 = 0;
	MMR2 = 0;
	MMR3 = 0;

	for (idx = 0; idx < TRAP_MAX; idx++)
		p11->intVec[idx] = intVectors[idx];

	// Clear all memory management settings
	for (idx = 0; idx < APR_NREGS; idx++)
		APR(idx) = 0;
}

extern P11_INST pdp11_Inst[];

void f11_BuildCPU(F11_CPU *f11)
{
	register P11_CPU *cpu = (P11_CPU *)f11;
	uint32 idxInst, idxOpnd;

	// Clear all instruction tables
	for (idxInst = 0; idxInst < NUM_INST; idxInst++)
		cpu->tblOpcode[idxInst] = INSNAM(p11, ILL);

	// Build instruction table
	for (idxInst = 0; pdp11_Inst[idxInst].Name; idxInst++) {
		uint16 opCode = pdp11_Inst[idxInst].opCode;
		uint16 opReg  = pdp11_Inst[idxInst].opReg;

		if (pdp11_Inst[idxInst].Execute == NULL)
			continue;
		for (idxOpnd = opCode; idxOpnd < (opCode + opReg + 1); idxOpnd++)
			cpu->tblOpcode[idxOpnd] = pdp11_Inst[idxInst].Execute;
	}
}

void *f11_Create(MAP_DEVICE *newMap, int argc, char **argv)
{
	F11_CPU    *f11 = NULL;
	P11_SYSTEM *p11sys;
	CLK_QUEUE  *Timer;

	if (f11 = (F11_CPU *)calloc(1, sizeof(F11_CPU))) {
		f11->cpu.Unit.devName    = newMap->devName;
		f11->cpu.Unit.keyName    = newMap->keyName;
		f11->cpu.Unit.emuName    = newMap->emuName;
		f11->cpu.Unit.emuVersion = newMap->emuVersion;

		// First, link it to VAX system device.
		p11sys            = (P11_SYSTEM *)newMap->sysDevice;
		f11->cpu.System   = p11sys;
//		f11->cpu.Console  = p11_ConsoleInit((P11_CPU *)f11);
		p11sys->cpu       = (P11_CPU *)f11;

		f11->cpu.Flags   = CNF_F11;
		f11->cpu.cpuType = PID_F11;

		// Build instruction table
		f11_BuildCPU(f11);
		f11_ResetCPU(f11);

		// Set main memory (will be moved to 'set memory' command later
		p11_InitMemory((P11_CPU *)f11, 1024 * 1024);

		// Set function calls for KDF11 Series Processor.
		f11->cpu.InitTimer  = f11_InitTimer;
		f11->cpu.StartTimer = f11_StartTimer;
		f11->cpu.StopTimer  = f11_StopTimer;

		// Finally, link them to its mapping device.
		newMap->Device        = f11;
#ifdef DEBUG
//		newMap->Breaks        = &f11->cpu.Breaks;
#endif /* DEBUG */
	}

	return f11;
}

int f11_Reset(void *dptr)
{
	f11_ResetCPU(dptr);
}

int f11_Boot(MAP_DEVICE *map, int argc, char **argv)
{
	F11_CPU          *f11 = (F11_CPU *)map->Device;
	register P11_CPU *p11 = (P11_CPU *)f11;

	// Tell operator that processor is being booted.
	printf("Booting %s: (%s)... ", p11->Unit.devName, p11->Unit.keyName);

	// Initialize F11 Processor
	p11->State = P11_HALT;
	f11_ResetCPU(f11);

	// Initial Power On
	PC = 0173000;

	// Enable F11 Processor to run.
	// Finally, transfer control to
	// PDP-11 processor.
	p11->State = P11_RUN;
	emu_State  = P11_RUN;

	// Tell operator that it was done.
	printf("done.\n");

	return TS10_OK;
}

extern COMMAND p11_Commands[];
extern COMMAND p11_SetCommands[];
extern COMMAND p11_ShowCommands[];
extern DEVICE  uq11_Device;

DEVICE *f11_Devices[] =
{
	&uq11_Device,  // Unibus/QBus I/O Devices
	NULL           // Null terminator
};

// CVAX Series System Definition
DEVICE f11_System =
{
	// Description Information
	F11_KEY,             // Device Type Name
	F11_NAME,            // Emulator Name
	F11_VERSION,         // Emulator Version

	f11_Devices,         // Unibus/Qbus Device Listings
	DF_USE|DF_SYSMAP,    // Device Flags
	DT_PROCESSOR,        // Device Type

	p11_Commands,        // Commands
       	p11_SetCommands,     // Set Commands
	p11_ShowCommands,    // Show Commands

	f11_Create,          // Create Routine
	NULL,                // Configure Routine
	NULL,                // Delete Routine
	f11_Reset,           // Reset Routine
	NULL,                // Attach Routine
	NULL,                // Detach Routine
	NULL,                // Info Routine
	f11_Boot,            // Boot Routine
	NULL,                // Execute Routine
#ifdef DEBUG
	NULL,                // Debug Routine
#endif /* DEBUG */
};
