/* -*- mode: c; c-file-style: "gnu" -*-
 * misc.c -- Miscellaneous routines
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file misc.c
 * Miscellaneous routines.
 *
 * This module contains various unsorted helper functions, ranging
 * from date formatters (rfc822_date() and log_date()), to userdir
 * support functions to follow_ifowner().
 */

#include <sys/types.h>
#include <ctype.h>
#include <grp.h>
#include <pwd.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>

#include "compat/compat.h"

#include "config.h"
#include "fabs.h"
#include "misc.h"
#include "options.h"
#include "thy.h"

/** @internal Convert a hexadecimal value to int (char).
 */
static int
_misc_hextochar (int a, int b)
{
  int ret = 0;

  a = tolower(a);
  b = tolower(b);

  if (isdigit (a))
    ret += (a - '0') * 16;
  else
    {
      if (a >= 'a' && a <= 'f')
	ret += (a - 'a' + 10) * 16;
      else
	return 0;
    }

  if (isdigit (b))
    ret += b - '0';
  else
    {
      if (b >= 'a' && b <= 'f')
	ret += b - 'a' + 10;
      else
	return 0;
    }

  return ret;
}

/** Returns the position of the end of the username in URL.
 */
size_t
userpos (const char *url)
{
  size_t i = 2;

  /* Find the username */
  while ((url[i] != '/') && i < strlen(url)) i++;
  return i;
}

/** Determine the HOME directory of a user.
 * Return the HOME directory of USER, or NULL if there is no such
 * user.
 */
char *
userhome (const char *user)
{
  struct passwd *pwd;

  pwd = getpwnam (user);
  return (pwd) ? pwd->pw_dir : NULL;
}

/** Return the user HTML directory associated with URL.
 * Determine the user HTML directory associated with an URL.
 *
 * @param url is the URL to work with.
 * @param udir is the subdirectory of the user, where HTML documents
 * are stored.
 *
 * @returns The full path to the directory on success, in a newly
 * allocated string, or NULL on error.
 */
char *
userdir (const char *url, const char *udir)
{
  size_t i;
  char *user, *ud, *uh;
  size_t len_uh, len_udir;

  i = userpos (url);
  user = strndup (&url[2], i-2);
  user[i-2] = 0;

  uh = userhome (user);

  if (!uh)
    {
      free (user);
      return NULL;
    }

  len_uh = strlen (uh);
  len_udir = strlen (udir);

  ud = (char *)bhc_malloc (len_udir + len_uh + 2);
  memcpy (mempcpy (mempcpy (ud, uh, len_uh), "/", 1), udir, len_udir);
  ud [len_udir + len_uh + 1] = 0;

  free (user);
  user = fabs_realpath (ud);
  free (ud);
  return user;
}

/** Decode URL in-place.
 */
void
urldecode (char *url)
{
  char *p = url;

  while (*p)
    {
      if (*p == '%')
	{
	  *url = _misc_hextochar ((int)*(p+1), (int)*(p+2));
	  p += 2;
	}
      else
	*url = *p;

      p++;
      url++;
    }
  *url = '\0';
}

/** Convert a time value to RFC822 format.
 * Given WHEN, convert it to a date string in RFC822 format.
 *
 * @note The returned string points to an internal static string, and
 * should not be freed.
 */
const char *
rfc822_date (time_t when)
{
  struct tm *stm;
  static char buffer[128];

  stm = gmtime (&when);
  strftime (buffer, sizeof (buffer), "%a, %d %b %Y %H:%M:%S GMT", stm);
  return buffer;
}

#if THY_OPTION_LOGGING
/** Convert a time value to a format suitalbe for logging.
 * Given WHEN, convert it to a date string suitable for logging.
 *
 * @note The returned string points to an internal static string, and
 * should not be freed.
 */
const char *
log_date (time_t when)
{
  struct tm *stm;
  static char buffer[128];

  stm = gmtime (&when);
  strftime (buffer, sizeof (buffer), "%d/%b/%Y:%H:%M:%S %z", stm);
  return buffer;
}
#endif

/** Check if symlinks should be followed on a given path.
 * Walk through the path associated with the URL requested in SESSION,
 * and if any of the components are symlinks, check if the owner of he
 * link and the destination are the same.
 *
 * @returns One if all the owners match, zero otherwise.
 */
int
follow_ifowner (const session_t *session)
{
  struct stat st, stf;
  char *rest, *tmp;
  int i, sym = 0;
  thy_mappable_config_t *config =
    config_get_mapped (session->absuri, session->request->resolved);

  if (config->options.followall)
    {
      free (config);
      return 1;
    }
  free (config);

  rest = fabs_urlmap (session->request->url,
		      session->request->host,
		      session->absuri);
  i = strlen (rest);
  tmp = NULL;
  do
    {
      tmp = strdup (rest);
      lstat (tmp, &st);
      if (S_ISLNK (st.st_mode))
	{
	  stat (tmp, &stf);
	  if (st.st_uid != stf.st_uid)
	    {
	      free (rest);
	      free (tmp);
	      return 0;
	    }
	  else
	    sym = 1;
	}
      free (tmp);
      tmp = strrchr (rest, '/');
      if (tmp)
	{
	  i -= strlen (tmp);
	  if (i > 0)
	    rest[i] = '\0';
	}
    } while (i > 0);

  free (rest);
  return sym;
}

/** Drop privileges.
 * Drop privileges, if running as root.
 * Upon failure, it will make Thy exit.
 */
void
thy_priv_drop (uid_t uid)
{
  struct passwd *pwd;

  if (getuid () != 0)
    {
      bhc_log ("%s", "Not running as root, not dropping privileges.");
      return;
    }

  if (uid == 0)
    bhc_log ("%s", "Warning! Running as the super user!");

  if ((pwd = getpwuid (uid)) == NULL)
    {
      bhc_error ("getpwuid(): Could not get the user of uid %u. Exiting.",
		 uid);
      bhc_exit (-1);
    }

  if (initgroups (pwd->pw_name, pwd->pw_gid) == -1)
    {
      bhc_error ("initgroups(): %s", strerror (errno));
      bhc_exit (-1);
    }

  if (setgid (pwd->pw_gid) == -1)
    {
      bhc_error ("setgid(): %s", strerror (errno));
      bhc_exit (-1);
    }

  if (setuid (uid))
    {
      bhc_error ("setuid(): %s", strerror (errno));
      bhc_exit (-1);
    }
}
