/*

    File: rfs.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "rfs.h"
#include "fnctdsk.h"
#include "intrf.h"
static int set_rfs_info(const t_param_disk *disk_car, const struct reiserfs_super_block *sb,t_partition *partition, const int debug, const int dump_ind);
static int test_rfs(const t_param_disk *disk_car, const struct reiserfs_super_block *sb,t_partition *partition,const int debug, const int dump_ind);

int check_rfs(t_param_disk *disk_car,t_partition *partition,const int debug)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset+128*512)!=0) /* 64k offset */
  { return 1; }
  if(test_rfs(disk_car,(struct reiserfs_super_block*)&buffer,partition,debug,0)!=0)
    return 1;
  set_rfs_info(disk_car,(struct reiserfs_super_block*)&buffer,partition,debug,0);
  return 0;
}

static int test_rfs(const t_param_disk *disk_car, const struct reiserfs_super_block *sb,t_partition *partition,const int debug, const int dump_ind)
{
  if (memcmp(sb->s_magic,REISERFS_SUPER_MAGIC,sizeof(REISERFS_SUPER_MAGIC)) == 0)
  {
    partition->upart_type = UP_RFS;
  }
  else
    if(memcmp(sb->s_magic,REISERFS2_SUPER_MAGIC,sizeof(REISERFS2_SUPER_MAGIC)) == 0)
    {
      partition->upart_type = UP_RFS2;
    }
    else
    if(memcmp(sb->s_magic,REISERFS3_SUPER_MAGIC,sizeof(REISERFS3_SUPER_MAGIC)) == 0)
    {
      partition->upart_type = UP_RFS3;
    }
    else
      return 1;
  if(debug!=0)
    ecrit_rapport("\nReiserFS Marker at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
  /*
   * sanity checks.
   */

  if (le32(sb->s_block_count) < le32(sb->s_free_blocks))
    return (1);

  if (le32(sb->s_block_count) < REISERFS_MIN_BLOCK_AMOUNT)
    return (1);

  if ((le16(sb->s_state) != REISERFS_VALID_FS) &&
      (le16(sb->s_state) != REISERFS_ERROR_FS))
    return (1);

  if (le16(sb->s_oid_maxsize) % 2!=0) /* must be even */
    return (1);

  if (le16(sb->s_oid_maxsize) < le16(sb->s_oid_cursize))
    return (1);

  if ((le16(sb->s_blocksize) != 4096) && (le16(sb->s_blocksize) != 8192))
    return (1);

  return 0;
}

int recover_rfs(t_param_disk *disk_car, const struct reiserfs_super_block *sb,t_partition *partition,const int debug, const int dump_ind)
{
  if(test_rfs(disk_car,sb,partition,debug,dump_ind)!=0)
    return 1;
  if(debug!=0 || dump_ind!=0)
  {
    ecrit_rapport("\nrecover_rfs\n");
    ecrit_rapport("block_count=%u\n",(unsigned int)le32(sb->s_block_count));
    ecrit_rapport("block_size=%u\n",le16(sb->s_blocksize));
    if(dump_ind!=0)
    {
      dump(stdscr,sb,DEFAULT_SECTOR_SIZE);
    }
  }

  partition->part_size = (uint64_t)le32(sb->s_block_count) * le16(sb->s_blocksize);
  partition->part_type = (unsigned char)0x83;
  set_rfs_info(disk_car,sb,partition,debug,dump_ind);
  return 0;
}

static int set_rfs_info(const t_param_disk *disk_car,const struct reiserfs_super_block *sb,t_partition *partition, const int debug, const int dump_ind)
{
  partition->info[0]='\0';
  switch(partition->upart_type)
  {
    case UP_RFS:
      strncpy(partition->info,"ReiserFS 3.5 with standard journal",sizeof(partition->info));
      break;
    case UP_RFS2:
      strncpy(partition->info,"ReiserFS 3.6 with standard journal",sizeof(partition->info));
      set_part_name(partition,sb->s_label,16);
      break;
    case UP_RFS3:
      if(le16(sb->sb_version)==1)
	strncpy(partition->info,"ReiserFS 3.5 with non standard journal",sizeof(partition->info));
      else 
      if(le16(sb->sb_version)==2)
	strncpy(partition->info,"ReiserFS 3.6 with non standard journal",sizeof(partition->info));
      else
	strncpy(partition->info,"ReiserFS ? with non standard journal",sizeof(partition->info));
      set_part_name(partition,sb->s_label,16);
      break;
    default:
      return 1;
  }
  if (le16(sb->s_state) == REISERFS_ERROR_FS)
  {
    strcat(partition->info,", need recovery");
  }
  return 0;
}
