/*

    File: photorec.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <stdarg.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>	/* geteuid */
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#include <ctype.h>      /* toupper, tolower */
#ifdef HAVE_LOCALE_H
#include <locale.h>	/* setlocale */
#endif
#ifdef HAVE_SIGNAL
#include <signal.h>
#endif
#include <errno.h>
#include "types.h"
#include "common.h"
#include "lang.h"
#include "intrf.h"
#include "godmode.h"
#include "fnctdsk.h"
extern const t_arch_fnct arch_i386;
extern const t_arch_fnct arch_sun;
#define INTER_SELECT_X	0
#define INTER_SELECT_Y	23
#define INTER_SELECT	15
enum state_type { STATE_NONE, STATE_OPEN, STATE_WRITE };
typedef enum state_type state_type_t;

static void interface_select(t_param_disk *disk_car, const int debug,const int paranoid, const char *recup_dir);
static void interface_options_photorec(int *paranoid, int *allow_partial_last_cylinder,int *halt_on_errors, const t_arch_fnct **arch);
int do_curses_photorec(int debug, const char *recup_dir, const t_list_disk *list_disk, const t_arch_fnct **arch);
static int photorec(t_param_disk *disk_car, t_partition *partition, const int debug, const int paranoid, const char *recup_dir);

static FILE* init_log(const char*filename,int argc, char**argv);

static FILE *f_rapport=NULL;
static int f_status=0;

void ecrit_rapport_string(const char *string,const int max_length)
{
  int i;
  for(i=0;(string[i]!='\0')&&(i<max_length);i++)
    ecrit_rapport("%c",string[i]);
}

int ecrit_rapport(const char *_format, ...)
{
  int res=0;
  if(f_rapport!=NULL)
  {
    va_list ap;
    fflush(stderr);
    va_start(ap,_format);
    res=vfprintf(f_rapport,_format,ap);
    va_end(ap);
    if(res<0)
    {
      f_status=1;
    }
    if(fflush(f_rapport))
    {
      f_status=1;
    }
  }
  return res;
}

static FILE* init_log(const char*filename,int argc, char**argv)
{
  FILE*f_file=fopen(filename,"a");
  if(f_file==NULL)
  {
    printf("\nCan not create photorec.log: %s\n",strerror(errno));
    exit(1);
  }
  else
  {
	int i;
	time_t my_time;
	my_time=time(NULL);
	fprintf(f_file,"\n\n%s",ctime(&my_time));
	fprintf(f_file,"PhotoRec command line :");
	for(i=1;i<argc;i++)
	  fprintf(f_file," %s", argv[i]);
	fprintf(f_file,"\n");
	fflush(f_file);
	dup2(fileno(f_file),2);
  }
  return f_file;
}

#ifdef HAVE_SIGACTION
void sighup_hdlr(int shup)
{
  ecrit_rapport("SIGHUP detected! PhotoRec has been killed.\n");
  if(f_rapport!=NULL)
  {
    fclose(f_rapport);
  }
  exit(1);
}
#endif

static int photorec(t_param_disk *disk_car, t_partition *partition, const int debug, const int paranoid, const char *recup_dir)
{
  uint64_t offset;
  unsigned char buffer[disk_car->sector_size];
  unsigned int file_nbr=0;
  const char *extension=NULL;
  char filename_out[2048];
  state_type_t state=STATE_NONE;
  FILE *file_out=NULL;
  int ind_stop=0;
  const unsigned char jpg_header[4]= { 0xff,0xd8,0xff,0xe1};
  const unsigned char jpg2_header[4]= { 0xff,0xd8,0xff,0xe0};
  const unsigned char mov_header[7]= { 0x00,0x00,0x00,0x14,0x70,0x6E,0x6f};	/* Bad definition */
  const unsigned char mpg_header[4]= { 0x00,0x00,0x01,0xbA};
#ifdef RECOVER_MP3
  const unsigned char mp3_header[2]= { 0xFF, 0xFB };
  /* mp3 recovery is not a very good idea because it has been designed for streaming so a single mp3 file will be recovered in a lot of files */
#endif

  filename_out[sizeof(filename_out)-1]='\0';
  aff_buffer(BUFFER_RESET,"Q");
    aff_copy(stdscr);
    wmove(stdscr,4,0);
    wdoprintf(stdscr,"%s",disk_car->description(disk_car));
    mvwaddstr(stdscr,5,0,msg_PART_HEADER_LONG);
    wmove(stdscr,6,0);
    aff_part(stdscr,AFF_PART_ORDER,disk_car,partition);
    wrefresh(stdscr);
    wmove(stdscr,22,0);
    wattrset(stdscr,A_STANDOUT);
    waddstr(stdscr,"  Stop  ");
    wattroff(stdscr,A_STANDOUT);
    if(debug>0)
    {
      ecrit_rapport("debug=%u, paranoid=%u\n",debug,paranoid);
    }

  for(offset=partition->part_offset;(offset<partition->part_offset+partition->part_size-1) && !ind_stop;offset+=disk_car->sector_size)
  {
    if(disk_car->read(disk_car,sizeof(buffer), &buffer, offset)==0)
    {
      if(offset%123*0x200==0)
      {
	wmove(stdscr,9,0);
	wclrtoeol(stdscr);
	wdoprintf(stdscr,"Reading sector %10lu/%lu, %u pictures or videos found\n",(unsigned long)((offset-partition->part_offset)/disk_car->sector_size),(unsigned long)(partition->part_size/disk_car->sector_size),file_nbr);
	wrefresh(stdscr);
	switch(wgetch_nodelay(stdscr))
	{
	  case KEY_ENTER:
#ifdef PADENTER
	  case PADENTER:
#endif
	  case '\n':
	  case '\r':
	  case 's':
	  case 'S':
	    ind_stop=1;
	    break;
	}
      }
      if((memcmp(buffer,jpg_header,sizeof(jpg_header))==0)
	  ||(memcmp(buffer,jpg2_header,sizeof(jpg2_header))==0))
      {
	extension="jpg";
	state=STATE_OPEN;
	ecrit_rapport("%u pictures or videos found, %s found at sector %10lu, \n",file_nbr+1,extension,(unsigned long)((offset-partition->part_offset)/disk_car->sector_size));
      }
      if(memcmp(buffer,mov_header,sizeof(mov_header))==0)
      {
	extension="mov";
	state=STATE_OPEN;
	ecrit_rapport("%u pictures or videos found, %s found at sector %lu.\n",file_nbr+1,extension,(unsigned long)((offset-partition->part_offset)/disk_car->sector_size));
      }
      if(memcmp(buffer,mpg_header,sizeof(mpg_header))==0)
      {
	if((extension && strcmp(extension,"mpg"))||(state!=STATE_WRITE))
	{
	  extension="mpg";
	  state=STATE_OPEN;
	  ecrit_rapport("%u pictures or videos found, %s found at sector %10lu, \n",file_nbr+1,extension,(unsigned long)((offset-partition->part_offset)/disk_car->sector_size));
	}
      }
#ifdef RECOVER_MP3
      if(memcmp(buffer,mp3_header,sizeof(mp3_header))==0)
      {
	if((extension && strcmp(extension,"mp3"))||(state!=STATE_WRITE))
	{
	  extension="mp3";
	  state=STATE_OPEN;
	  ecrit_rapport("%u pictures or videos found, %s found at sector %10lu, \n",file_nbr+1,extension,(unsigned long)((offset-partition->part_offset)/disk_car->sector_size));
	}
      }
#endif
      if(paranoid==0)
      {
	unsigned int i;
	unsigned int car=buffer[0];
	for(i=0;(i<sizeof(buffer))&&(buffer[i]==car);i++);
	if(i==sizeof(buffer))
	{
	  state=STATE_NONE;
	  if(file_out)
	  {
	    fclose(file_out);
	    file_out=NULL;
	  }
	}
	else
	{
	  if(state==STATE_NONE)
	  {
	    extension="biz";
	    /* state=STATE_OPEN; */
	  }
	}
      }
      if(state==STATE_OPEN)
      {
	if(file_out)
	{
	  fclose(file_out);
	  file_out=NULL;
	}
	snprintf(filename_out,sizeof(filename_out)-1,"%s/%d.%s",recup_dir,++file_nbr,extension);
	if(!(file_out=fopen(filename_out,"wb")))
	{
	  perror("out");
	  state=STATE_NONE;
	}
	else
	  state=STATE_WRITE;
      }
      if(file_out && (state==STATE_WRITE))
	fwrite(buffer,sizeof(buffer),1,file_out);
    }
  }
  if(file_out)
  {
    fclose(file_out);
    file_out=NULL;
  }
  wmove(stdscr,9,0);
  wclrtoeol(stdscr);
  wdoprintf(stdscr,"%u pictures or videos found\n",file_nbr);
  if(ind_stop)
    ecrit_rapport("PhotoRec has been stopped, ");
  ecrit_rapport("%u pictures or videos found\n",file_nbr);
  aff_buffer(BUFFER_DISPLAY,"Q",stdscr);
  return 0;
}

int do_curses_photorec(int debug, const char *recup_dir, const t_list_disk *list_disk, const t_arch_fnct **arch)
{
  int command;
  int done=0;
  int allow_partial_last_cylinder=0;
  int halt_on_errors=1;
  int paranoid=1;
  const t_list_disk *element_disk;
  const t_list_disk *current_disk=list_disk;
  static struct MenuItem menuMain[]=
  {
	{'S',"Search","Search pictures"},
	{'O',"Options","Modify options"},
	{'Q',"Quit","Quit program"},
	{0,NULL,NULL}
  };
  if(list_disk==NULL)
    return 0;
/* use_env(TRUE); */
#ifdef HAVE_SETENV
#ifdef DJGPP
#elif defined(BSD)
    setenv("TERM","cons25",0);
#elif defined(LINUX)
    setenv("TERM","linux",0);
#elif defined(__CYGWIN__)
    setenv("TERM","cygwin",0);
#endif
#endif
#ifdef DJGPP
  initscr();
#else
  if(newterm(NULL,stdout,stdin)==NULL)
  {
#ifdef HAVE_SETENV
    setenv("TERMINFO",".",1);
#endif
    if(newterm(NULL,stdout,stdin)==NULL)
    {
      ecrit_rapport("Terminfo file is missing\n");
      printf("Terminfo file is missing\n");
      return 1;
    }
  }
#endif
  if(LINES<25)
  {
    ecrit_rapport("Terminal has only %u lines\n",LINES);
    endwin();
    return 1;
  }
  noecho();
#ifndef DJGPP
  nonl(); /*don't use for Dos version but enter will work with it... dilema */
#endif
  /*  intrflush(stdscr, FALSE); */
  cbreak();
  /* Should solve a problem with users who redefined the colors */
  if(has_colors())
  {
    start_color();
  }
  curs_set(0);
  if(mkdir(recup_dir, 0700)!=0 && errno==EEXIST)
  {
    display_message("WARNING: recovered pictures will overwrite the previous ones.");
  }
  /* ncurses interface */
  while(done==0)
  {
    aff_copy(stdscr);
    wmove(stdscr,5,0);
    for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
    {
      if(element_disk!=current_disk)
	wdoprintf(stdscr,"%s\n",element_disk->disk->description(element_disk->disk));
      else
      {
	wattrset(stdscr,A_STANDOUT);
	wdoprintf(stdscr,"%s\n",element_disk->disk->description(element_disk->disk));
	wattroff(stdscr,A_STANDOUT);
      }
    }
    current_disk->disk->halt_on_errors=halt_on_errors;
    command = wmenuSelect(stdscr,INTER_MAIN_Y, INTER_MAIN_X, menuMain, 8,
	  "SOQ", MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, 0);
    switch(command)
    {
      case 's':
      case 'S':
	interface_select(current_disk->disk, debug, paranoid,recup_dir);
	break;
      case 'q':
      case 'Q':
	done = 1;
	break;
      case 'o':
      case 'O':
	{
	  int old_allow_partial_last_cylinder=allow_partial_last_cylinder;
	  interface_options_photorec(&paranoid,&allow_partial_last_cylinder,&halt_on_errors,arch);
	  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
	  {
	    element_disk->disk->arch=*arch;
	  }
	  if(old_allow_partial_last_cylinder!=allow_partial_last_cylinder)
	    hd_parse_bis(list_disk,allow_partial_last_cylinder,*arch);
	}
	break;
      case MENU_UP:
	if(current_disk->prev!=NULL)
	  current_disk=current_disk->prev;
	break;
      case MENU_DOWN:
	if(current_disk->next!=NULL)
	  current_disk=current_disk->next;
	break;
    }
  }
  wclear(stdscr);
  wrefresh(stdscr);
  nl();
  endwin();
  ecrit_rapport("\n");
  return 0;
}

int main( int argc, char **argv )
{
  int i;
  int help=0, create_log=0, debug=0;
  const char *recup_dir="recup_dir";
  t_list_disk *list_disk=NULL;
  t_list_disk *element_disk;
#ifdef SOLARIS
  const t_arch_fnct *arch=&arch_sun;
#else
  const t_arch_fnct *arch=&arch_i386;
#endif
#ifdef HAVE_SIGACTION
  struct sigaction action, old_action;
#endif
#ifdef TESTING
  srand(1);
#endif
#ifdef HAVE_SIGACTION
  /* set up the signal handler for SIGHUP */
  action.sa_handler  = sighup_hdlr;
  action.sa_flags = 0;
  if(sigaction(SIGHUP, &action, &old_action)==-1)
  {
    printf("Error on SIGACTION call\n");
    return -1;
  }
#endif
#ifdef __CYGWIN__
  /* Average Windows user doesn't know how to specify a parameter */
  if(argc==1)
  {
    create_log=1;
    debug=1;
  }
#endif
  for(i=1;i<argc;i++)
  {
    if((strcmp(argv[i],"/log")==0) ||(strcmp(argv[i],"-log")==0))
      create_log=1;
    else if((strcmp(argv[i],"/debug")==0) || (strcmp(argv[i],"-debug")==0))
    {
      debug++;
      create_log=1;
    }
    else if(((strcmp(argv[i],"/d")==0)||(strcmp(argv[i],"-d")==0)) &&(i+1<argc))
    {
      recup_dir=argv[i+1];
    }
    else if((strcmp(argv[i],"/help")==0) || (strcmp(argv[i],"-help")==0) || (strcmp(argv[i],"--help")==0) ||
      (strcmp(argv[i],"/h")==0) || (strcmp(argv[i],"-h")==0))
      help=1;
    else
    {
      list_disk=insert_new_disk(list_disk,file_test_availability(argv[i],debug,arch));
      if(list_disk==NULL)
      {
	help=1;
      }
    }
  }
  printf("PhotoRec %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
  if(help!=0)
  {
    printf("\nUsage: photorec [/log] [/debug] [/d recup_dir] [file or device]"\
	    "\n" \
	    "\n/log          : create a photorec.log file" \
	    "\n/debug        : add debug information" \
	    "\n" \
	    "\nPhotoRec searches JPEG and MOV files, it stores them in recup_dir directory." \
	    "\n" \
	    "\nIf you have problems with PhotoRec or bug reports, please contacte me.\n");
    return 0;
  }
  if(create_log!=0)
  {
/*    const char *ext2fs_version=NULL; */
    f_rapport=init_log("photorec.log",argc,argv);
  ecrit_rapport("PhotoRec %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
#ifdef DJGPP
    ecrit_rapport("Dos version");
#elif defined(BSD)
    ecrit_rapport("BSD version");
#elif defined(LINUX)
    ecrit_rapport("Linux version");
#elif defined(__CYGWIN__)
    ecrit_rapport("Windows version");
#else
    ecrit_rapport("Undefined OS");
#endif
#ifdef COMPILE_BY
#ifdef COMPILE_HOST
#ifdef COMPILE_TIME
    ecrit_rapport(" (%s@%s, %s)",COMPILE_BY,COMPILE_HOST,COMPILE_TIME);
#endif
#endif
#endif
    ecrit_rapport("\n");
  }
  printf("Please wait...\n");
  {
    const char *locale;
    locale = setlocale (LC_ALL, "");
    if (locale==NULL) {
      locale = setlocale (LC_ALL, NULL);
      ecrit_rapport("Failed to set locale, using default '%s'.\n", locale);
    } else {
      ecrit_rapport("Using locale '%s'.\n", locale);
    }
  }
  aff_buffer(BUFFER_RESET,"Q");
  list_disk=hd_parse(list_disk,debug,arch);
  if(list_disk==NULL)
  {
    printf("No harddisk found\n");
#ifdef __CYGWIN__
    printf("You need to be administrator to use PhotoRec\n");
    printf("Under Win9x, use the DOS version instead\n");
    ecrit_rapport("You need to be administrator to use PhotoRec\n");
#else
#ifndef DJGPP
    if(geteuid()!=0)
    {
      printf("You need to be root to use PhotoRec\n");
      ecrit_rapport("You need to be root to use PhotoRec\n");
    }
#endif
#endif
  }
#ifdef DJGPP
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
  {
    printf("%s\n",element_disk->disk->description(element_disk->disk));
  }
#endif
  hd_parse_bis(list_disk,0,arch);
  /* save disk parameters to rapport */
  ecrit_rapport("Hard disk list\n");
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
  {
    printf("%s, sector size=%u\n",element_disk->disk->description(element_disk->disk),element_disk->disk->sector_size);
    ecrit_rapport("%s, sector size=%u\n",element_disk->disk->description(element_disk->disk),element_disk->disk->sector_size);
  }
  printf("\n");
  ecrit_rapport("\n");

  {
    if(do_curses_photorec(debug,recup_dir,list_disk,&arch))
    {
      printf("PhotoRec need 25 lines to work.\nPlease enlarge the terminal and restart PhotoRec.\n");
    }
  }
  for(element_disk=list_disk;element_disk!=NULL;)
  {
    t_list_disk *element_disk_next=element_disk->next;
    if(element_disk->disk->clean!=NULL)
      element_disk->disk->clean(element_disk->disk);
    FREE(element_disk->disk);
    FREE(element_disk);
    element_disk=element_disk_next;
  }
  if(f_rapport!=NULL)
  {
    ecrit_rapport("PhotoRec exited normally.\n");
    if(fclose(f_rapport))
    {
      f_status=1;
    }
  }
  if(f_status!=0)
  {
    printf("PhotoRec: Log file corrupted!\n");
  }
  else
  {
    printf("PhotoRec exited normally.\n");
  }
  return 0;
}

static void interface_options_photorec(int *paranoid, int *allow_partial_last_cylinder,int *halt_on_errors, const t_arch_fnct **arch)
{
  int done = FALSE;
  int menu = 4;
  while (done==FALSE)
  {
    int car;
    struct MenuItem menuOptions[]=
    {
      { 'P', NULL, "Search all partitions" },
      { 'A',NULL,"" },
      { 'H',NULL,"Halt on Disk I/O errors" },
      { 'T',NULL,"Partition table type"},
      { 'Q',"Quit","Return to main menu"},
      { 0, NULL, NULL }
    };
    menuOptions[0].name=*paranoid?"Paranoid : Yes":"Paranoid : No";
    menuOptions[1].name=*allow_partial_last_cylinder?"Allow partial last cylinder : Yes":"Allow partial last cylinder : No";
    menuOptions[2].name=*halt_on_errors?"Halt on errors : Yes":"Halt on errors : No";
    menuOptions[3].name=(*arch==&arch_i386?"Partition table type : Intel":"Partition table type : Sun");
    car=toupper( wmenuSelect(stdscr,INTER_OPTION_Y, INTER_OPTION_X, menuOptions, 0, "PAHTQ", MENU_VERT, menu));
    switch(car)
    {
      case 'P':
	*paranoid=!*paranoid;
	menu=0;
	break;
      case 'A':
	*allow_partial_last_cylinder=!*allow_partial_last_cylinder;
	menu=1;
	break;
      case 'H':
	*halt_on_errors=!*halt_on_errors;
	menu=2;
	break;
      case 'T':
       if(*arch==&arch_i386)
	 *arch=&arch_sun;
       else
	 *arch=&arch_i386;
	menu=3;
	break;
      case key_ESC:
      case 'Q':
	done = TRUE;
	break;
    }
  }
  /* write new options to log file */
  ecrit_rapport("New options :\n Paranoid : %s\n",
      *paranoid?"Yes":"No");
  ecrit_rapport("\n Allow partial last cylinder : %s\n Halt on errors : %s\n Partition table type : %s\n",
      *allow_partial_last_cylinder?"Yes":"No",
      *halt_on_errors?"Yes":"No",
      (*arch==&arch_i386?"Intel":"Sun"));
}

static void interface_select(t_param_disk *disk_car, const int debug,const int paranoid, const char *recup_dir)
{
  t_partition *fake_partition;
  t_list_part *list_part;
  t_list_part *element;
  t_list_part *current_element;
  int quit;
  int offset=0;
  int current_element_num=0;
  int rewrite=1;
  ecrit_rapport("\nInterface Select\n");
  list_part=disk_car->arch->read_part(disk_car,debug,0);
  fake_partition=partition_new();
  fake_partition->part_offset=0;
  fake_partition->part_size=disk_car->disk_size;
  fake_partition->part_type=disk_car->disk_size;
  strncpy(fake_partition->name,"Whole disk",sizeof(fake_partition->name)-1);
  list_part=insert_new_partition(list_part,fake_partition);
  current_element=list_part;
  for(element=list_part;element!=NULL;element=element->next)
  {
    aff_part_rapport(disk_car,element->part);
  }
  do
  {
    static struct MenuItem menuAdv[]=
    {
      {'s',"Search","Start photo recovery"},
      {'q',"Quit","Return to main menu"},
      {0,NULL,NULL}
    };
    int i;
    int command;
    if(rewrite!=0)
    {
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description(disk_car));
      mvwaddstr(stdscr,6,0,msg_PART_HEADER_LONG);
      rewrite=0;
    }
    for(i=0,element=list_part;(element!=NULL) && (i<offset);element=element->next,i++);
    for(i=offset;(element!=NULL) && ((i-offset)<INTER_SELECT);i++,element=element->next)
    {
      wmove(stdscr,5+2+i-offset,0);
      wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
      if(element==current_element)
      {
	wattrset(stdscr,A_STANDOUT);
	aff_part(stdscr,AFF_PART_ORDER,disk_car,element->part);
	wattroff(stdscr,A_STANDOUT);
      } else
      {
	aff_part(stdscr,AFF_PART_ORDER,disk_car,element->part);
      }
    }
    quit=0;
    command = wmenuSelect(stdscr,INTER_SELECT_Y, INTER_SELECT_X, menuAdv, 8,
	"sq", MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, 0);
    switch(command)
    {
      case MENU_UP:
	if(current_element!=NULL)
	{
	  if(current_element->prev!=NULL)
	  {
	    current_element=current_element->prev;
	    current_element_num--;
	  }
	  if(current_element_num<offset)
	    offset--;
	}
	break;
      case MENU_DOWN:
	if(current_element!=NULL)
	{
	  if(current_element->next!=NULL)
	  {
	    current_element=current_element->next;
	    current_element_num++;
	  }
	  if(current_element_num>=offset+INTER_SELECT)
	    offset++;
	}
	break;
      case 'q':
      case 'Q':
	quit=1;
	break;
      case 's':
      case 'S':
	if(current_element!=NULL)
	{
	  photorec(disk_car, current_element->part, debug, paranoid, recup_dir);
	  rewrite=1;
	}
	break;
    }
  } while(quit==0);
  delete_list_part(list_part);
}

void aff_copy(WINDOW *window)
{
  wclear(window);
  keypad(window, TRUE); /* Need it to get arrow key */
  wmove(window,0,0);
  wdoprintf(window, "PhotoRec %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
}


