# $Id: FormatStrategy.py,v 1.30 2003/01/20 06:43:02 chalky Exp $
#
# This file is a part of Synopsis.
# Copyright (C) 2000, 2001 Stephen Davies
# Copyright (C) 2000, 2001 Stefan Seefeld
#
# Synopsis is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
# 02111-1307, USA.
#
# $Log: FormatStrategy.py,v $
# Revision 1.30  2003/01/20 06:43:02  chalky
# Refactored comment processing. Added AST.CommentTag. Linker now determines
# comment summary and extracts tags. Increased AST version number.
#
# Revision 1.29  2002/12/30 13:48:17  chalky
# Use stylesheets for ModuleListing and classes' headings
#
# Revision 1.28  2002/12/23 13:49:23  chalky
# Fix namespace display
#
# Revision 1.27  2002/12/23 12:15:12  chalky
# Show separate "namespace foo::bar" on class pages, remove the namespace part
# from the class name to make it cleaner. Template decl is same font as class
# name.
#
# Revision 1.26  2002/12/19 05:31:03  chalky
# In class pages, show the template before the class
#
# Revision 1.25  2002/12/19 05:07:13  chalky
# Quote the names of classes
#
# Revision 1.24  2002/12/09 04:00:59  chalky
# Added multiple file support to parsers, changed AST datastructure to handle
# new information, added a demo to demo/C++. AST Declarations now have a
# reference to a SourceFile (which includes a filename) instead of a filename.
#
# Revision 1.23  2002/11/02 06:37:37  chalky
# Allow non-frames output, some refactoring of page layout, new modules.
#
# Revision 1.22  2002/11/01 07:21:15  chalky
# More HTML formatting fixes eg: ampersands and stuff
#
# Revision 1.21  2002/11/01 04:26:19  chalky
# Quote ampersand as modifier
#
# Revision 1.20  2002/11/01 03:39:21  chalky
# Cleaning up HTML after using 'htmltidy'
#
# Revision 1.19  2002/10/28 17:39:34  chalky
# Cross referencing support
#
# Revision 1.18  2002/10/28 08:16:53  chalky
# Undo previous table change. Put non-breaking spaces in first column instead
#
# Revision 1.17  2002/10/28 06:33:31  chalky
# Add heading for namespaces
#
# Revision 1.16  2002/10/28 06:13:49  chalky
# Fix summary display: templates use special div, use nested table to fix
# formatting of first column
#
# Revision 1.15  2002/10/27 12:05:17  chalky
# Support putting the identifier in the right place in funcptr parameters.
#
# Revision 1.14  2002/10/26 04:17:58  chalky
# Show templates on previous line. Hide constructors in base class. Commas
# between inherited members
#
# Revision 1.13  2002/10/25 05:13:36  chalky
# Show method names on new line after template definition
#
# Revision 1.12  2002/10/20 15:38:08  chalky
# Much improved template support, including Function Templates.
#
# Revision 1.11  2002/07/19 14:26:32  chalky
# Revert prefix in FileLayout but keep relative referencing elsewhere.
#
# Revision 1.10  2002/07/11 02:09:33  chalky
# Patch from Patrick Mauritz: Use png support in latest graphviz. If dot not
# present, don't subvert what the user asked for but instead tell them.
#
# Revision 1.9  2002/07/04 06:43:18  chalky
# Improved support for absolute references - pages known their full path.
#
# Revision 1.8  2002/04/26 01:21:14  chalky
# Bugs and cleanups
#
# Revision 1.7  2002/03/14 00:19:47  chalky
# Added demo of template specializations, and fixed HTML formatter to deal with
# angle brackets in class names :)
#
# Revision 1.6  2001/07/19 04:03:05  chalky
# New .syn file format.
#
# Revision 1.5  2001/07/15 08:28:43  chalky
# Added 'Inheritance' page Part
#
# Revision 1.4  2001/07/15 06:41:57  chalky
# Factored summarizer and detailer into 'Parts', and added a separate one for
# the top of the page (Heading)
#
# Revision 1.3  2001/07/11 01:45:03  stefan
# fix Dot and HTML formatters to adjust the references depending on the filename of the output
#
# Revision 1.2  2001/07/10 14:41:08  chalky
# Fix inlined inheritance graph filenames
#
# Revision 1.1  2001/07/10 05:08:28  chalky
# Refactored ASTFormatters into FormatStrategies, and simplified names all round
#
#
"""AST Formatting Strategies.

This module contains all the builtin formatting strategies used to make the
main scope pages. These strategies are used by Strategy.Summary/Detail to
create the entries for each declaration - each has a list of strategies that
it calls in turn to generate a HTML fragment for each declaration. So for
example, a link to source code is added just by adding a Strategy (eg:
FilePages) that outputs the link to the source. This can be done without
changing the strategies that generate the declaration name, type or
comments.
"""
# System modules
import types, os, string

# Synopsis modules
from Synopsis.Core import AST, Type, Util

# HTML modules
import Tags, core
from core import config
from Tags import *

class Strategy:
    """Generates HTML fragment for a declaration. Multiple strategies are
    combined to generate the output for a single declaration, allowing the
    user to customise the output by choosing a set of strategies. This follows
    the Strategy design pattern.

    The key concept of this class is the format* methods. Any
    class derived from Strategy that overrides one of the format methods
    will have that method called by the Summary and Detail formatters when
    they visit that AST type. Summary and Detail maintain a list of
    Strategies, and a list for each AST type.
    
    For example, when Strategy.Summary visits a Function object, it calls
    the formatFunction method on all Strategys registed with
    SummaryFormatter that implemented that method. Each of these format
    methods returns a string, which may contain a TD tag to create a new
    column.

    An important point to note is that only Strategies which override a
    particular format method are called - if that format method is not
    overridden then it is not called for that declaration type.
    """
    def __init__(self, formatter):
	"""Store formatter as self.formatter. The formatter is either a
	SummaryFormatter or DetailFormatter, and is used for things like
	reference() and label() calls. Local references to the formatter's
	reference and label methods are stored in self for more efficient use
	of them."""
	self.formatter = formatter
	self.label = formatter.label
	self.reference = formatter.reference
	self.formatType = formatter.formatType
	self.page = formatter.page()

    #
    # Utility methods
    #
    def formatModifiers(self, modifiers):
	"""Returns a HTML string from the given list of string modifiers. The
	modifiers are enclosed in 'keyword' spans."""
	def keyword(m):
	    if m == '&': return span('keyword', '&amp;')
	    return span('keyword', m)
	return string.join(map(keyword, modifiers))


    #
    # AST Formatters
    #
    def formatDeclaration(self, decl):	pass
    def formatForward(self, decl):	pass
    def formatGroup(self, decl):	pass
    def formatScope(self, decl):	pass
    def formatModule(self, decl):	pass
    def formatMetaModule(self, decl):	pass
    def formatClass(self, decl):	pass
    def formatTypedef(self, decl):	pass
    def formatEnum(self, decl):		pass
    def formatVariable(self, decl):	pass
    def formatConst(self, decl):	pass
    def formatFunction(self, decl):	pass
    def formatOperation(self, decl):	pass

class BaseAST(Strategy):
    """Base class for SummaryAST and DetailAST.
    
    The two classes SummaryAST and DetailAST are actually
    very similar in operation, and so most of their methods are defined here.
    Both of them print out the definition of the declarations, including type,
    parameters, etc. Some things such as exception specifications are only
    printed out in the detailed version.
    """
    col_sep = '<td class="summ-info">'
    row_sep = '</tr><tr><td class="summ-info">'
    whole_row = '<td class="summ-start" colspan="2">'
    def formatParameters(self, parameters):
	"Returns formatted string for the given parameter list"
        return string.join(map(self.formatParameter, parameters), ", ")

    def formatDeclaration(self, decl):
	"""The default is to return no type and just the declarations name for
	the name"""
	return self.col_sep + self.label(decl.name())

    def formatForward(self, decl): return self.formatDeclaration(decl)
    def formatGroup(self, decl):
	return self.col_sep + ''
    def formatScope(self, decl):
	"""Scopes have their own pages, so return a reference to it"""
        name = decl.name()
        link = rel(self.formatter.filename(), config.files.nameOfScope(name))
	return self.col_sep + href(link, anglebrackets(name[-1]))
    def formatModule(self, decl): return self.formatScope(decl)
    def formatMetaModule(self, decl): return self.formatModule(decl)
    def formatClass(self, decl): return self.formatScope(decl)

    def formatTypedef(self, decl):
	"(typedef type, typedef name)"
	type = self.formatType(decl.alias())
	return type + self.col_sep +  self.label(decl.name())

    def formatEnumerator(self, decl):
	"""This is only called by formatEnum"""
	self.formatDeclaration(decl)

    def formatEnum(self, decl):
	"(enum name, list of enumerator names)"
	type = self.label(decl.name())
	name = map(lambda enumor:enumor.name()[-1], decl.enumerators())
	name = string.join(name, ', ')
	return type + self.col_sep + name

    def formatVariable(self, decl):
	# TODO: deal with sizes
        type = self.formatType(decl.vtype())
	return type + self.col_sep + self.label(decl.name())

    def formatConst(self, decl):
	"(const type, const name = const value)"
	type = self.formatType(decl.ctype())
        name = self.label(decl.name()) + " = " + decl.value()
	return type + self.col_sep + name

    def formatFunction(self, decl):
	"(return type, func + params + formatFunctionExceptions)"
	premod = self.formatModifiers(decl.premodifier())
	type = self.formatType(decl.returnType())
	name = self.label(decl.name(), decl.realname())
	# Special C++ functions  TODO: maybe move to a separate AST formatter...
	if decl.language() == 'C++' and len(decl.realname())>1:
	    if decl.realname()[-1] == decl.realname()[-2]: type = '<i>constructor</i>'
	    elif decl.realname()[-1] == "~"+decl.realname()[-2]: type = '<i>destructor</i>'
	    elif decl.realname()[-1] == "(conversion)":
		name = "(%s)"%type
	params = self.formatParameters(decl.parameters())
	postmod = self.formatModifiers(decl.postmodifier())
	raises = self.formatOperationExceptions(decl)
	type = '%s %s'%(premod,type)
	# Prevent linebreaks on shorter lines
	if len(type) < 60:
	    type = replace_spaces(type)
        if decl.type() == "attribute": name = '%s %s %s'%(name, postmod, raises)
	else: name = '%s(%s) %s %s'%(name, params, postmod, raises)
	if decl.template():
	    templ = 'template &lt;%s&gt;'%(self.formatParameters(decl.template().parameters()),)
	    templ = div('template', templ)
	    return self.whole_row + templ + self.row_sep + type + self.col_sep + name
	return type + self.col_sep + name

    # Default operation is same as function, and quickest way is to assign:
    def formatOperation(self, decl): return self.formatFunction(decl)

    def formatParameter(self, parameter):
	"""Returns one string for the given parameter"""
	str = []
	keyword = lambda m,span=span: span("keyword", m)
	# Premodifiers
	str.extend(map(keyword, parameter.premodifier()))
	# Param Type
	id_holder = [parameter.identifier()]
	typestr = self.formatType(parameter.type(), id_holder)
	if typestr: str.append(typestr)
	# Postmodifiers
	str.extend(map(keyword, parameter.postmodifier()))
	# Param identifier
        if id_holder and len(parameter.identifier()) != 0:
            str.append(span("variable", parameter.identifier()))
	# Param value
	if len(parameter.value()) != 0:
	    str.append(" = " + span("value", parameter.value()))
	return string.join(str)

class SummaryAST (BaseAST):
    """Derives from BaseStrategy to provide summary-specific methods.
    Currently the only one is formatOperationExceptions"""
    def formatOperationExceptions(self, oper):
	"""Returns a reference to the detail if there are any exceptions."""
        if len(oper.exceptions()):
            return self.reference(oper.name(), " raises")
	return ''

class Default (Strategy):
    """A base AST strategy that calls formatDeclaration for all types"""
    # All these use the same method:
    def formatForward(self, decl): return self.formatDeclaration(decl)
    def formatGroup(self, decl): return self.formatDeclaration(decl)
    def formatScope(self, decl): return self.formatDeclaration(decl)
    def formatModule(self, decl): return self.formatDeclaration(decl)
    def formatMetaModule(self, decl): return self.formatDeclaration(decl)
    def formatClass(self, decl): return self.formatDeclaration(decl)
    def formatTypedef(self, decl): return self.formatDeclaration(decl)
    def formatEnum(self, decl): return self.formatDeclaration(decl)
    def formatVariable(self, decl): return self.formatDeclaration(decl)
    def formatConst(self, decl): return self.formatDeclaration(decl)
    def formatFunction(self, decl): return self.formatDeclaration(decl)
    def formatOperation(self, decl): return self.formatDeclaration(decl)
 
class SummaryCommenter (Default):
    """Adds summary comments to all declarations"""
    def formatDeclaration(self, decl):
	summary = config.comments.format_summary(self.page, decl)
	if summary:
	    return '<br>'+span('summary', summary)
	return ''
    def formatGroup(self, decl):
	"""Override for group to use the div version of commenting, and no
	<br> before"""
	summary = config.comments.format(self.page, decl)
	if summary:
	    return desc(summary)
	return ''
    
class SourceLinker (Default):
    """Adds a link to the decl on the file page to all declarations"""
    def formatDeclaration(self, decl):
	if not decl.file(): return ''
	filename = config.files.nameOfFileSource(decl.file().filename())
	line = decl.line()
	link = filename + "#%d" % line
	return href(rel(self.formatter.filename(), link), "[Source]")

class XRefLinker (Default):
    """Adds an xref link to all declarations"""
    def __init__(self, formatter):
	Default.__init__(self, formatter)
	self.xref = core.config.xref
    def formatDeclaration(self, decl):
	info = self.xref.get_info(decl.name())
	if not info:
	    return ''
	page = self.xref.get_page_for(decl.name())
	filename = config.files.nameOfSpecial('xref%d'%page)
	filename = filename + "#" + Util.quote(string.join(decl.name(), '::'))
	return href(rel(self.formatter.filename(), filename), "[xref]")
	
class Heading (Strategy):
    """Formats the top of a page - it is passed only the Declaration that the
    page is for (a Module or Class)."""

    def formatName(self, scoped_name):
	"""Formats a reference to each parent scope"""
	scope, text = [], []
	for name in scoped_name[:-1]:
	    scope.append(name)
	    text.append(self.reference(scope))
	text.append(anglebrackets(scoped_name[-1]))
	return string.join(text, "::\n") + '\n'

    def formatNameInNamespace(self, scoped_name):
	"""Formats a reference to each parent scope, starting at the first
	non-module scope"""
	scope, text = [], []
	for name in scoped_name[:-1]:
	    scope.append(name)
	    if config.types.has_key(scope):
		ns_type = config.types[scope]
		if isinstance(ns_type, Type.Declared):
		    decl = ns_type.declaration()
		    if isinstance(decl, AST.Module):
			# Skip modules (including namespaces)
			continue
	    text.append(self.reference(scope))
	text.append(anglebrackets(scoped_name[-1]))
	return string.join(text, "::\n") + '\n'

    def formatNamespaceOfName(self, scoped_name):
	"Formats a reference to each parent scope and this one"
	scope, text = [], []
	last_decl = None
	for name in scoped_name:
	    scope.append(name)
	    if config.types.has_key(scope):
		ns_type = config.types[scope]
		if isinstance(ns_type, Type.Declared):
		    decl = ns_type.declaration()
		    if isinstance(decl, AST.Module):
			# Only do modules and namespaces
			text.append(self.reference(scope))
			last_decl = decl
			continue
	    break
	return last_decl, string.join(text, "::") + '\n'

    def formatModule(self, module):
	"""Formats the module by linking to each parent scope in the name"""
	# Module details are only printed at the top of their page
	if not module.name():
	    type, name = "Global", "Namespace"
	else:
	    type = string.capitalize(module.type())
	    name = self.formatName(module.name())
	name = entity('h1', "%s %s"%(type, name))
	return name

    def formatMetaModule(self, module):
	"""Calls formatModule"""
	return self.formatModule(module)

    def formatClass(self, clas):
	"""Formats the class by linking to each parent scope in the name"""
	# Calculate the namespace string
	decl, namespace = self.formatNamespaceOfName(clas.name())
	if decl:
	    namespace = '%s %s'%(decl.type(), namespace)
	    namespace = div('class-namespace', namespace)
	else:
	    namespace = ''

	# Calculate template string
	templ = clas.template()
	if templ:
	    params = templ.parameters()
	    params = string.join(map(self.formatParameter, params), ', ')
	    templ = div('class-template', "template &lt;%s&gt;"%params)
	else:
	    templ = ''

	# Calculate class name string
	type = clas.type()
	name = self.formatNameInNamespace(clas.name())
	name = div('class-name', "%s %s"%(type, name))

	# Calculate file-related string
        file_name = rel(config.base_dir, clas.file().filename())
        # Try the file index page first
        file_link = config.files.nameOfFileIndex(clas.file().filename())
        if core.manager.filename_info(file_link):
            file_ref = href(rel(self.formatter.filename(), file_link), file_name, target="index")
        else:
            # Try source file next
            file_link = config.files.nameOfFileSource(clas.file().filename())
            if core.manager.filename_info(file_link):
                file_ref = href(rel(self.formatter.filename(), file_link), file_name)
            else:
                file_ref = file_name
	files = "Files: "+file_ref + "<br>"

	return '%s%s%s%s'%(namespace, templ, name, files)

    def formatParameter(self, parameter):
	"""Returns one string for the given parameter"""
	str = []
	keyword = lambda m,span=span: span("keyword", m)
	# Premodifiers
	str.extend(map(keyword, parameter.premodifier()))
	# Param Type
	typestr = self.formatType(parameter.type())
	if typestr: str.append(typestr)
	# Postmodifiers
	str.extend(map(keyword, parameter.postmodifier()))
	# Param identifier
        if len(parameter.identifier()) != 0:
            str.append(span("variable", parameter.identifier()))
	# Param value
	if len(parameter.value()) != 0:
	    str.append(" = " + span("value", parameter.value()))
	return string.join(str)

class DetailAST (BaseAST):
    """Derives BaseAST to provide detail-specific AST formatting."""

    col_sep = ' '
    row_sep = '<br>'
    whole_row = ''

    def formatOperationExceptions(self, oper):
	"""Prints out the full exception spec"""
        if len(oper.exceptions()):
            raises = span("keyword", "raises")
            exceptions = []
            for exception in oper.exceptions():
                exceptions.append(self.reference(exception.name()))
	    exceptions = span("raises", string.join(exceptions, ", "))
            return '%s (%s)'%(raises, exceptions)
	return ''

    def formatEnum(self, enum):
        name = span("keyword", "enum ") + self.label(enum.name())
        start = '<div class="enum">'
	enumors = string.join(map(self.formatEnumerator, enum.enumerators()))
        end = "</div>"
	return '%s%s%s%s'%(name, start, enumors, end)

    def formatEnumerator(self, enumerator):
        text = self.label(enumerator.name())
        if len(enumerator.value()):
            value = " = " + span("value", enumerator.value())
	else: value = ''
	comments = config.comments.format(self.page, enumerator)
	return '<div class="enumerator">%s%s%s</div>'%(text,value,comments)

class DetailCommenter (Default):
    """Adds summary comments to all declarations"""
    def formatDeclaration(self, decl):
	text = config.comments.format(self.page, decl)
	if text:
	    return desc(text)
	return ''

class ClassHierarchySimple (Strategy):
    "Prints a simple text hierarchy for classes"
    def formatInheritance(self, inheritance):
	return '%s %s'%( self.formatModifiers(inheritance.attributes()),
	    self.formatType(inheritance.parent()))

    def formatClass(self, clas):
	# Print out a list of the parents
	super = sub = ''
	if clas.parents():
	    parents = map(self.formatInheritance, clas.parents())
	    super = string.join(parents, ", ")
	    super = div('superclasses', "Superclasses: "+super)

	# Print subclasses
	subs = config.classTree.subclasses(clas.name())
	if subs:
	    refs = map(self.reference, subs)
	    sub = string.join(refs, ", ")
	    sub = div('subclasses', "Known subclasses: "+sub) 
	
	return super + sub

class ClassHierarchyGraph (ClassHierarchySimple):
    """Prints a graphical hierarchy for classes, using the Dot formatter.

    @see Formatter.Dot
    """
    def formatClass(self, clas):
        try:
            import tempfile
            from Synopsis.Formatter import Dot
        except:
            print "HierarchyGraph: Dot not found"
            return ""
	if 1:
	    super = config.classTree.superclasses(clas.name())
	    sub = config.classTree.subclasses(clas.name())
	    if len(super) == 0 and len(sub) == 0:
		# Skip classes with a boring graph
		return ''
        #label = config.files.nameOfScopedSpecial('inheritance', clas.name())
	label = self.formatter.filename()[:-5] + '-inheritance.html'
	tmp = os.path.join(config.basename, label)
        dot_args = ['-o', tmp, '-f', 'html', '-R', self.formatter.filename(), '-s', '-t', label]
        #if core.verbose: args.append("-v")
        Dot.toc = config.toc
        Dot.nodes = {}
	ast = AST.AST({}, [clas], config.types)
        Dot.format(dot_args, ast, None)
        text = ''
        input = open(tmp, "r+")
        line = input.readline()
        while line:
            text = text + line
            line = input.readline()
        input.close()
        os.unlink(tmp)
	return text

class Inheritance (Default):
    """Prints just the name of each declaration, with a link to its doc"""
    def formatDeclaration(self, decl, label=None):
	if not label: label = decl.name()[-1]
	fullname = Util.ccolonName(decl.name(), self.formatter.scope())
	title = decl.type() + " " + anglebrackets(fullname)
	return self.reference(decl.name(), label=label, title=title) + ' '

    def formatFunction(self, decl):
	return self.formatDeclaration(decl, label=decl.realname()[-1])

    def formatOperation(self, decl): return self.formatFunction(decl)
