/*==================================================================
 * pref.c - User interface routines for preferences
 *
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include <libswami/SwamiConfig.h>
#include <libswami/SwamiWavetbl.h>

#include "SwamiUIObject.h"
#include "SwamiUISpanWin.h"
#include "pref.h"
#include "glade_interface.h"
#include "i18n.h"
#include "util.h"

/* Prototypes */

static void pref_cb_path_browse_ok (GtkWidget *btn, GtkWidget *filesel);

static void pref_save_piano_keytable (GtkWidget *prefwin);
static void pref_cb_piano_key_change_octave (GtkToggleButton *radbtn,
					     GtkWidget *prefwin);
static void pref_load_piano_key_clist (GtkWidget *prefwin, gboolean lowoct);
static void pref_update_keycapture_msg (GtkWidget *prefwin);
static gboolean pref_cb_key_pressed (GtkWidget *prefwin, GdkEventKey *event);
static void pref_cb_keymap_change_clicked (GtkWidget *btn, GtkWidget *prefwin);
static void pref_cb_keymap_change_all_clicked (GtkWidget *btn,
					       GtkWidget *prefwin);
static gboolean pref_cb_pianokey_clist_event (GtkWidget *widget,
					      GdkEvent *event,
					      GtkWidget *prefwin);
static char *note_names[] =
{
  "C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B"
};

/* FIXME - determine FluidSynth drivers dynamically */
char *audio_drivers[] = {"alsa", "jack", "oss"};
char *midi_drivers[] = {"alsa_seq", "alsa", "oss"};


void
swamiui_pref_create (void)
{
  SwamiUISpanWin *spanwin;
  GtkWidget *prefwin;
  GtkWidget *widg;
  GtkWidget *menu, *item;
  guint *srckeys, *tmpkeys;
  char *s;
  int i, i2;

  if (swamiui_util_activate_unique_dialog ("pref", 0)) return;

  prefwin = create_prefwin ();

  swamiui_util_register_unique_dialog (prefwin, "pref", 0);

  /* to capture keys for piano key map */
  gtk_signal_connect (GTK_OBJECT (prefwin), "key_press_event",
		      GTK_SIGNAL_FUNC (pref_cb_key_pressed), prefwin);

  /* Initialize the General notebook page */

  s = swami_config_get_string ("gui", "patch_path");
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENPatchPath");
  gtk_entry_set_text (GTK_ENTRY (widg), s);

  s = swami_config_get_string ("gui", "sample_path");
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENSamplePath");
  gtk_entry_set_text (GTK_ENTRY (widg), s);

  s = swami_config_get_string ("swami", "search_path");
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENSearchPath");
  gtk_entry_set_text (GTK_ENTRY (widg), s);

  i = swami_config_get_int ("gui", "tips", NULL);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKTips");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), i != 0);

  i = swami_config_get_int ("gui", "splash", NULL);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKSplash");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), i != 0);

  i = swami_config_get_int ("gui", "restore_geometry", NULL);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKPaneSize");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), i != 0);

  i = swami_config_get_int ("gui", "quit_confirm", NULL);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPQuit");
  if (i <= 2) gtk_option_menu_set_history (GTK_OPTION_MENU (widg), i);

  i = swami_config_get_int ("gui", "save_geometry", NULL);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKSaveGeometry");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), i != 0);

  /* Initialize the Sound Font notebook page */

  i = swami_config_get_int ("swami", "temp_bank", NULL);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBTempBank");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i);

  i = swami_config_get_int ("swami", "temp_preset", NULL);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBTempPreset");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i);

  s = swami_config_get_string ("gui", "sample_left_postfix");
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENLeftChan");
  gtk_entry_set_text (GTK_ENTRY (widg), s);

  s = swami_config_get_string ("gui", "sample_right_postfix");
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENRightChan");
  gtk_entry_set_text (GTK_ENTRY (widg), s);

  i = swami_config_get_int ("swami", "swap_max_waste", NULL);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBSwapWaste");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i);


  /* Initialize the FluidSynth config page */

  s = swami_config_get_string ("fluidsynth", "audio_type");
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPAudioType");
  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (widg));

  /* FIXME - dynamically determine available drivers */
  for (i = 0, i2 = 0; i < (sizeof (audio_drivers)
			   / sizeof (audio_drivers[0])); i++)
    {
      /* if this driver is the current setting, save index for later */
      if (s && strcmp (s, audio_drivers[i]) == 0)
	i2 = i + 1;
      item = gtk_menu_item_new_with_label (audio_drivers[i]);
      gtk_object_set_data (GTK_OBJECT (item), "name", audio_drivers[i]);
      gtk_widget_show (item);
      gtk_menu_append (GTK_MENU (menu), item);
    }

  gtk_option_menu_set_history (GTK_OPTION_MENU (widg), i2);

  s = swami_config_get_string ("fluidsynth", "audio_device");
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENAudioDevice");
  gtk_entry_set_text (GTK_ENTRY (widg), s);

  /* use default audio device name when blank */
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKAudioDefault");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg),
				(!s || strlen (s) == 0));

  i = swami_config_get_int ("fluidsynth", "audio_bufsize", NULL);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBBufSize");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i);

  i = swami_config_get_int ("fluidsynth", "audio_bufcount", NULL);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBBufCount");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i);


  s = swami_config_get_string ("fluidsynth", "midi_type");
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPMidiType");
  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (widg));

  /* FIXME - dynamically determine available drivers */
  for (i = 0, i2 = 0; i < (sizeof (midi_drivers)
			   / sizeof (midi_drivers[0])); i++)
    {
      /* if this driver is the current setting, save index for later */
      if (s && strcmp (s, midi_drivers[i]) == 0)
	i2 = i + 1;
      item = gtk_menu_item_new_with_label (midi_drivers[i]);
      gtk_object_set_data (GTK_OBJECT (item), "name", midi_drivers[i]);
      gtk_widget_show (item);
      gtk_menu_append (GTK_MENU (menu), item);
    }

  gtk_option_menu_set_history (GTK_OPTION_MENU (widg), i2);

  s = swami_config_get_string ("fluidsynth", "midi_device");
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENMidiDevice");
  gtk_entry_set_text (GTK_ENTRY (widg), s);

  /* use default MIDI device name when blank */
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKMidiDefault");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg),
				(!s || strlen (s) == 0));

  /* Piano key mapping frame */

  /* FIXME */
  spanwin = SWAMIUI_SPANWIN (swamiui_lookup_object ("SwamiUISpanWin"));
  srckeys = swamiui_spanwin_get_keytable (spanwin);

  tmpkeys = g_malloc (sizeof (guint) * SWAMIUI_SPANWIN_PIANO_TOTAL_NUMKEYS);
  gtk_object_set_data_full (GTK_OBJECT (prefwin), "keys",
			    tmpkeys, (GtkDestroyNotify)g_free);

  memcpy (tmpkeys, srckeys,	/* copy key codes to temporary array */
	  sizeof (guint) * SWAMIUI_SPANWIN_PIANO_TOTAL_NUMKEYS);

  /* load piano key CLIST */
  pref_load_piano_key_clist (prefwin, TRUE);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CLSTkeys");
  gtk_signal_connect (GTK_OBJECT (widg), "event",
		      GTK_SIGNAL_FUNC (pref_cb_pianokey_clist_event),
		      prefwin);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "RADlowoct");
  gtk_signal_connect (GTK_OBJECT (widg), "toggled",
		      GTK_SIGNAL_FUNC (pref_cb_piano_key_change_octave),
		      prefwin);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "BTNChange");
  gtk_signal_connect (GTK_OBJECT (widg), "clicked",
		      GTK_SIGNAL_FUNC (pref_cb_keymap_change_clicked),
		      prefwin);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "BTNAll");
  gtk_signal_connect (GTK_OBJECT (widg), "clicked",
		      GTK_SIGNAL_FUNC (pref_cb_keymap_change_all_clicked),
		      prefwin);

  gtk_widget_show (prefwin);
}

/**
 * Apply changes in preferences dialog
 * @prefwin Preference dialog widget
 */
void
swamiui_pref_apply (GtkWidget *prefwin)
{
  GtkWidget *widg;
  SwamiUISpanWin *spanwin;
  char *s;
  int i;

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENPatchPath");
  s = gtk_entry_get_text (GTK_ENTRY (widg));
  swami_config_set_string ("gui", "patch_path", s);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENSamplePath");
  s = gtk_entry_get_text (GTK_ENTRY (widg));
  swami_config_set_string ("gui", "sample_path", s);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENSearchPath");
  s = gtk_entry_get_text (GTK_ENTRY (widg));
  swami_config_set_string ("swami", "search_path", s);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKTips");
  i = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg));
  swami_config_set_int ("gui", "tips", i);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKSplash");
  i = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg));
  swami_config_set_int ("gui", "splash", i);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKPaneSize");
  i = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg));
  swami_config_set_int ("gui", "restore_geometry", i);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPQuit");
  i = swamiui_util_option_menu_index (widg);
  swami_config_set_int ("gui", "quit_confirm", i);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKSaveGeometry");
  i = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg));
  swami_config_set_int ("gui", "save_geometry", i);

  /* Sound Font notebook page */

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBTempBank");
  i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  swami_config_set_int ("swami", "temp_bank", i);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBTempPreset");
  i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  swami_config_set_int ("swami", "temp_preset", i);
 
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENLeftChan");
  s = gtk_entry_get_text (GTK_ENTRY (widg));
  swami_config_set_string ("gui", "sample_left_postfix", s);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENRightChan");
  s = gtk_entry_get_text (GTK_ENTRY (widg));
  swami_config_set_string ("gui", "sample_right_postfix", s);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBSwapWaste");
  i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  swami_config_set_int ("swami", "swap_max_waste", i);

  /* FluidSynth config page */

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPAudioType");
  widg = gtk_option_menu_get_menu (GTK_OPTION_MENU (widg));
  widg = gtk_menu_get_active (GTK_MENU (widg));
  s = gtk_object_get_data (GTK_OBJECT (widg), "name");
  swami_config_set_string ("fluidsynth", "audio_type",s ? s : "");

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENAudioDevice");
  s = gtk_entry_get_text (GTK_ENTRY (widg));
  swami_config_set_string ("fluidsynth", "audio_device", s);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBBufSize");
  i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  swami_config_set_int ("fluidsynth", "audio_bufsize", i);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBBufCount");
  i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  swami_config_set_int ("fluidsynth", "audio_bufcount", i);


  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPMidiType");
  widg = gtk_option_menu_get_menu (GTK_OPTION_MENU (widg));
  widg = gtk_menu_get_active (GTK_MENU (widg));
  s = gtk_object_get_data (GTK_OBJECT (widg), "name");
  swami_config_set_string ("fluidsynth", "midi_type", s ? s : "");

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENMidiDevice");
  s = gtk_entry_get_text (GTK_ENTRY (widg));
  swami_config_set_string ("fluidsynth", "midi_device", s);


  /* Piano key mapping frame */
  pref_save_piano_keytable (prefwin);

  /* FIXME */
  spanwin = SWAMIUI_SPANWIN (swamiui_lookup_object ("SwamiUISpanWin"));
  swamiui_spanwin_update_keytable ();
}

/** user clicked "OK" button on preferences dialog  */
void
swamiui_pref_cb_okay_clicked (GtkWidget *btn, GtkWidget *prefwin)
{
  swamiui_pref_apply (prefwin);
  gtk_widget_destroy (prefwin);
}

/** user clicked "Save Preferences" button save SwamiConfig */
void
swamiui_pref_cb_save_pref_clicked (GtkWidget *btn, GtkWidget *prefwin)
{
  swamiui_pref_apply (prefwin);
  swami_config_save (FALSE);
}

/** (glade) pop browse dialog for default sfont and sample paths */
void
swamiui_pref_cb_path_browse_clicked (GtkWidget *btn, GtkWidget *entry)
{
  GtkWidget *filesel;
  gchar *s;

#if 0
  dirsel = dirbrowse_create (_("Choose directory"));

  gtk_object_set_data (GTK_OBJECT (dirsel), "entry", entry);
#endif

  filesel = gtk_file_selection_new (_("Choose directory"));

  gtk_object_set_data (GTK_OBJECT (filesel), "entry", entry);

  s = gtk_entry_get_text (GTK_ENTRY (entry));
  if (strlen (s))
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), s);

  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
    "clicked", (GtkSignalFunc) pref_cb_path_browse_ok, filesel);

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->
      cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy,
    GTK_OBJECT (filesel));

  gtk_widget_show (filesel);
}

/* user says Okay to selected path in file dialog */
static void
pref_cb_path_browse_ok (GtkWidget * btn, GtkWidget * filesel)
{
  GtkWidget *entry;
  gchar *s;

  s = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filesel));

  entry = gtk_object_get_data (GTK_OBJECT (filesel), "entry");
  gtk_entry_set_text (GTK_ENTRY (entry), s);

  gtk_widget_destroy (filesel);
}

static void
pref_save_piano_keytable (GtkWidget *pref)
{
  guint *tmpkeys;
  char *s;
  int i;

  tmpkeys = gtk_object_get_data (GTK_OBJECT (pref), "keys");

  /* check if piano key map is the default */
  for (i = 0; i < SWAMIUI_SPANWIN_PIANO_TOTAL_NUMKEYS; i++)
    if (tmpkeys[i] != swamiui_spanwin_default_keytable[i])
      break;

  if (i == SWAMIUI_SPANWIN_PIANO_TOTAL_NUMKEYS)  /* same as default? */
    {
      swami_config_set_string ("gui", "piano_lowoctkeys", "");
      swami_config_set_string ("gui", "piano_hioctkeys", "");
      return;
    }

  /* keytable is not the same as the default one */

  s = swamiui_spanwin_encode_octkeys (tmpkeys,
				      SWAMIUI_SPANWIN_PIANO_LOW_NUMKEYS);
  if (s)
    {
      swami_config_set_string ("gui", "piano_lowoctkeys", s);
      g_free (s);
    }

  s = swamiui_spanwin_encode_octkeys (tmpkeys +
				      SWAMIUI_SPANWIN_PIANO_LOW_NUMKEYS,
				      SWAMIUI_SPANWIN_PIANO_HI_NUMKEYS);
  if (s)
    {
      swami_config_set_string ("gui", "piano_hioctkeys", s);
      g_free (s);
    }
}

static void
pref_cb_piano_key_change_octave (GtkToggleButton *radbtn, GtkWidget *prefwin)
{
  pref_load_piano_key_clist (prefwin, gtk_toggle_button_get_active (radbtn));
}

static void
pref_load_piano_key_clist (GtkWidget *prefwin, gboolean lowoct)
{
  GtkWidget *clist;
  guint *keyp;
  char *coltxt[2];
  char note[5];
  int i, c;

  clist = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (prefwin), "CLSTkeys"));
  keyp = gtk_object_get_data (GTK_OBJECT (prefwin), "keys");

  if (lowoct) c = SWAMIUI_SPANWIN_PIANO_LOW_NUMKEYS;
  else
    {
      c = SWAMIUI_SPANWIN_PIANO_HI_NUMKEYS;
      keyp += SWAMIUI_SPANWIN_PIANO_LOW_NUMKEYS;
    }

  gtk_clist_clear (GTK_CLIST (clist));

  coltxt[0] = note;

  for (i = 0; i < c; i++)
    {
      sprintf (note, "%s-%d", note_names[i % 12], i / 12);

      if (!(coltxt[1] = gdk_keyval_name (*(keyp++))))
	coltxt[1] = _("INVALID");

      gtk_clist_append (GTK_CLIST (clist), coltxt);
    }

  gtk_object_set_data (GTK_OBJECT (clist), "lowoct", GINT_TO_POINTER (lowoct));
}

static void
pref_update_keycapture_msg (GtkWidget *prefwin)
{
  GtkWidget *lbl;
  char *s;

  lbl = gtk_object_get_data (GTK_OBJECT (prefwin), "LBLkeymsg");

  if (gtk_object_get_data (GTK_OBJECT (prefwin), "capture"))
    s = _("Press a key, click list to cancel");
  else
    s = "";

  gtk_label_set_text (GTK_LABEL (lbl), s);
}

static gboolean
pref_cb_key_pressed (GtkWidget *prefwin, GdkEventKey *event)
{
  GtkWidget *clist;
  guint *tmpkeys;
  int keyndx;
  int lowoct;
  char *keystr;
  gboolean change_all;

  if (!gtk_object_get_data (GTK_OBJECT (prefwin), "capture"))
    return (FALSE);

  clist = gtk_object_get_data (GTK_OBJECT (prefwin), "CLSTkeys");

  if (!GTK_CLIST (clist)->selection)
    return (FALSE);

  keyndx = GPOINTER_TO_UINT (GTK_CLIST (clist)->selection->data);

  lowoct = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (clist),
						 "lowoct"));
  if ((lowoct && keyndx >= SWAMIUI_SPANWIN_PIANO_LOW_NUMKEYS)
      || (!lowoct && keyndx >= SWAMIUI_SPANWIN_PIANO_HI_NUMKEYS))
    return (FALSE);

  keystr = gdk_keyval_name (event->keyval);

  if (!keystr) return (FALSE);

  tmpkeys = gtk_object_get_data (GTK_OBJECT (prefwin), "keys");
  tmpkeys[keyndx + (lowoct ? 0 : SWAMIUI_SPANWIN_PIANO_LOW_NUMKEYS)]
    = event->keyval;

  gtk_clist_set_text (GTK_CLIST (clist), keyndx, 1, keystr);

  change_all = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (prefwin),
						     "all"));
  if (change_all && (++keyndx < (lowoct ? SWAMIUI_SPANWIN_PIANO_LOW_NUMKEYS :
				 SWAMIUI_SPANWIN_PIANO_HI_NUMKEYS)))
    {
      gtk_clist_moveto (GTK_CLIST (clist), keyndx, 0, 0.5, 0.0);
      gtk_clist_select_row (GTK_CLIST (clist), keyndx, 0);
      return (FALSE);
    }
  else if (change_all)		/* turn off capture of all keys */
    gtk_object_set_data (GTK_OBJECT (prefwin), "all", GINT_TO_POINTER (FALSE));

  gtk_object_set_data (GTK_OBJECT (prefwin), "capture",
		       GINT_TO_POINTER (FALSE));
  pref_update_keycapture_msg (prefwin);

  return (FALSE);
}

static void
pref_cb_keymap_change_clicked (GtkWidget *btn, GtkWidget *prefwin)
{
  GtkWidget *clist;
  guint keyndx;
  int lowoct;

  clist = gtk_object_get_data (GTK_OBJECT (prefwin), "CLSTkeys");

  if (!GTK_CLIST (clist)->selection)   /* an item must be selected */
    return;

  keyndx = GPOINTER_TO_UINT (GTK_CLIST (clist)->selection->data);
  lowoct = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (clist),
						 "lowoct"));
  if ((lowoct && keyndx >= SWAMIUI_SPANWIN_PIANO_LOW_NUMKEYS)
      || (!lowoct && keyndx >= SWAMIUI_SPANWIN_PIANO_HI_NUMKEYS))
    return;

  gtk_object_set_data (GTK_OBJECT (prefwin), "capture",
		       GINT_TO_POINTER (TRUE));
  pref_update_keycapture_msg (prefwin);
}

static void
pref_cb_keymap_change_all_clicked (GtkWidget *btn, GtkWidget *prefwin)
{
  GtkWidget *clist;

  clist = gtk_object_get_data (GTK_OBJECT (prefwin), "CLSTkeys");
  gtk_clist_select_row (GTK_CLIST (clist), 0, 0);
  gtk_object_set_data (GTK_OBJECT (prefwin), "all", GINT_TO_POINTER (TRUE));

  pref_cb_keymap_change_clicked (btn, prefwin);
}

static gboolean
pref_cb_pianokey_clist_event (GtkWidget *widget, GdkEvent *event,
			      GtkWidget *prefwin)
{
  if (!gtk_object_get_data (GTK_OBJECT (prefwin), "capture"))
    return (FALSE);

  switch (event->type)
    {
    case GDK_MOTION_NOTIFY:
    case GDK_EXPOSE:
      /* do nothing */
      break;
    default:
      gtk_object_set_data (GTK_OBJECT (prefwin), "capture",
			   GINT_TO_POINTER (FALSE));
      pref_update_keycapture_msg (prefwin);
      break;
    }

  return (FALSE);
}
