import pygtk
pygtk.require('2.0')
import gtk
import straw
import time

class FindLimit(object):
    __slots__ = ('text', 'start', 'end')

    def __init__(self, text, start = None, end = None):
        self.text = text
        self.start = start
        self.end = end

    def is_subset(self, fl):
        if self.text.find(fl.text) == -1:
            return 0
        if fl.end is not None and (self.end is None or self.end > fl.end):
                return 0
        if fl.start is not None and (self.start is None or
                                     self.start < fl.start):
            return 0
        return 1

    def equals(self, fl):
        return (self.text == fl.text and
                self.end == fl.end and
                self.start == fl.start)

    def time_contains(self, item):
        if item.pub_date is None:
            # pub_date hasn't been initialized in older versions of straw
            return 1
        if self.start is not None:
            if self.start > item.pub_date:
                return 0
        if self.end is not None:
            if self.end < item.pub_date:
                return 0
        return 1

class FindResults:
    """ stack of find results """
    def __init__(self, feeds):
        self._feeds = feeds
        self._stack = []

    def gather_all_items(self):
        il = []
        for f in self._feeds:
            il += f.items
        return il

    def find_matching(self, limit, items):
        matches = []
        for item in items:
            if limit.time_contains(item):
                if item.match(limit.text):
                    matches.append(item)
        return matches

    def find(self, limit):
        if not len(limit.text):
            # empty string, clear out the stack and return empty list
            self._stack = []
            return []
        if len(self._stack):
            if limit.is_subset(self._stack[-1][0]):
                # the new search string is longer than the previous
                if len(self._stack[-1][1]):
                    # only search when there were results with a shorter text
                    m = self.find_matching(limit, self._stack[-1][1])
                    self._stack.append((limit, m))
                    return m
            else:
                # text was deleted
                i = len(self._stack)
                foundprev = 0
                while i > 0:
                    i -= 1
                    if limit.equals(self._stack[i][0]):
                        foundprev = 1
                        break
                    if limit.is_subset(self._stack[i][0]):
                        break
                self._stack = self._stack[:i+1]
                if foundprev:
                    return self._stack[-1][1]
                if not len(self._stack):
                    items = self.gather_all_items()
                else:
                    items = self._stack[-1][1]
                m = self.find_matching(limit, items)
                self._stack.append((limit, m))
                return m
        else:
            m = self.find_matching(limit, self.gather_all_items())
            self._stack.append((limit, m))
            return m

class FindDialog(straw.SignalEmitter):

    SECS_PER_DAY = 86400

    def __init__(self, xml):
        straw.SignalEmitter.__init__(self)
        self.initialize_slots(straw.FindInterruptSignal)
        self._window = xml.get_widget("find_dialog")
        self._window.set_transient_for(straw.main_window.get_window())
        nameFuncMap = {}
        for key in dir(self.__class__):
            if key[:3] == 'on_':
                nameFuncMap[key] = getattr(self, key)
        xml.signal_autoconnect(nameFuncMap)
        self._fr = FindResults(straw.FeedList.get_instance())
        self._start_date_edit = xml.get_widget("find_start_date_edit")
        self._end_date_edit = xml.get_widget("find_end_date_edit")
        self._text_entry = xml.get_widget("find_text_entry")

        self._current_text = ""
        self._current_start_date = None
        self._current_end_date = None
        straw.main_window.get_find_results_view().register_find_dialog(self)
        self._in_render = 0

    def show(self):
        self._window.present()
        self._text_entry.grab_focus()

    def hide(self):
        self._window.hide()
        straw.main_window.hide_find_mode()

    def on_find_close_button_clicked(self, *args):
        self.hide()

    def on_find_text_entry_insert_text(
        self, widget, text, length, position, data=None):
        gtk.timeout_add(1, self.after_change)

    def on_find_text_entry_delete_text(self, widget, start, end):
        gtk.timeout_add(1, self.after_change)
        return
        
    def after_change(self):
        widget = self._text_entry
        newtext = widget.get_text()
        if self._in_render:
            # terminate an existing result rendering in progress
            self.emit_signal(straw.FindInterruptSignal(self))
        changed = 0
        if newtext != self._current_text:
            changed = 1
            self._current_text = newtext
        while changed and not self._in_render:
            self._in_render = 1
            t = self._current_text
            limit = FindLimit(
                t, self._current_start_date, self._current_end_date)
            results = self._fr.find(limit)
            straw.main_window.show_find_results(results or [])
            changed = t != self._current_text
            self._in_render = 0
        return

    def on_find_start_time_limit_check_toggled(self, button, data=None):
        state = button.get_active()
        self._start_date_edit.set_sensitive(state)
        if state:
            self.read_start_date()
        else:
            self._current_start_date = None

    def on_find_end_time_limit_check_toggled(self, button, data=None):
        state = button.get_active()
        self._end_date_edit.set_sensitive(state)
        if state:
            self.read_end_date()
        else:
            self._current_end_date = None

    def on_find_start_date_edit_date_changed(self, widget):
        self.read_start_date()

    def on_find_end_date_edit_date_changed(self, widget):
        self.read_end_date()

    def on_find_dialog_delete_event(self, *args):
        self.hide()
        return gtk.TRUE

    def read_start_date(self):
        date = time.gmtime(self._start_date_edit.get_time())
        if date != self._current_start_date:
            limit = FindLimit(
                self._current_text, date, self._current_end_date)
            straw.main_window.show_find_results(self._fr.find(limit))
            self._current_start_date = date

    def read_end_date(self):
        date = time.gmtime(self._end_date_edit.get_time() + self.SECS_PER_DAY)
        if date != self._current_end_date:
            limit = FindLimit(
                self._current_text, self._current_start_date, date)
            straw.main_window.show_find_results(self._fr.find(limit))
            self._current_end_date = date
