#!/usr/local/bin/perl
#
# This object represents an list of acknowledgments associated with a host.  It
# knows how to load the Acks associated with a given host (since there can be
# more then one).  Acks do not know how to load themselves, they must be 
# brought to life from an AckList object.
#
#   host          - string FQDN of the machine its associated with
#   acks          - list of Ack objects
#
# + new()         - constructor (sets instance vars to arguments passed in)
# + gets/sets()   - magical set/get functions (autoloaded based on func name)
# + display()     - output format and view
#
# History:
# (1) Cleaned up (Ed July 30, 1997);

use Spong::Ack;

package Spong::AckList;

# Constructor.  This requires the FQDN of the host that the Acks are associated
# with.  If there are no Acks associated with that host, then undef is returned
# otherwise the AckList object is returned and the list of Acks are loaded.

sub new {
   my( $class, $host ) = @_;
   my( @list, $file, $message );
   my $self = {};


   if( $host ) {
      # Go through the list of acknowledgments for this host and add Ack
      # objects for each one.
      
      opendir( DIR, "$main::SPONGDB/$host/acks" );
      while( defined( $file = readdir( DIR ) ) ) {
	 if( $file =~ /^\d+-(\d+)-(\d+)$/ ) {
	    my( $s, $e ) = ($1, $2);
	    if( time() >= $s && time() <= $e ) {
	       $message = "";
	       open( FILE, "$main::SPONGDB/$host/acks/$file" );
	       my $header = <FILE>; chomp $header;
	       my( $user, $serv ) = ( $header =~ /^(\S+) (.*)$/ );
	       while( <FILE> ) { $message .= $_; }
	       close( FILE );
	       
	       push(@list, 
		    Spong::Ack->new( $host, $s, $e, $user, $serv, $message ));
	    }
	 }
      }
      closedir( DIR );
      
      # If we found any, then continue setting up and return self, otherwise
      # return undef.
      
      if( $#list >= 0 ) {
	 $self->{'host'} = $host;
	 $self->{'acks'} = \@list;
	 bless $self;
	 return $self;
      } else { 
	 return undef;
      }
   } else {
      $self->{'host'} = $host;
      $self->{'acks'} = \@list;

      bless $self;
      return $self;
   }
}

# Get/Set methods, acks() is a little fancy in that it return a list rather
# then just the list reference (easier for others to deal with).

sub host { my $var = 'host';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub acks { my $var = 'acks';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return @{$_[0]->{$var}}; }


# Used to add acknowledgements to the list.  Typically used in the case where
# you want to show acknowledgements on a variety of hosts and you start with
# a blank list.

sub add_acks {
   my( $self, $add ) = @_;

   if( ref( $add ) eq "Spong::AckList" ) {
      foreach( $add->acks() ) { 
	 push( @{$self->{'acks'}}, $_ );
      }
   } elsif( ref( $add ) eq "Spong::Ack" ) {
      push( @{$self->{'acks'}}, $add );
   }
}


# Pretty simple display, just iterate over the Acks and tell each of them to
# display themselves in the way that we were told.

sub display {
   my( $self, $format, $view ) = @_;

   if( $view eq "standard" && $format eq "text" ) {
      print "Host          Service  Until       Message\n";
      print "------------  -------  ----------  ";
      print "------------------------------------------\n";
   }

   sub bytime { return $b->end() <=> $a->end(); }

   foreach( sort bytime $self->acks() ) { 
      $_->display( $format, $view ); 
      
      if( $format eq "text" ) { print "\n"; }
      if( $format eq "html" ) { print "<p>"; }
   }
}


1;
