/*
 * (C) Copyright 2002, Schlund+Partner AG
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// Local configuration
#include "config.h"

// Implementation
#include "XMLDump.hpp"

// STDC++
#include <cassert>

namespace SP {
namespace GXML {

// Copy of XPathContext from XMLTree.cpp
class XPathContextDP
{
public:
	XPathContextDP(xmlDocPtr const doc)
		:_context(xmlXPathNewContext(doc))
	{
		if (!_context)
		{
			throw XMLException(XMLException::XMLTREE_CREATE_CONTEXT);
		}
	}

	~XPathContextDP()
	{
		xmlXPathFreeContext(_context);
	}

	xmlXPathContextPtr get() const
	{
		return _context;
	}

private:
	xmlXPathContextPtr const _context;
};

//
// XSLTDump
//
XSLTOutputBuf::XSLTOutputBuf(xmlDocPtr result, xsltStylesheetPtr style)
	:encoder_(0)
{
	const xmlChar *encoding;

	if ((result == 0) || (result->children == 0)) {
		throw RESULT_INVALID;
	}
	if (style == NULL) {
		throw STYLE_INVALID;
	}

	XSLT_GET_IMPORT_PTR(encoding, style, encoding) ;
	if (encoding != NULL) {
		encoder_ = xmlFindCharEncodingHandler((char *)encoding);
		if ((encoder_ != NULL) &&
				(xmlStrEqual((const xmlChar *)encoder_->name, (const xmlChar *) "UTF-8"))) {
			encoder_ = 0;
		}
		buf_ = xmlAllocOutputBuffer(encoder_);
	} else {
		buf_ = xmlAllocOutputBuffer(0);
	}
	if (buf_ == 0) {
		throw MEM_ERR;
	}
	xsltSaveResultTo(buf_, result, style);
}

XSLTOutputBuf::~XSLTOutputBuf()
{
	xmlOutputBufferClose(buf_);
}

char *
XSLTOutputBuf::getAddr() const
{
	if (encoder_) {
		return (char *) buf_->conv->content;
	} else {
		return (char *) buf_->buffer->content;
	}
}

int
XSLTOutputBuf::getSize() const
{
	if (encoder_) {
		return buf_->conv->use;
	} else {
		return buf_->buffer->use;
	}
}

//
// XMLDump
//
char * XMLDump::nullString_ = "";

// genDump: Master Create Dump Function. Must be used in Constructors only
// Note: xmlNodeDump segfaults with path="/"; this is the only reason why differentiate docDump, nodeDump... :(
void
XMLDump::genDump(const xmlDocPtr doc, const xmlXPathContextPtr ctxt, const std::string & path)
{
	xsltDump_ = 0;
	assert(doc);

	if (path == "/") {
		nodeDump_ = 0;
		xmlDocDumpMemoryEnc(doc, &docDump_.first, &docDump_.second, "UTF-8");
		if (!docDump_.first) {
			throw MEM_ERR;
		}
	} else {
		assert(ctxt);
		nodeDump_ = xmlBufferCreate();
		if (!nodeDump_) {
			throw MEM_ERR;
		}
		xmlNodePtr node = XMLTree::nodeFromPath(ctxt, path);
		if (!node) {
			xmlBufferFree(nodeDump_);
			throw NODE_NOT_FOUND_ERR;
		}

		xmlNodeDump(nodeDump_, doc, node, 0, 0);
		if (!nodeDump_->content) {
			xmlBufferFree(nodeDump_);
			throw MEM_ERR;
		}
	}
};

XMLDump::XMLDump(const xmlDocPtr doc, const xmlXPathContextPtr ctxt, const std::string & path)
{
	genDump(doc, ctxt, path);
}

XMLDump::XMLDump(const XMLTree * tree, const std::string & path)
{
	XPathContextDP context(tree->getDocPtr());

	genDump(tree->getDocPtr(), context.get(), path);
}

XMLDump::XMLDump(const xmlDocPtr doc)
{
	genDump(doc, 0, "/");
}


XMLDump::XMLDump(const XSLTOutputBuf * buf)
	:nodeDump_(0), docDump_(std::make_pair((xmlChar *)0,0)),
	 xsltDump_((XSLTOutputBuf *) buf)
{}

XMLDump::~XMLDump()
{
	if (nodeDump_) xmlBufferFree(nodeDump_);
	if (docDump_.first) xmlFree(docDump_.first);
	if (xsltDump_) delete xsltDump_;
}

char *
XMLDump::getAddr() const
{
	if (xsltDump_) {
		return xsltDump_->getAddr();
	} else if (nodeDump_) {
		return (char *) nodeDump_->content;
	} else if (docDump_.first) {
		return (char *) docDump_.first;
	}
	return nullString_;
}

std::string
XMLDump::get() const
{
	return getAddr();
}

int
XMLDump::getSize() const
{
	if (xsltDump_) {
		return xsltDump_->getSize();
	} else if (nodeDump_) {
		return xmlBufferLength(nodeDump_);
	} else if (docDump_.first) {
		return docDump_.second;
	}
	return 0;
}

}}
