# AlarmClock.pm V0.8 by Kevin Deane-Freeman (kevindf@shaw.ca) March 2003
# Adapted from code by Lukas Hinsch
# Updated by Dean Blackketter
#
# This code is derived from code with the following copyright message:
#
# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.

package SliMP3::Buttons::AlarmClock;

use SliMP3::Playlist;
use SliMP3::Strings qw (string);

my $interval = 1; # check every x seconds
my @browseMenuChoices;
my %menuSelection;
my %searchCursor;


# the routines
sub setMode() {
	my $client = shift;
	@browseMenuChoices = (
		string('ALARM_SET'),
		string('ALARM_SELECT_PLAYLIST'),
		string('ALARM_SET_VOLUME'),
		string('ALARM_OFF'),
		);
	if (!defined($menuSelection{$client})) { $menuSelection{$client} = 0; };
	$client->lines(\&lines);
	#get previous alarm time or set a default
	my $time = SliMP3::Prefs::clientGet($client, "alarmtime");
	if (!defined($time)) { SliMP3::Prefs::clientSet($client, "alarmtime", 9 * 60 * 60 ); }
   }

my %functions = (
	'up' => sub  {
		my $client = shift;
		my $newposition = SliMP3::Buttons::Common::scroll($client, -1, ($#browseMenuChoices + 1), $menuSelection{$client});

		$menuSelection{$client} =$newposition;
		SliMP3::Display::update($client);
	},
	'down' => sub  {
	   my $client = shift;
		my $newposition = SliMP3::Buttons::Common::scroll($client, +1, ($#browseMenuChoices + 1), $menuSelection{$client});

		$menuSelection{$client} =$newposition;
		SliMP3::Display::update($client);
	},
	'left' => sub  {
		my $client = shift;

		SliMP3::Buttons::Common::popModeRight($client);
	},
	'right' => sub  {
		my $client = shift;
		my @oldlines = SliMP3::Display::curLines($client);

		if ($browseMenuChoices[$menuSelection{$client}] eq string('ALARM_SET')) {
			SliMP3::Buttons::Common::pushModeLeft($client, 'alarmset');
		}
		if ($browseMenuChoices[$menuSelection{$client}] eq string('ALARM_SELECT_PLAYLIST')) {
			SliMP3::Buttons::Common::pushModeLeft($client, 'alarmplaylist');
		}
		elsif ($browseMenuChoices[$menuSelection{$client}] eq string('ALARM_OFF')) {
			SliMP3::Prefs::clientSet($client, "alarm", 1);
			$browseMenuChoices[$menuSelection{$client}] = string('ALARM_ON');
			SliMP3::Animation::showBriefly($client,string('ALARM_TURNING_ON'),'');
			setTimer($client);
		}
		elsif ($browseMenuChoices[$menuSelection{$client}] eq string('ALARM_ON')) {
			SliMP3::Prefs::clientSet($client, "alarm", 0);
			$browseMenuChoices[$menuSelection{$client}] = string('ALARM_OFF');
			SliMP3::Animation::showBriefly($client,string('ALARM_TURNING_OFF'),'');
			setTimer($client);
		}
		elsif ($browseMenuChoices[$menuSelection{$client}] eq string('ALARM_SET_VOLUME')) {
			SliMP3::Buttons::Common::pushModeLeft($client, 'alarmvolume');
		}
	},
	'play' => sub {
		my $client = shift;
	},
);

sub setTimer {
#timer to check alarms on an interval
	SliMP3::Timers::setTimer(0, Time::HiRes::time() + $interval, \&checkAlarms);
}

sub checkAlarms
{
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);

	my $time = $hour * 60 * 60 + $min * 60;

	if ($sec == 0) { # once we've reached the beginning of a minute, only check every 60s
		$interval = 60;
	}
	if ($sec >= 50) { # if we end up falling behind, go back to checking each second
		$interval = 1;
	}

	foreach my $client (SliMP3::Client::clients()) {
		if (SliMP3::Prefs::clientGet($client, "alarm")) {
			my $alarmtime =  SliMP3::Prefs::clientGet($client, "alarmtime");
			if ($alarmtime) {
			   if ($time == $alarmtime +60 ) {$interval=1;}; #alarm is done, so reset to find the beginning of a minute
				if ($time == $alarmtime) {
					SliMP3::Command::execute($client, ['stop']);
					my $volume = SliMP3::Prefs::clientGet($client, "alarmvolume");
					if (defined ($volume)) {
						SliMP3::Command::execute($client, ["mixer", "volume", $volume]);
					}
					if (defined SliMP3::Prefs::clientGet($client, "alarmplaylist")) {
						SliMP3::Command::execute($client, ["power", 1]);
						SliMP3::Buttons::Block::block($client,alarmLines($client));
						SliMP3::Command::execute($client, ["playlist", "load", SliMP3::Prefs::clientGet($client, "alarmplaylist")], \&playDone, [$client]);
					} else {
						SliMP3::Command::execute($client, ['play']);
						SliMP3::Timers::setTimer($client, Time::HiRes::time() + 2, \&visibleAlarm, $client);	
					}
				}
			}
		}
	}
	setTimer();
}

sub playDone {
	my $client = shift;
	SliMP3::Buttons::Block::unblock($client);
	# show the alarm screen after a couple of seconds when the song has started playing and the display is updated
	SliMP3::Timers::setTimer($client, Time::HiRes::time() + 2, \&visibleAlarm, $client);	
}

sub alarmLines {
	my $client = shift;
	my $line1 = string('ALARM_NOW_PLAYING');
	my $line2 = SliMP3::Prefs::clientGet($client, "alarmplaylist") ? SliMP3::Info::standardTitle($client,SliMP3::Prefs::clientGet($client, "alarmplaylist")) : "";
	return ($line1, $line2);
}

sub visibleAlarm {
	my $client = shift;
	my ($line1, $line2) =  alarmLines($client);
#show visible alert for 30s
	SliMP3::Animation::showBriefly($client,$line1, $line2,30);
}

sub lines {
	my $client = shift;
	my ($line1, $line2, $overlay);
	my $timeFormat = SliMP3::Prefs::get("timeFormat");

	$overlay = overlay($client);
	$line1 = string('ALARM');

	if (SliMP3::Prefs::clientGet($client, "alarm") && $browseMenuChoices[$menuSelection{$client}] eq string('ALARM_OFF')) {
		$browseMenuChoices[$menuSelection{$client}] = string('ALARM_ON');
	}
	$line2 = "";

	$line2 = $browseMenuChoices[$menuSelection{$client}];
	return ($line1, $line2, undef, $overlay);
}

sub overlay {
	my $client = shift;

	return SliMP3::VFD::symbol('rightarrow');
	
	return undef;
}

sub getFunctions() {
	return \%functions;
}

#################################################################################
# Alarm Volume Mode
my %alarmVolumeSettingsFunctions = (
	'left' => sub { SliMP3::Buttons::Common::popModeRight(shift); },
	'up' => sub {
		my $client = shift;
		my $volume = SliMP3::Prefs::clientGet($client, "alarmvolume");
		my $inc = 1;
		my $rate = 50; #Hz maximum
		my $accel = 15; #Hz/s

		if (SliMP3::IR::holdTime($client) > 0) {
			$inc *= SliMP3::IR::repeatCount($client,$rate,$accel);
		} else {
			$inc = 2.5;
		}

		if (!defined($volume)) { $volume = SliMP3::Prefs::clientGet($client, "volume"); }
		$volume += $inc;
		if ($volume > $SliMP3::Control::maxVolume) { $volume = $SliMP3::Control::maxVolume; };
		SliMP3::Prefs::clientSet($client, "alarmvolume", $volume);
		SliMP3::Display::update($client);
	},

	'down' => sub {
		my $client = shift;
		my $volume = SliMP3::Prefs::clientGet($client, "alarmvolume");
		my $inc = 1;
		my $rate = 50; #Hz maximum
		my $accel = 15; #Hz/s

		if (SliMP3::IR::holdTime($client) > 0) {
			$inc *= SliMP3::IR::repeatCount($client,$rate,$accel);
		} else {
			$inc = 2.5;
		}

		if (!defined($volume)) { $volume = SliMP3::Prefs::clientGet($client, "volume"); }
		$volume -= $inc;
		if ($volume < 0) { $volume = 0; };
		SliMP3::Prefs::clientSet($client, "alarmvolume", $volume);
		SliMP3::Display::update($client);
	},

	'right' => sub { SliMP3::Animation::bumpRight(shift); },
	'add' => sub { SliMP3::Animation::bumpRight(shift); },
	'play' => sub { SliMP3::Animation::bumpRight(shift); },

);

sub getAlarmVolumeFunctions {
	return \%alarmVolumeSettingsFunctions;
}

sub setAlarmVolumeMode {
	my $client = shift;
	$client->lines(\&alarmVolumeLines);
}

sub alarmVolumeLines {
	my $client = shift;
	my $volume = SliMP3::Prefs::clientGet($client, "alarmvolume");

	if (!defined($volume)) { $volume = SliMP3::Prefs::clientGet($client, "volume"); }

	my $level = int($volume / $SliMP3::Control::maxVolume * 40);

	my $line1 = SliMP3::Prefs::clientGet($client,'doublesize') ? string('ALARM_SET_VOLUME_SHORT') : string('ALARM_SET_VOLUME');
	my $line2;

	if ($level < 0) {
		$line1 .= " (". string('MUTED') . ")";
		$level = 0;
	} else {
		$line1 .= " (".$level.")";
	}

	$line2 = SliMP3::Display::progressBar($client, 40, $level / 40);

	if (SliMP3::Prefs::clientGet($client,'doublesize')) { $line2 = $line1; }
	return ($line1, $line2);
}


#################################################################################

my %alarmSetFunctions = (
	'up' => sub {
		my $client = shift;
		scrollDigit($client, +1);
	},
	'down' => sub {
		my $client = shift;
		scrollDigit($client, -1);
	},

	'left' => sub {
		my $client = shift;
		$searchCursor{$client}--;
		if ($searchCursor{$client} < 0) {
			SliMP3::Buttons::Common::popModeRight($client);
		} else {
			SliMP3::Display::update($client);
		}
	 },
	'right' => sub {
		my $client = shift;

		my ($h0, $h1, $m0, $m1, $p) = timeDigits($client);

		$searchCursor{$client}++;

		my $max = defined($p) ? 4 : 3;
		if ($searchCursor{$client} > $max) {
			$searchCursor{$client} = $max;
			#SliMP3::Buttons::Common::popModeRight($client);
		}
		SliMP3::Display::update($client);
	},

	'add' => sub { SliMP3::Animation::bumpRight(shift); },
	'play' => sub { SliMP3::Animation::bumpRight(shift); },
	'numberScroll' => sub  {
		my $client = shift;
		my $button = shift;
		my $digit = shift;

		my ($h0, $h1, $m0, $m1, $p) = timeDigits($client);

		my $h = $h0 * 10 + $h1;
		if ($p && $h == 12) { $h = 0 };

		my $c = $searchCursor{$client};
		if ($c == 0 && $digit < ($p ? 2:3)) { $h0 = $digit; $searchCursor{$client}++; };
		if ($c == 1 && (($h0 * 10 + $digit) < 24)) { $h1 = $digit; $searchCursor{$client}++; };
		if ($c == 2) { $m0 = $digit; $searchCursor{$client}++; };
		if ($c == 3) { $m1 = $digit };

		$p = (defined $p && $p eq 'PM') ? 1 : 0;
		if ($c == 4) { $p = $digit % 2; }

		my $time = ($h0 * 10 + $h1) * 60 * 60 + $m0 * 10 * 60 + $m1 * 60 + $p * 12 * 60 * 60;
		SliMP3::Prefs::clientSet($client, "alarmtime", $time);
		SliMP3::Display::update($client);

		#update the display
		SliMP3::Display::update($client);
	}
);

sub getAlarmSetFunctions {
	return \%alarmSetFunctions;
}

sub setAlarmSetMode {
	my $client = shift;
	$searchCursor{$client} = 0;
	$client->lines(\&alarmSetSettingsLines);
}

 sub alarmSetSettingsLines {
	my $client = shift;

	my ($h0, $h1, $m0, $m1, $p) = timeDigits($client);

	my $cs = SliMP3::VFD::symbol('cursorpos');
	my $c = $searchCursor{$client};

	my $timestring = ($c == 0 ? $cs : '') . ((defined($p) && $h0 == 0) ? ' ' : $h0) . ($c == 1 ? $cs : '') . $h1 . ":" . ($c == 2 ? $cs : '') .  $m0 . ($c == 3 ? $cs : '') . $m1 . " " . ($c == 4 ? $cs : '') . (defined($p) ? $p : '');

	return (string('ALARM_SET'), $timestring);
}

sub scrollDigit {
		my $client = shift;
		my $dir = shift;
		my ($h0, $h1, $m0, $m1, $p) = timeDigits($client);
		my $h = $h0 * 10 + $h1;
		
		if ($p && $h == 12) { $h = 0 };
		
		if ($searchCursor{$client} == 0) {$searchCursor{$client}++;};
		my $c = $searchCursor{$client};
		
		$p = ($p && $p eq 'PM') ? 1 : 0;
		
		if ($c == 1) {
		   $h = SliMP3::Buttons::Common::scroll($client, $dir, ($p == 1) ? 12 : 24, $h);
		   #change AM and PM if we scroll past midnight or noon boundary
		   if (SliMP3::Prefs::get('timeFormat') =~ /%p/) {
		   	if (($h == 0 && $dir == 1)||($h == 11 && $dir == -1)) { $p = SliMP3::Buttons::Common::scroll($client, +1, 2, $p); };
			};
		};
		if ($c == 2) { $m0 = SliMP3::Buttons::Common::scroll($client, $dir, 6, $m0) };
		if ($c == 3) { $m1 = SliMP3::Buttons::Common::scroll($client, $dir, 10, $m1)};
		if ($c == 4) { $p = SliMP3::Buttons::Common::scroll($client, +1, 2, $p); }

		my $time = $h * 60 * 60 + $m0 * 10 * 60 + $m1 * 60 + $p * 12 * 60 * 60;

		SliMP3::Prefs::clientSet($client, "alarmtime", $time);
		SliMP3::Display::update($client);
}

sub timeDigits {
	my $client = shift;
	my $time = SliMP3::Prefs::clientGet($client, "alarmtime");

	my $h = int($time / (60*60));
	my $m = int(($time - $h * 60 * 60) / 60);
	my $p = undef;

	my $timestring;

	if (SliMP3::Prefs::get('timeFormat') =~ /%p/) {
		$p = 'AM';
		if ($h > 11) { $h -= 12; $p = 'PM'; }
		if ($h == 0) { $h = 12; }
	} #else { $p = " "; };

	if ($h < 10) { $h = '0' . $h; }

	if ($m < 10) { $m = '0' . $m; }

	my $h0 = substr($h, 0, 1);
	my $h1 = substr($h, 1, 1);
	my $m0 = substr($m, 0, 1);
	my $m1 = substr($m, 1, 1);

	return ($h0, $h1, $m0, $m1, $p);
}


#################################################################################
# Alarm Playlist Mode
my %alarmPlaylistSettingsFunctions = (
	'left' => sub {
		my $client = shift;

		@{$client->dirItems}=(); #Clear list and get outta here
		SliMP3::Buttons::Common::popModeRight($client);
	},

	'up' => sub {
		my $client = shift;
		my $newposition = SliMP3::Buttons::Common::scroll($client, -1, $client->numberOfDirItems(), $client->currentDirItem());

		$client->currentDirItem($newposition);
		SliMP3::Prefs::clientSet($client, "alarmplaylist", $client->dirItems($client->currentDirItem));

		SliMP3::Display::update($client);
	},

	'down' => sub {
		my $client = shift;
		my $newposition = SliMP3::Buttons::Common::scroll($client, +1, $client->numberOfDirItems(), $client->currentDirItem());

		$client->currentDirItem($newposition);
		SliMP3::Prefs::clientSet($client, "alarmplaylist", $client->dirItems($client->currentDirItem) );

		SliMP3::Display::update($client);
	},

	'right' => sub { SliMP3::Animation::bumpRight(shift); },
	'add' => sub { SliMP3::Animation::bumpRight(shift); },
	'play' => sub { SliMP3::Animation::bumpRight(shift); },

	'numberScroll' => sub  {
		my $client = shift;
		my $button = shift;
		my $digit = shift;
		my $i = SliMP3::Buttons::Common::numberScroll($client, $digit, $client->dirItems);

		$client->currentDirItem($i);
		SliMP3::Display::update($client);
	},

);

sub getAlarmPlaylistFunctions {
	return \%alarmPlaylistSettingsFunctions;
}

sub setAlarmPlaylistMode {
	my $client = shift;
	$client->lines(\&alarmPlaylistLines);

	@{$client->dirItems}=();	
	
	SliMP3::Scan::addToList($client->dirItems, SliMP3::Prefs::get('playlistdir'), 0);
	if (SliMP3::iTunes::useiTunesLibrary()) {
		push @{$client->dirItems}, @{SliMP3::iTunes::playlists()};
	} 

	$client->numberOfDirItems(scalar @{$client->dirItems});
	$client->currentDirItem(0);
	my $list = SliMP3::Prefs::clientGet($client, "alarmplaylist");
	if ($list) {
		my $i = 0;
		my $items = $client->dirItems;
		foreach my $cur (@$items) {
			if ($list eq $cur) {
				$client->currentDirItem($i);
				last;
			}
			$i++;
		}
	}
	SliMP3::Prefs::clientSet($client, "alarmplaylist", $client->dirItems($client->currentDirItem) );
}

sub alarmPlaylistLines {
	my $client = shift;
	my $line1;
	my $line2;

	$line1 = string('ALARM_PLAYLIST_ENTRY');

	if (defined $client->dirItems($client->currentDirItem)) {
		$line2 = SliMP3::Info::standardTitle($client,$client->dirItems($client->currentDirItem));
	} else {
		$line2 = string('EMPTY');
	}

	if ($client->numberOfDirItems()) {
		$line1 .= sprintf(" (%d ".string('OUT_OF')." %s)", $client->currentDirItem + 1, $client->numberOfDirItems());
	}
	
	return ($line1, $line2);
}

# some initialization code, adding modes for this module
SliMP3::Buttons::Common::addMode('alarm', getFunctions(), \&SliMP3::Buttons::AlarmClock::setMode);
SliMP3::Buttons::Common::addMode('alarmvolume', getAlarmVolumeFunctions(), \&SliMP3::Buttons::AlarmClock::setAlarmVolumeMode);
SliMP3::Buttons::Common::addMode('alarmset', getAlarmSetFunctions(), \&SliMP3::Buttons::AlarmClock::setAlarmSetMode);
SliMP3::Buttons::Common::addMode('alarmplaylist', getAlarmPlaylistFunctions(), \&SliMP3::Buttons::AlarmClock::setAlarmPlaylistMode);
setTimer();


1;

__END__

