
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer
//          Malolan Chetlur    
//          Krishnan Subramani 
//          Radharamanan Radhakrishnan
//          Umesh Kumar V. Rajasekaran
//          Narayanan Thondugulam
//          Magnus Danielson    cfmd@swipnet.se

#include "IIR_SignalDeclaration.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_EnumerationLiteral.hh"
#include "IIR_RangeTypeDefinition.hh"
#include "IIR_LeftAttribute.hh"
#include "IIR_IntegerSubtypeDefinition.hh"
#include "IIR_EnumerationSubtypeDefinition.hh"
#include "IIR_FunctionDeclaration.hh"
#include "symbol_table.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "StandardPackage.hh"
#include "savant.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

extern symbol_table *cgen_sym_tab_ptr;

IIRScram_ScalarTypeDefinition::~IIRScram_ScalarTypeDefinition() {}

IIR*
IIRScram_ScalarTypeDefinition::_get_direction() {
  return get_direction();
}

IIR *
IIRScram_ScalarTypeDefinition::_get_base_type_left() {
  IIR *retval = 0;

  if( get_left() != NULL ){
    retval = get_left();
  } 
  else if( _get_base_type() != NULL ){
    retval = _get_base_type()->_get_base_type_left();
  }
  return retval;
}

IIR *
IIRScram_ScalarTypeDefinition::_get_base_type_right() {
  IIR *retval = 0;

  if( get_right() != NULL ){
    retval = get_right();
  } 
  else if( _get_base_type() != NULL ){
    retval = _get_base_type()->_get_base_type_right();
  }
  return retval;
}

IIR *
IIRScram_ScalarTypeDefinition::_get_base_type_direction() {
  IIR *retval = 0;

  if( get_direction() != NULL ){
    retval = get_direction();
  } 
  else if( _get_base_type() != NULL ){
    retval = _get_base_type()->_get_base_type_direction();
  }
  return retval;
}

IIR_TypeDefinition *
IIRScram_ScalarTypeDefinition::_get_base_type() {
  IIR_TypeDefinition *retval = get_base_type();

  if( _is_subtype_decl() == TRUE || 
      ( _is_anonymous() == TRUE && _get_type_mark() != NULL )){
    retval = _get_type_mark();
  }
  return retval;
}

void 
IIRScram_ScalarTypeDefinition::_publish_vhdl_decl(ostream &_vhdl_out){
  ASSERT( get_left() != NULL && get_left()->_is_resolved() );
  ASSERT( get_direction() != NULL && get_direction()->_is_resolved() );
  ASSERT( get_right() != NULL && get_right()->_is_resolved() );

  if( _get_resolution_function() != NULL ){
    _get_resolution_function()->get_declarator()->_publish_vhdl(_vhdl_out);
    _vhdl_out << " ";
  }
  if( _get_type_mark() != NULL ){
    _get_type_mark()->_get_declaration()->get_declarator()->_publish_vhdl(_vhdl_out);
    
    if( _get_type_mark()->_is_iir_scalar_type_definition() == TRUE ){
      IIR_ScalarTypeDefinition *as_scalar = (IIR_ScalarTypeDefinition *)_get_type_mark();
      if( get_left() == as_scalar->get_left() &&
	  get_right() == as_scalar->get_right() &&
	  get_direction() == as_scalar->get_direction() ){
	// If none of the fields is different  don't bother with ranges
	// just quit;
	return;
      }
    }
  }
  _vhdl_out << " range ";
  get_left()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " ";
  if( _is_ascending_range() == TRUE ){
    _vhdl_out << " to ";
  }
  else {
    _vhdl_out << " downto ";
  }
  _vhdl_out << " ";
  get_right()->_publish_vhdl(_vhdl_out);
}
    
void 
IIRScram_ScalarTypeDefinition::_publish_vhdl(ostream &_vhdl_out){
  if( _is_anonymous() == TRUE ){
    if( _get_type_mark() != NULL ){
      ASSERT( _get_type_mark()->_is_iir_scalar_type_definition() == TRUE );
      IIR_ScalarTypeDefinition *as_scalar = (IIR_ScalarTypeDefinition *)_get_type_mark();
      _get_type_mark()->_publish_vhdl( _vhdl_out );
      if( get_left() != as_scalar->get_left() ||
	  get_right() != as_scalar->get_right() ||
	  get_direction() != as_scalar->get_direction() ){
	// If one of the fields is different, print the range.  otherwise,
	// don't bother.
	_vhdl_out << " range ";
	_publish_vhdl_range( _vhdl_out );
      }
    }
    else{
      _publish_vhdl_range( _vhdl_out );
    }
  } 
  else {
    ASSERT( _get_declaration()->_is_resolved() == TRUE );
    _get_declaration()->_publish_vhdl(_vhdl_out);
  }
}

void
IIRScram_ScalarTypeDefinition::_publish_vhdl_index(ostream &_vhdl_out) {
  if( (_is_iir_type_definition() == TRUE) && (_is_anonymous() == FALSE) ){
    _get_declaration()->_get_declarator()->_publish_vhdl(_vhdl_out);
  }
  else {
    if( _get_type_mark() != NULL ){
      _get_type_mark()->_publish_vhdl(_vhdl_out);
      _vhdl_out << " range "; 
    }
    _publish_vhdl_range(_vhdl_out);
  }
}

void 
IIRScram_ScalarTypeDefinition::_publish_vhdl_range(ostream &_vhdl_out) {

  if( get_left() == NULL ){
    ASSERT( get_right() == NULL );
    ASSERT( get_direction() == NULL );
    _vhdl_out << "<>";
  } 
  else {
    ASSERT( get_right() != NULL );
    ASSERT( get_direction() != NULL );
    get_left()->_publish_vhdl(_vhdl_out);
    _vhdl_out << " ";
    if( _is_ascending_range() == TRUE ){
      _vhdl_out << "to";
    } else {
      _vhdl_out << "downto";
    }
    _vhdl_out << " ";
    get_right()->_publish_vhdl(_vhdl_out);
  }
}

void 
IIRScram_ScalarTypeDefinition::_publish_vhdl_constraint(ostream &_vhdl_out) {
  _report_undefined_scram_fn("_publish_vhdl_constraint(ostream &)");
}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_left( published_file &_cc_out ) {
  ASSERT(get_left() != NULL);
  get_left()->_publish_cc_lvalue( _cc_out );
}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_right( published_file &_cc_out ) {
  get_right()->_publish_cc_lvalue( _cc_out );
}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_universal_left( published_file &_cc_out ) {
  get_left()->_publish_cc_universal_value( _cc_out );
}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_universal_right( published_file &_cc_out ) {
  get_right()->_publish_cc_universal_value( _cc_out );
}

IIR_Boolean
IIRScram_ScalarTypeDefinition::_is_bit_type() {
  IIR_Boolean retval;

  if( this == StandardPackage::get_bit_type() ){
    retval = TRUE;
  }
  else{
    retval = FALSE;
  }
  return retval;
}

IIR_Boolean
IIRScram_ScalarTypeDefinition::_is_boolean_type() {
  IIR_Boolean retval;

  if( this == StandardPackage::get_boolean_type() ){
    retval = TRUE;
  }
  else{
    retval = FALSE;
  }
  return retval;
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_range( published_file &_cc_out ) {
  IIR_ScalarTypeDefinition *to_publish;
  if( get_left() != NULL ){
    to_publish = (IIR_ScalarTypeDefinition *)this;
  }
  else{
    ASSERT( get_base_type() != NULL );
    ASSERT( get_base_type()->_is_iir_scalar_type_definition() == TRUE );
    to_publish = (IIR_ScalarTypeDefinition *)get_base_type();
  }
  ASSERT( to_publish->get_left() != NULL );
  ASSERT( to_publish->get_left()->_is_resolved() == TRUE );
  ASSERT( to_publish->get_direction() != NULL );
  ASSERT( to_publish->get_direction()->get_kind() == IIR_ENUMERATION_LITERAL );
  ASSERT( to_publish->get_right() != NULL );
  ASSERT( to_publish->get_right()->_is_resolved() == TRUE );
  
  to_publish->get_left()->_publish_cc_value( _cc_out );
  _cc_out << "," << NL();
  ((IIR_EnumerationLiteral *)to_publish->get_direction())->_publish_cc_direction( _cc_out );
  _cc_out << "," << NL();
  to_publish->get_right()->_publish_cc_value( _cc_out );
}

IIR_Boolean
IIRScram_ScalarTypeDefinition::_is_ascending_range() {
  if( get_direction() == NULL ){
    ostringstream err;
    err << "Get direction called on type that has no direction.";
    report_error( this, err.str(), WARNING );

    return FALSE;
  }
  else{
    return get_direction()->_is_ascending_range();
  }
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_init_signal( published_file &_cc_out ) {
  if(((IIR_Declaration*)_get_current_publish_node())->_is_implicit_declaration() == true) {
    _publish_cc_init_implicit_signal( _cc_out ); 
  }
  else {
   _publish_cc_init_explicit_signal( _cc_out ); 
  }
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_composite_init( published_file &_cc_out ) {
  _cc_out << OS("{") << "Variable<";
  _publish_cc_universal_type( _cc_out );
  _cc_out << "> *variable = (Variable<"; 
  _publish_cc_universal_type( _cc_out );
  _cc_out << "> *)(*(ScalarType*)&" << _get_current_publish_name() << ")"
	  << ".getObject();" << NL();
  _cc_out << "variable->val = ";
  _publish_cc_universal_left( _cc_out );
  _cc_out << ";" << CS("}");
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_init_implicit_signal( published_file &_cc_out ) {
  _cc_out << "ImplicitSignal<";
  _publish_cc_universal_type( _cc_out );
  _cc_out << ">* signal = (ImplicitSignal<"; 
  _publish_cc_universal_type( _cc_out );
  ASSERT( _get_current_publish_name() != "" );
  _cc_out << ">*) (*(ScalarType*)&this->" << _get_current_publish_name() << ")"
	  << ".getObject();" << NL();

  // I expect _current_another_name to be set to the string representation
  // of the signal's netinfo structure when I get here.  It is set in
  // IIRScram_SignalDeclaration_publish_cc_init_signal( published_file &_cc_out ).
  ASSERT( _get_current_another_name() != "" );

  _cc_out << "signal->type =";
  ASSERT( _get_current_publish_node()->_is_iir_declaration() == TRUE );
  ((IIR_Declaration*)_get_current_publish_node())->_get_attribute_name()->_publish_cc_sigtype( _cc_out );

  _cc_out << "signal->drvVal = ";
  ((IIR_Declaration*)_get_current_publish_node())->_get_attribute_name()->_publish_cc_init_val( _cc_out );
  _cc_out << NL();

  _cc_out << "signal->effVal = ";
  ((IIR_Declaration*)_get_current_publish_node())->_get_attribute_name()->_publish_cc_init_val( _cc_out );
  _cc_out << ";" << NL();
  _cc_out << "signal->name = \"" << _get_current_publish_name() << "\";" << NL();
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_init_explicit_signal( published_file &_cc_out ) {
  // NOTE : The values set in _current_publish_name and _current_another_name
  //        are important. Care must be taken to set these names correctly.
  //        Incase of publishing for RecordTypes, the function expects
  //        _current_publish_name to point to the record name and
  //        _current_another_name to point to the field name.
  //
  //        When publishing scalar types, the variable _current_publish_name
  //        should be set to the scalar variable name while
  //        _current_another_name must be set to NULL

  _cc_out << "Signal<";
  _publish_cc_universal_type( _cc_out );
  _cc_out << ">* signal = (Signal<"; 
  _publish_cc_universal_type( _cc_out );
  _cc_out << ">*) (*(ScalarType*)&this->" << _get_current_publish_name();
  _cc_out << ")"  << ".getObject();" << NL();

  //  _cc_out << "  signal->sensitive = true;\n";
  // XXX not all signals are sensitive; this will be slightly inefficient

  if( _get_current_publish_node()->_is_signal()) {
    if( _get_current_publish_node()->get_value() == NULL) {
      _cc_out << "signal->drvVal = ";
      _publish_cc_universal_left( _cc_out );
      _cc_out << ";" << NL();
      
      _cc_out << "signal->effVal = ";
      _publish_cc_universal_left( _cc_out );
      _cc_out << ";" << NL();
    }
  }
  _cc_out << "  signal->name = \"" << _get_current_publish_name() << "\";\n";
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_init_last_event( published_file &_cc_out ) {
  _cc_out << _get_current_publish_name() << ";" << NL();
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_class_last_event( published_file & ){}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_class_event( published_file & ){}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_array_info( published_file &_cc_out ){
  _cc_out << OS("ArrayInfo(");
  if( get_left() != NULL) {
    ASSERT( get_right() != NULL);
    ASSERT( get_direction() != NULL);
    _publish_cc_range( _cc_out );
  }
  else {
    _cc_out << "left1, dirn1, right1";
  }
  _cc_out << CS(")");
}

ostream &
IIRScram_ScalarTypeDefinition::_print( ostream &os ){
  if( get_left() == NULL && get_right() == NULL && get_direction() == NULL ){
    // This is an unconstrained type.
    if( get_base_type() != NULL ){
      os << *get_base_type()->_get_declarator() << " range <> ";
    }
    else{
      os << *_get_declarator() << " range <> ";
    }
  }
  else{
    if( get_left() != NULL ){
      os << *get_left();
    }
    if( _is_ascending_range() == TRUE ){
      os << " to ";
    }
    else{
      os << " downto ";
    }
    if( get_right() != NULL ){
      os << *get_right();
    }
  }
  return os;
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_init_val( published_file &_cc_out ) {
  _publish_cc_lvalue( _cc_out );
  _cc_out.open_scope( "(ObjectBase::VARIABLE," );
  _publish_cc_universal_left( _cc_out );
  _cc_out.close_scope( ")" );
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_signal_value_left( published_file &_cc_out ) {
  if( _get_current_publish_node()->_is_signal() ){
    if( _get_current_publish_node()->get_value() == NULL ){
      _publish_cc_universal_left( _cc_out );
    }
    else {
      _get_current_publish_node()->get_value()->_publish_cc_universal_value( _cc_out );
    }
  }
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_decl_type_attributes( published_file &_cc_out ) {
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_define_type_attributes( published_file &_cc_out ) {}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_range_check( published_file &_cc_out ) {
  if(_is_ascending_range()) {
    _cc_out.open_scope( "(val >= (" );
    _publish_cc_universal_type( _cc_out );
    _cc_out.open_scope( "(" );
    get_left()->_publish_cc_value( _cc_out );
    _cc_out.close_scope( ")" );
    _cc_out.close_scope( ")" );
    _cc_out.open_scope( " && val <= (" );
    _publish_cc_universal_type( _cc_out );
    _cc_out.open_scope( "(" );
    get_right()->_publish_cc_value( _cc_out );
    _cc_out.close_scope( ")" );
    _cc_out.close_scope( ")" );
  }
  else {
    _cc_out.open_scope( "(val <= (" );
    _publish_cc_universal_type( _cc_out );
    _cc_out.open_scope( "(" );
    get_left()->_publish_cc_value( _cc_out );
    _cc_out.close_scope( ")" );
    _cc_out.close_scope( ")" );
    _cc_out.open_scope( " && val >= (" );
    _publish_cc_universal_type( _cc_out );
    _cc_out.open_scope( "(" );
    get_right()->_publish_cc_value( _cc_out );
    _cc_out.close_scope( ")" );
    _cc_out.close_scope( ")" );
    _cc_out.close_scope( ")" );
  }
}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_decl_destructors( published_file &_cc_out ) {
  _cc_out << "virtual ~" << _get_cc_type_name() << "(){}" << NL();
}

const string
IIRScram_ScalarTypeDefinition::_get_cc_type_name(){
  return _get_cc_kernel_type();
}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_constructor_args( published_file &_cc_out ) {
  //Nothing to be done here
}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_decl_constructors( published_file &_cc_out ) {
  if(_is_subtype_decl()) {
    _publish_cc_subtype_constructors( _cc_out );
  } else {
    _publish_cc_type_constructors( _cc_out );
  }
}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_type_constructors( published_file &_cc_out ){}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_subtype_constructors( published_file &_cc_out ) {}

void 
IIRScram_ScalarTypeDefinition::_publish_cc_headers( published_file &_cc_out ) {
  if( !_is_kernel_type() ){
    SCRAM_CC_REF( _cc_out,
		  "IIRScram_ScalarTypeDefinition::_publish_cc_headers( published_file &_cc_out )");
    if( this != _get_parent_type() ){
      _get_parent_type()->_publish_cc_headers( _cc_out );
    }
    if(_is_subtype_decl() == TRUE) {
      ASSERT(_get_type_mark() != NULL);
      _get_type_mark()->_publish_cc_headers( _cc_out );
    }
    //For a type its ranges can be specified in terms of othher types.
    // and that type's attributes. Then in that case those types.hh have to
    //be included in the .cc file
    if( get_left() != 0 ){
      get_left()->_publish_cc_headers( _cc_out );
    }
    if( get_right() != 0 ){
      get_right()->_publish_cc_headers( _cc_out );
    }
    IIRScram::_publish_cc_include( _cc_out, "strstream", true );
  }
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_necessary_decl_in_state( published_file &_cc_out ) {
  // Dummy function.
}

void
IIRScram_ScalarTypeDefinition::_publish_cc_constructor_args_type_info( published_file &_cc_out,
								       IIR *initializer ){
  if( initializer != 0 ){
    _cc_out << "," << NL();
    if( !initializer->_is_literal() && 
	!initializer->_is_enumeration_literal() && 
	initializer->get_kind() != IIR_PHYSICAL_LITERAL ){
      _cc_out << "(const ";
      initializer->_publish_cc_kernel_type( _cc_out );
      _cc_out << " &) ";
    }
    initializer->_publish_cc_initialization_value( _cc_out );
  }
  else if ( _is_anonymous() ){
    _cc_out << "," << NL() << "(const ";
    _publish_cc_universal_type( _cc_out );
    _cc_out << " &)";
    get_subtype()->_publish_cc_universal_left( _cc_out );
  }
  _publish_cc_object_type_info( _cc_out );
}

IIR_ScalarTypeDefinition *
IIRScram_ScalarTypeDefinition::_determine_discrete_type( IIR_RangeTypeDefinition *range ){

  if( range->get_left() == NULL ){
    ostringstream err;
    err << "Internal error in IIRScram_ScalarTypeDefinition::_determine_discrete_type(";
    err << "IIR_RangeTypeDefinition *range ) - this problem is most likely due to a known bug ";
    err << "in the intermediate form.  Attempting work around it.";
    report_error( range, err.str(), WARNING );
    return range;
  }

  ASSERT( range->get_left() != NULL );
  ASSERT( range->get_right() != NULL );
  ASSERT( range->get_direction() != NULL );

  // This is the type that this is a subtype of.
  IIR_TypeDefinition *subtype_of = NULL;
  
  set<IIR_TypeDefinition> *left_set = range->get_left()->_get_rval_set();
  if( left_set == NULL ){
    report_undefined_symbol( range->get_left() );
    return NULL;
  }
  
  set<IIR_TypeDefinition> *right_set = range->get_right()->_get_rval_set();
  if( right_set == NULL ){
    report_undefined_symbol( range->get_right() );
    return NULL;
  }
  
  ASSERT( left_set != NULL );
  left_set->reduce_set( &IIR::_is_discrete_type );
  
  reconcile_sets( left_set, right_set );  
  switch( left_set->num_elements() ){

  case 0:{
    ostringstream err;
    err << "|" << *range << "| is not a proper range in this context.";
    report_error( range, err.str() );
    return NULL;
  }

  case 1:{
    IIR_TypeDefinition *left_type = left_set->get_element();
    IIR_TypeDefinition *right_type = right_set->get_element();

    ASSERT( left_type->_is_scalar_type() == TRUE );
    ASSERT( right_type->_is_scalar_type() == TRUE );


    if( left_type->_get_bottom_base_type() != right_type->_get_bottom_base_type() ){
      // That means that the two types aren't "naturally" related, so one of them must be
      // a universal type.
      ASSERT( left_type == StandardPackage::get_savant_universal_integer() ||
	      right_type == StandardPackage::get_savant_universal_integer() );

      if( left_type == StandardPackage::get_savant_universal_integer() ){
	subtype_of = (IIR_ScalarTypeDefinition *)right_type;
      }
      else{
	ASSERT( right_type == StandardPackage::get_savant_universal_integer() );
	subtype_of = (IIR_ScalarTypeDefinition *)left_type;
      }

    }
    else{
      // The types are naturally compatible, and "reconcile_sets" should
      // have handed us the correct type back in the left set.
      subtype_of = (IIR_ScalarTypeDefinition *)left_set->get_element();
    }

    ASSERT( subtype_of->_is_scalar_type() == TRUE );
    break;
  }
  default:{
    ostringstream err;
    err << "The type of a range used in this manner must be "
	<< "determinable without context.";
    report_error( range, err.str() );
    report_ambiguous_error( range, left_set );
    return NULL;
  }
  }

  delete left_set;
  delete right_set;

  // This method is following the rules of page 42, line 350 of the LRM.  
  if( subtype_of == StandardPackage::get_savant_universal_integer() &&
      ( range->get_left()->_is_integer_literal() == TRUE ||
	range->get_left()->_is_attribute() == TRUE ) &&
      ( range->get_right()->_is_integer_literal() == TRUE ||
	range->get_right()->_is_attribute() == TRUE ) ){
    subtype_of = StandardPackage::get_integer_type();
  }

  return (IIR_ScalarTypeDefinition *)subtype_of;
}

IIR_Boolean 
IIRScram_ScalarTypeDefinition::_is_resolved(){
  IIR_Boolean retval = TRUE;

  if( this != StandardPackage::get_savant_universal_integer() 
      && this != (IIR_ScalarTypeDefinition*)StandardPackage::get_savant_universal_real() ){
    if( get_left() == NULL || get_right() == NULL || get_direction() == NULL ){
      ASSERT( get_left() == NULL);
      ASSERT( get_right() == NULL);
      ASSERT( get_direction() == NULL);
    }
    else{
      ASSERT( get_left() != NULL);
      ASSERT( get_right() != NULL);
      ASSERT( get_direction() != NULL);
      
      if( get_left()->_is_resolved() == FALSE ||
	  get_right()->_is_resolved() == FALSE || 
	  get_direction()->_is_resolved() == FALSE ){
	retval = FALSE;
      }
    }
  }
  
  return retval;
}

IIR_TypeDefinition *
IIRScram_ScalarTypeDefinition::_construct_new_subtype( IIR_Name *resolution_function,
						       IIR_ScalarTypeDefinition *new_constraint ){
  
  IIR_FunctionDeclaration *resolution_function_decl = NULL;

  IIR_TypeDefinition *subtype = _get_new_subtype();
  ASSERT( subtype->_is_iir_scalar_type_definition() == TRUE );
  IIR_ScalarTypeDefinition *scalar_subtype = (IIR_ScalarTypeDefinition *)subtype;
  
  if( resolution_function != NULL ){
    resolution_function_decl = _resolve_resolution_function( resolution_function );
    scalar_subtype->set_resolution_function( resolution_function_decl );
  }
  
   // Enumeration types don't have anonymous base types unlike the
  // other scalar types - hence all of the special cases here.
  if( _is_subtype() == TRUE ){
    // If we are a subtype, and we're NOT an enumeration type:
    // 1) base type isn't null
    // 2) base type is a type - not a subtype, and it's anonymous.
    ASSERT( get_base_type() != NULL );
    ASSERT( get_base_type()->_is_subtype() == FALSE 
	    || get_base_type()->_is_enumeration_type() == TRUE );
    ASSERT( get_base_type()->_is_anonymous() == FALSE );

    scalar_subtype->set_base_type( get_base_type() );
  }
  else{
    // We are a type.  If we're not an enumeration type, we must:
    // 1) have a NULL base type.
    // 2) be anonymous
    ASSERT( _is_enumeration_type() == TRUE || get_base_type() == NULL && _is_anonymous() == TRUE );
    scalar_subtype->set_base_type( this );
  }
    
  if( new_constraint != NULL ){
    copy_location( new_constraint, subtype );

    if( new_constraint->get_left() != NULL ){
      ASSERT( new_constraint->get_right() != NULL );
      ASSERT( new_constraint->get_direction() != NULL );

      // Resolve the left side, if needed
      if( new_constraint->get_left()->_is_resolved() == FALSE ){
	new_constraint->set_left( new_constraint->get_left()->_semantic_transform( this ) );
	new_constraint->get_left()->_type_check( this );
	new_constraint->set_left( new_constraint->get_left()->_rval_to_decl( this ) );	
      }
      // Modified by SK -- Begin
      // set the subtype of left if it is an enumeration literal.
      if(new_constraint->get_left()->_is_enumeration_literal() == TRUE) {
	if(new_constraint->get_left()->get_subtype() == NULL) {
	  ASSERT(_is_enumeration_type() == TRUE);
	  ((IIR_EnumerationLiteral *) new_constraint->get_left())->set_subtype((IIR_EnumerationTypeDefinition*)this);
	}
      }
      // Modified by SK -- End      
      scalar_subtype->set_left( new_constraint->get_left() );

      // No resolution need for direction...
      scalar_subtype->set_direction( new_constraint->get_direction() );


      // Resolve the right side, if needed
      if( new_constraint->get_right()->_is_resolved() == FALSE ){
	new_constraint->set_right( new_constraint->get_right()->_semantic_transform( this ) );
	new_constraint->get_right()->_type_check( this );
	new_constraint->set_right( new_constraint->get_right()->_rval_to_decl( this ) );	
      }

      // Modified by SK -- Begin
      // set the subtype of right if it is an enumeration literal.
      if(new_constraint->get_right()->_is_enumeration_literal() == TRUE) {
	if(new_constraint->get_right()->get_subtype() == NULL) {
	  ASSERT(_is_enumeration_type() == TRUE);
	  ((IIR_EnumerationLiteral *) new_constraint->get_right())->set_subtype((IIR_EnumerationTypeDefinition*)this);
	}
      }
      // Modified by SK -- End
      scalar_subtype->set_right( new_constraint->get_right() );
    }
#ifdef DEVELOPER_ASSERTIONS
    else{
      ASSERT( new_constraint->get_direction() == NULL );
      ASSERT( new_constraint->get_right() == NULL );
    }
#endif
  }
  else{
    // No new constraint 
    copy_location( this, scalar_subtype );
    scalar_subtype->set_left( get_left() );
    scalar_subtype->set_direction( get_direction() );
    scalar_subtype->set_right( get_right() );

    if( scalar_subtype->_is_iir_enumeration_type_definition() == TRUE ){
      ASSERT( _is_iir_enumeration_type_definition() == TRUE );
      IIR_EnumerationTypeDefinition *this_as_enumeration;
      IIR_EnumerationTypeDefinition *scalar_subtype_as_enumeration;
      this_as_enumeration = (IIR_EnumerationTypeDefinition *)this;
      scalar_subtype_as_enumeration = (IIR_EnumerationTypeDefinition *)scalar_subtype;
      scalar_subtype_as_enumeration->enumeration_literals = 
	this_as_enumeration->enumeration_literals;
    }
  }

  return scalar_subtype;
}

void 
IIRScram_ScalarTypeDefinition::set_resolution_function( IIR_FunctionDeclaration * ){
  _report_undefined_scram_fn("set_resolution_function");  
}

void 
IIRScram_ScalarTypeDefinition::_add_decl_into_cgen_symbol_table() {
  if(get_base_type() != NULL) {
    get_base_type()->_add_decl_into_cgen_symbol_table();
  }
  if(get_left() != NULL) {
    get_left()->_add_decl_into_cgen_symbol_table();
  }
  if(get_right() != NULL) {
    get_right()->_add_decl_into_cgen_symbol_table();
  }
}

void 
IIRScram_ScalarTypeDefinition::_clone( IIR *copy_into ){
  ASSERT( copy_into->_is_iir_scalar_type_definition() == TRUE );
  
  IIR_ScalarTypeDefinition *as_scalar_type = (IIR_ScalarTypeDefinition *)copy_into;
  
  as_scalar_type->set_left( get_left() );
  as_scalar_type->set_right( get_right() );
  as_scalar_type->set_direction( get_direction() );

  IIR_TypeDefinition::_clone( copy_into );
}

IIR_Boolean 
IIRScram_ScalarTypeDefinition::_is_locally_static(){
  if( get_left() != NULL && get_left()->_is_locally_static() == TRUE &&
      get_right() != NULL && get_right()->_is_locally_static() == TRUE ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

IIR_Boolean 
IIRScram_ScalarTypeDefinition::_is_globally_static(){
  if( get_left() != NULL && get_left()->_is_globally_static() == TRUE &&
      get_right() != NULL && get_right()->_is_globally_static() == TRUE ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

void 
IIRScram_ScalarTypeDefinition::_build_implicit_operators( set<IIR_Declaration> *add_to ){
  IIR_TypeDefinition::_build_implicit_operators( add_to );
  _build_ordering_operators( add_to );
}

void 
IIRScram_ScalarTypeDefinition::_init_scalar_type( IIR_RangeTypeDefinition *init_info,
						  IIR_ScalarTypeDefinition *base_type,
						  IIR_ScalarTypeDefinition *subtype,
						  IIR_TypeDeclaration *type_decl ){
  IIR *left = init_info->get_left();
  left = left->_semantic_transform( subtype );
  left->_type_check( subtype );
  left = left->_rval_to_decl( subtype );

  IIR *right = init_info->get_right();
  right = right->_semantic_transform( subtype );
  right->_type_check( subtype );
  right = right->_rval_to_decl( subtype );

  subtype->set_base_type( base_type );
  subtype->set_left( left );
  subtype->set_right( right );
  subtype->set_direction( init_info->get_direction() );
  subtype->_set_declaration( type_decl );
  copy_location( init_info, subtype );

  base_type->set_left( left );
  base_type->set_right( right );
  base_type->set_direction( init_info->get_direction() );
  base_type->_set_declaration( type_decl );
  copy_location( init_info, base_type );
}
