
// Copyright (c) 1996-2001 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
// THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES
// SUFFERED BY LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING
// OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found
// in the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Malolan Chetlur
//          Umesh Kumar V. Rajasekaran
//          Timothy J. McBrayer
//          Narayanan Thondugulam

//---------------------------------------------------------------------------
#include "IIRScram_ConcurrentSelectedSignalAssignment.hh"
#include "IIR_CaseStatement.hh"
#include "IIR_CaseStatementAlternativeByChoices.hh"
#include "IIR_CaseStatementAlternativeByExpression.hh"
#include "IIR_CaseStatementAlternativeByOthers.hh"
#include "IIR_Identifier.hh"
#include "IIR_Label.hh"
#include "IIR_ProcessStatement.hh"
#include "IIR_SelectedWaveform.hh"
#include "IIR_SignalAssignmentStatement.hh"
#include "IIR_WaitStatement.hh"
#include "error_func.hh"
#include "IIR_IfStatement.hh"
#include "IIR_StringLiteral.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_WaveformElement.hh"
using std::cerr;

IIRScram_ConcurrentSelectedSignalAssignment::~IIRScram_ConcurrentSelectedSignalAssignment() {}


void 
IIRScram_ConcurrentSelectedSignalAssignment::_publish_vhdl(ostream &_vhdl_out) {

  _publish_vhdl_stmt_label(_vhdl_out);

  if (get_postponed() == true) {
    _vhdl_out << "postponed ";
  }
  _vhdl_out << " with ";
  
  get_expression()->_publish_vhdl(_vhdl_out);
  
  _vhdl_out << " select\n";
  
  get_target()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " <= ";

  _publish_vhdl_delay_mechanism(_vhdl_out, get_delay_mechanism(), get_reject_time_expression());

  selected_waveforms._publish_vhdl(_vhdl_out, ",\n");
  _vhdl_out << ";\n";

}


IIR * 
IIRScram_ConcurrentSelectedSignalAssignment::_transmute() {
  //### yet to take care of guarded signal assignment

  IIR_ProcessStatement* pstmt = new IIR_ProcessStatement;
  IIR_WaitStatement* wstmt = new IIR_WaitStatement;
  IIR_IfStatement* guardIfStmt = NULL;

  copy_location( this, pstmt );
  copy_location( this, wstmt );

  IIR_SignalAssignmentStatement* sastmt                  = NULL;
  IIR_CaseStatement* casestmt                            = NULL;
  IIR* cndtion;

  pstmt->set_label(get_label());
  pstmt->set_postponed(get_postponed());

  if(selected_waveforms.num_elements() == 0) {
    cerr << "IIRScram_ConcurrentSelectedSignalAssignment: Encountered Conditional Signal Assignment Stmt with an empty waveform list.\n";
    abort();
    return NULL;
  }

  // If it is a guarded sign assignemnt then the transform is done as per
  // LRM chapter 9.5  
  if (get_guarded() == TRUE) {
    ASSERT(_get_guard_signal() != NULL);
    cndtion = _get_guard_signal();
    guardIfStmt = new IIR_IfStatement();
    copy_location( this, guardIfStmt );
    guardIfStmt->set_condition(cndtion);
      
    // else clause 
    // The disconnection specification applies only if the target is a
    // guarded target - section 9.5 of LRM
    if (get_target()->_get_signal_kind() == IIR_BUS_KIND || get_target()->_get_signal_kind() == IIR_REGISTER_KIND) {
      sastmt = new IIR_SignalAssignmentStatement;
      copy_location( this, sastmt );
      sastmt->set_target(get_target());
      sastmt->set_delay_mechanism(get_delay_mechanism());
      sastmt->set_reject_time_expression(get_reject_time_expression());
      IIR_WaveformElement *wave = new IIR_WaveformElement();
      copy_location( this, wave );
      // Post a null transaction 
      IIR_StringLiteral *nullLit = IIRBase_StringLiteral::get("null", 4);
      nullLit->_set_subtype(get_target()->get_subtype());
      wave->set_value(nullLit);
      // Set the disconnection specification
      wave->set_time(NULL);
      sastmt->waveform.append(wave);
      guardIfStmt->else_sequence.append(sastmt);
    }

    ASSERT (guardIfStmt != NULL);
    pstmt->process_statement_part.append(guardIfStmt);
  }
  
  casestmt = new IIR_CaseStatement;
  copy_location( this, casestmt );
  casestmt->set_expression(get_expression());
  IIR_CaseStatementAlternativeList *new_list = _build_alternative_list();
  casestmt->case_statement_alternatives = *new_list;

  if (guardIfStmt != NULL) {
    guardIfStmt->then_sequence.append(casestmt);
  } else {
    pstmt->process_statement_part.append(casestmt);
  }

  IIR_DesignatorList sensitivity_list;
  this->_build_sensitivity_list(&sensitivity_list);
  // Adding the guard signal to the sensitivity list for GSA's
  if (get_guarded() == TRUE) {
    _get_guard_signal()->_build_sensitivity_list(&sensitivity_list);
  }
  wstmt->sensitivity_list._add_signals_to_sensitivity_list(&sensitivity_list);
  pstmt->process_statement_part.append(wstmt);  
  return pstmt;  
}


void 
IIRScram_ConcurrentSelectedSignalAssignment::_type_check(){
  // We need to do several things here.  First, we'll type check the
  // expression, and the choices.
  _type_check_expression_and_choices();

  // Next, we'll type check the target and the waveforms.
  _type_check_target_and_waveforms();

  if( get_guarded() == TRUE ){
    _resolve_guard_signal( _get_symbol_table() );
  }

  ASSERT( _is_resolved() == TRUE );
}


IIR_Boolean 
IIRScram_ConcurrentSelectedSignalAssignment::_is_resolved(){
  IIR_Boolean retval = TRUE;

  if( _get_target()->_is_resolved() == FALSE ){
    retval = FALSE;
  }
  
  if( _get_reject_time_expression() != NULL &&
      _get_reject_time_expression()->_is_resolved() == FALSE ){
    retval = FALSE;
  }

  if( selected_waveforms._is_resolved() == FALSE ){
    retval = FALSE;
  }

  return retval;
}


void 
IIRScram_ConcurrentSelectedSignalAssignment::_type_check_expression_and_choices(){
  IIR_CaseStatementAlternativeList *temp_alternative_list = _build_alternative_list();
  ASSERT( temp_alternative_list->num_elements() == selected_waveforms.num_elements() );
  _type_check_case_statement_expression( temp_alternative_list );
  _read_alternative_list( temp_alternative_list );
  delete temp_alternative_list;
}


void 
IIRScram_ConcurrentSelectedSignalAssignment::_type_check_target_and_waveforms(){
  IIR_SelectedWaveform *current_waveform = selected_waveforms.first();
  while( current_waveform != NULL ){
    _type_check_target_and_waveform( &current_waveform->waveform );
    _type_check_mechanism_and_time( &current_waveform->waveform );    
    current_waveform = selected_waveforms.successor( current_waveform );
  }
}


IIR_CaseStatementAlternativeList *
IIRScram_ConcurrentSelectedSignalAssignment::_build_alternative_list(){
  IIR_SelectedWaveform *current_waveform;
  IIR_CaseStatementAlternativeList *retval = new IIR_CaseStatementAlternativeList();
  IIR_SignalAssignmentStatement *sastmt;
  
  //  copy_location( this, retval );
  current_waveform = selected_waveforms.first();
  while( current_waveform != NULL ){
    IIR *current_choice = current_waveform->get_choice();
    ASSERT( current_choice->_is_iir_case_statement_alternative() == TRUE );
   
    if (_get_currently_publishing_unit() != NONE_PARSING) {
      sastmt = new IIR_SignalAssignmentStatement();
      copy_location (this, sastmt);
      ASSERT ( get_target()->_is_resolved() == TRUE );
      
      sastmt->set_target(get_target());
      sastmt->set_delay_mechanism(get_delay_mechanism());
      sastmt->set_reject_time_expression(get_reject_time_expression());
      sastmt->waveform = current_waveform->waveform;
      
      ((IIR_CaseStatementAlternative *) current_choice)->sequence_of_statements.append(sastmt);
    }
    
    retval->append( (IIR_CaseStatementAlternative *)current_choice );
      
    current_waveform = selected_waveforms.successor( current_waveform );    
  }

  ASSERT( retval->num_elements() == selected_waveforms.num_elements() );

  return retval;
}


void 
IIRScram_ConcurrentSelectedSignalAssignment::_read_alternative_list( IIR_CaseStatementAlternativeList *alt_list ){
  ASSERT( alt_list != NULL );
  ASSERT( alt_list->num_elements() == selected_waveforms.num_elements() );

  IIR_SelectedWaveform *current_waveform;
  IIR_CaseStatementAlternative *current_alt;

  current_waveform = selected_waveforms.first();
  current_alt = alt_list->first();
  while( current_waveform != NULL ){
    ASSERT( current_alt != NULL );

    ASSERT( current_alt->_is_iir_case_statement_alternative() == TRUE );
    current_waveform->set_choice( current_alt );

    current_alt = alt_list->successor( current_alt );
    current_waveform = selected_waveforms.successor( current_waveform );    
  }
}


void
IIRScram_ConcurrentSelectedSignalAssignment::_build_sensitivity_list(IIR_DesignatorList* sensitivity_list) {
  get_expression()->_build_sensitivity_list(sensitivity_list);
  selected_waveforms._build_sensitivity_list(sensitivity_list);
}


IIR *
IIRScram_ConcurrentSelectedSignalAssignment::_clone() {
  IIR *cstmt;
  cstmt = _transmute();
  cstmt = cstmt->_clone();
  return cstmt;
}


#ifdef PROCESS_COMBINATION
void
IIRScram_ConcurrentSelectedSignalAssignment::
_static_elaborate(IIR_ArchitectureDeclaration *arch,
		  IIR_DeclarationList *cfglist,
		  char *hier_location) {
  ostringstream newname;
  IIR_Label *label;
  IIR_Char *text;
  IIR_Int32 i;

  newname << hier_location;
  label = get_label();
  if (label != NULL) {
    newname << *label->get_declarator();
  }
  else {
    newname << "UNKNOWNCSA";
  }
  text = newname.str();
#ifdef DEBUG_ELAB
  cout << "elaborated ConcSelSigAssign |" << text << "|\n";
#endif

  ASSERT(label->get_declarator()->get_kind() == IIR_IDENTIFIER);
  ((IIR_Identifier *)label->get_declarator())->release();
  label->set_declarator(IIR_Identifier::get(text, strlen(text)));
}
#endif


IIR *
IIRScram_ConcurrentSelectedSignalAssignment::_get_target(){
  return get_target();
}


void 
IIRScram_ConcurrentSelectedSignalAssignment::_set_target( IIR *new_target ){
  set_target( new_target );
}


IIR_DelayMechanism 
IIRScram_ConcurrentSelectedSignalAssignment::_get_delay_mechanism(){
  return get_delay_mechanism();
}


void 
IIRScram_ConcurrentSelectedSignalAssignment::_set_delay_mechanism( IIR_DelayMechanism new_mech ){
  set_delay_mechanism( new_mech );
}


IIR *
IIRScram_ConcurrentSelectedSignalAssignment::_get_reject_time_expression(){
  return get_reject_time_expression();
}


void 
IIRScram_ConcurrentSelectedSignalAssignment::_set_reject_time_expression( IIR *new_reject_time ){
  set_reject_time_expression( new_reject_time );
}


IIR *
IIRScram_ConcurrentSelectedSignalAssignment::_get_case_statement_expression(){
  return get_expression();
}


void 
IIRScram_ConcurrentSelectedSignalAssignment::_set_case_statement_expression( IIR *new_expression ){
  set_expression( new_expression );
}

visitor_return_type *IIRScram_ConcurrentSelectedSignalAssignment::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_ConcurrentSelectedSignalAssignment(this, arg);
};
