#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc ProjectionDialog {frame id} {
    global menu
    global marker
    global ds9

    # see if we already have a header window visible

    set w ".marker$id"
    set mb ".mb$id"

    if [winfo exist $w] {
	raise $w
	return
    }

    # init marker variables

    set marker($frame,$id,system) $marker(dialog,system)
    set marker($frame,$id,clabel) $marker(dialog,system)
    set marker($frame,$id,sky) $marker(dialog,sky)
    set marker($frame,$id,skyformat) $marker(dialog,skyformat)

    set marker($frame,$id,tcoord) $marker(dialog,dist,system)
    set marker($frame,$id,tlabel) $marker(dialog,dist,system)
    set marker($frame,$id,tformat) $marker(dialog,dist,format)
    set marker($frame,$id,dcoord) $marker(dialog,dist,system)
    set marker($frame,$id,dlabel) $marker(dialog,dist,system)
    set marker($frame,$id,dformat) $marker(dialog,dist,format)

    # projection specific callbacks

    $frame marker $id callback move EditProjectionCB $frame
    $frame marker $id callback edit EditProjectionCB $frame
    $frame marker $id callback text TextMarkerCB $frame
    $frame marker $id callback color ColorMarkerCB $frame
    $frame marker $id callback width LineWidthMarkerCB $frame
    $frame marker $id callback property PropertyMarkerCB $frame
    $frame marker $id callback font FontMarkerCB $frame
    $frame marker $id callback delete DeleteProjectionDialog $frame

    # define menus

    MBMarkerDialog $frame $mb
    FileMenuMarkerDialog $frame $mb $id
    ColorMenuMarkerDialog $frame $mb $id
    WidthMenuMarkerDialog $frame $mb $id
    PropMenuProjectionDialog $frame $mb $id
    FontMenuMarkerDialog $frame $mb $id
    CoordMenuMarkerDialog $frame $mb $id CoordProjectionCB
    DistMenuMarkerDialog $frame $mb $id ThickProjectionCB thick tcoord tformat
    $mb add cascade -label Thickness -menu $mb.thick
    DistMenuMarkerDialog $frame $mb $id DistProjectionCB dist dcoord dformat
    $mb add cascade -label Distance -menu $mb.dist
    MethodMenuProjectionDialog $frame $mb $id
    $mb add cascade -label Method -menu $mb.method

    # Init projection variables

    EditProjectionCB $frame $id
    TextMarkerCB $frame $id
    ColorMarkerCB $frame $id
    LineWidthMarkerCB $frame $id
    PropertyMarkerCB $frame $id
    FontMarkerCB $frame $id
    CoordMarkerCB $frame $id
    ThickProjectionCB $frame $id
    DistProjectionCB $frame $id

    # file menu items

    $mb.file add command -label "Apply" \
	-command "ApplyProjectionDialog $frame $id"
    $mb.file add separator
    $mb.file add command -label "Close" \
	-command "CloseProjectionDialog $frame $id"

    # create window

    set type [string toupper "[$frame get marker $id type]"]

    toplevel $w -colormap $ds9(main)
    wm title $w $type
    wm iconname $w $type
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW "CloseProjectionDialog $frame $id"
    wm minsize $w 530 225

    $w configure -menu $mb -width 600

    # Dialog

    frame $w.basic -relief groove -borderwidth 2
    frame $w.basic.f
    frame $w.ref -relief groove -borderwidth 2
    frame $w.ref.f
    frame $w.dist -relief groove -borderwidth 2
    frame $w.dist.f
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.basic.f $w.ref.f $w.dist.f -anchor w -padx 4 -pady 4
    pack $w.basic $w.ref $w.dist -fill x 
    pack $w.buttons -fill x -ipadx 4 -ipady 4

    # ID

    label $w.basic.f.idTitle -text "Id"
    label $w.basic.f.idValue -text "$id"

    # Text

    label $w.basic.f.textTitle -text "Text"
    entry $w.basic.f.textValue -textvariable marker($frame,$id,text) -width 40

    grid $w.basic.f.idTitle $w.basic.f.idValue -padx 4 -sticky w
    grid $w.basic.f.textTitle $w.basic.f.textValue -padx 4 -sticky w

    # Points

    label $w.ref.f.title -text "Points"
    entry $w.ref.f.x -textvariable marker($frame,$id,x) -width 13
    entry $w.ref.f.y -textvariable marker($frame,$id,y) -width 13
    label $w.ref.f.coord -relief groove -width 9 -padx 4 \
	-textvariable marker($frame,$id,system)
    entry $w.ref.f.x2 -textvariable marker($frame,$id,x2) -width 13
    entry $w.ref.f.y2 -textvariable marker($frame,$id,y2) -width 13

    # Thick

    label $w.ref.f.title2 -text "Thickness"
    entry $w.ref.f.thick -textvariable marker($frame,$id,thick) -width 13
    label $w.ref.f.tformat -relief groove -width 9 -padx 4 \
	-textvariable marker($frame,$id,tlabel)

    grid $w.ref.f.title $w.ref.f.x $w.ref.f.y $w.ref.f.coord -padx 4 -sticky w
    grid x $w.ref.f.x2 $w.ref.f.y2 -padx 4 -sticky w
    grid $w.ref.f.title2 $w.ref.f.thick x $w.ref.f.tformat -padx 4 -sticky w

    # Length

    label $w.dist.f.title -text "Length   "
    label $w.dist.f.dist -relief groove -width 12 -padx 4 \
	-textvariable marker($frame,$id,dist)
    label $w.dist.f.dummy -padx 4 -width 13
    label $w.dist.f.format -relief groove -width 9 -padx 4 \
	-textvariable marker($frame,$id,dlabel)

    grid $w.dist.f.title $w.dist.f.dist $w.dist.f.dummy $w.dist.f.format \
	-padx 4 -sticky w

    # Buttons

    button $w.buttons.apply -text "Apply" \
	-command "ApplyProjectionDialog $frame $id"
    button $w.buttons.close -text "Close" \
	-command "CloseProjectionDialog $frame $id"
    pack $w.buttons.apply $w.buttons.close -side left -padx 10 -expand true

    bind $w <Return> "ApplyProjectionDialog $frame $id"

    # some window managers need a hint
    raise $w

    # make sure its opened
    ProjectionPlot $frame $id
}

proc PropMenuProjectionDialog {frame mb id} {
    global menu
    global marker

    menu $mb.properties -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.properties add checkbutton -label "Can Edit" \
	-variable marker($frame,$id,edit) \
	-command "PropertyMarkerDialog $frame $id edit"
    $mb.properties add checkbutton -label "Can Move" \
	-variable marker($frame,$id,move) \
	-command "PropertyMarkerDialog $frame $id move"
    $mb.properties add checkbutton -label "Can Delete" \
	-variable marker($frame,$id,delete) \
	-command "PropertyMarkerDialog $frame $id delete"
}

proc MethodMenuProjectionDialog {frame mb id} {
    global menu
    global marker

    menu $mb.method -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.method add radiobutton -label "Average" \
	-variable marker($frame,$id,method) -value 1 \
	-command "EditProjectionDialog $frame $id"
    $mb.method add radiobutton -label "Sum" \
	-variable marker($frame,$id,method) -value 0 \
	-command "EditProjectionDialog $frame $id"
}

proc ApplyProjectionDialog {frame id} {
    global marker

    EditProjectionDialog $frame $id
    $frame marker $id text \{$marker($frame,$id,text)\}

    UpdateMarkerMenu
}

proc CloseProjectionDialog {frame id} {
    global marker

    $frame marker $id delete callback edit EditProjectionCB
    $frame marker $id delete callback move EditProjectionCB
    $frame marker $id delete callback text TextMarkerCB
    $frame marker $id delete callback color ColorMarkerCB
    $frame marker $id delete callback width LineWidthMarkerCB
    $frame marker $id delete callback property PropertyMarkerCB
    $frame marker $id delete callback font FontMarkerCB
    $frame marker $id delete callback delete DeleteProjectionDialog

    DeleteProjectionDialog $frame $id
}

proc DeleteProjectionDialog {frame id} {
    global marker

    DeleteMarkerDialog $frame $id

    unset marker($frame,$id,x2)
    unset marker($frame,$id,y2)
    unset marker($frame,$id,tcoord)
    unset marker($frame,$id,tformat)
    unset marker($frame,$id,tlabel)
    unset marker($frame,$id,dcoord)
    unset marker($frame,$id,dformat)
    unset marker($frame,$id,dlabel)
    unset marker($frame,$id,method)
}

proc EditProjectionDialog {frame id} {
    global marker

    $frame marker $id projection \
	$marker($frame,$id,system) $marker($frame,$id,sky) \
	$marker($frame,$id,x) $marker($frame,$id,y) \
	$marker($frame,$id,x2) $marker($frame,$id,y2) \
    	$marker($frame,$id,thick) $marker($frame,$id,method) \
	$marker($frame,$id,tcoord) $marker($frame,$id,tformat)
}

proc EditProjectionCB {frame id} {
    global marker

    set p [$frame get marker $id projection point \
	       $marker($frame,$id,system) $marker($frame,$id,sky) \
	       $marker($frame,$id,skyformat)]

    set marker($frame,$id,x) [lindex $p 0]
    set marker($frame,$id,y) [lindex $p 1]
    set marker($frame,$id,x2) [lindex $p 2]
    set marker($frame,$id,y2) [lindex $p 3]

    set marker($frame,$id,method) [$frame get marker $id projection method]

    ThickProjectionCB $frame $id
    DistProjectionCB $frame $id
}

proc CoordProjectionCB {frame id} {
    global marker

    CoordMarkerCB $frame $id

    $frame marker $id projection system $marker($frame,$id,system) \
	$marker($frame,$id,sky)
    ProjectionXAxisTitle $frame $id
    EditProjectionCB $frame $id

    ProjectionPlot $frame $id
}

proc DistProjectionCB {frame id} {
    global marker
    set mb ".mb$id"

    AdjustDist $frame marker($frame,$id,dcoord)

    set marker($frame,$id,dlabel) $marker($frame,$id,dcoord)
    switch -- $marker($frame,$id,dcoord) {
	image -
	physical -
	amplifier -
	detector {}
	default {
	    if [$frame has wcs $marker($frame,$id,dcoord)] {
		if [$frame has wcs equatorial $marker($frame,$id,dcoord)] {
		    set marker($frame,$id,dlabel) $marker($frame,$id,dformat)
		} else {
		    set name [$frame get wcs name $marker($frame,$id,dcoord)]
		    if {$name != ""} {
			set marker($frame,$id,dlabel) $name
		    }
		}
	    }
	}
    }

    set marker($frame,$id,dist) [$frame get marker $id projection distance \
		$marker($frame,$id,dcoord) $marker($frame,$id,dformat)]
}

proc ThickProjectionCB {frame id} {
    global marker
    set mb ".mb$id"

    AdjustDist $frame marker($frame,$id,tcoord)

    set marker($frame,$id,tlabel) $marker($frame,$id,tcoord)
    switch -- $marker($frame,$id,tcoord) {
	image -
	physical -
	amplifier -
	detector {}
	default {
	    if [$frame has wcs $marker($frame,$id,tcoord)] {
		if [$frame has wcs equatorial $marker($frame,$id,tcoord)] {
		    set marker($frame,$id,tlabel) $marker($frame,$id,tformat)
		} else {
		    set name [$frame get wcs name $marker($frame,$id,tcoord)]
		    if {$name != ""} {
			set marker($frame,$id,tlabel) $name
		    }
		}
	    }
	}
    }

    set marker($frame,$id,thick) [$frame get marker $id projection thick \
		$marker($frame,$id,tcoord) $marker($frame,$id,tformat)]
}

proc SetupProjectionPlot {id} {
    global current

    ProjectionPlot $current(frame) $id

    $current(frame) marker $id callback move ProjectionPlot $current(frame)
    $current(frame) marker $id callback edit ProjectionPlot $current(frame)
    $current(frame) marker $id callback update ProjectionPlot $current(frame)
    $current(frame) marker $id callback delete \
	DeleteProjectionPlot $current(frame)
}

# hardcoded into ds9parser.Y
proc ProjectionPlot {frame id} {
    global marker 
    global ap

    set tt proj${id}${frame}
    set xx proj${id}${frame}x
    set yy proj${id}${frame}y
    set xe proj${id}${frame}xe
    set ye proj${id}${frame}ye

    set xc proj${id}${frame}xc
    set yc proj${id}${frame}yc

    global $xx $yy $xe $ye $xc $yc
    if {[APPing $tt]} {
	$frame get marker $id projection $xx $yy $xc $yc 
#	ProjectionStats $frame $id
    } else {
	APDialog $tt Projection Projection "physical coordinates" Counts
	$ap($tt,graph) xaxis configure -command "ProjectionXAxis $frame $id"

	set ap($tt,xdata) $xx
	set ap($tt,ydata) $yy
	set ap($tt,xedata) $xe
	set ap($tt,yedata) $ye

	blt::vector create $xx $yy $xe $ye $xc $yc
	$frame marker $id projection system physical fk5
	$frame get marker $id projection $xx $yy $xc $yc
	
	ProjectionXAxisTitle $frame $id
	APExternal $tt
#	ProjectionStats $frame $id
    }
}

# hardcoded into ds9parser.Y
proc DeleteProjectionPlot {frame id} {
    set xc proj${id}${frame}xc
    set yc proj${id}${frame}yc

    global $xc $yc
    catch {blt::vector destroy $xc $yc}
    # clear any errors
    global errorInfo
    set errorInfo {}

    APDestroy proj${id}${frame}
}

proc ProjectionStats {frame id} {
    global ap

    set yy proj${id}${frame}y
    global $yy

    set min [format "%6.3f" [blt::vector expr min($yy)]]
    set max [format "%6.3f" [blt::vector expr max($yy)]]
    set mean [format "%6.3f" [blt::vector expr mean($yy)]]
    set median [format "%6.3f" [blt::vector expr median($yy)]]
    
    set var [format "%6.3f" [expr [blt::vector expr var($yy)]]]
    set sdev [format "%6.3f" [expr [blt::vector expr sdev($yy)]]]
    
    set title {}
    append title "min $min max $max"
    append title "\nmean $mean median $median"
    append title "\nvar $var sdev $sdev"

    set tt proj${id}${frame}
    $ap($tt,graph) configure -title $title
}

proc ProjectionXAxisTitle {frame id} {
    global ap

    set system [$frame get marker $id projection system]
    set sky  [$frame get marker $id projection sky]
    switch -- $system {
	image -
	physical -
	amplifier -
	detector {set title "$system coordinates"}
	default {
	    if {[$frame has wcs equatorial $system]} {
		set title "$sky coordinates"
	    } else {
		set title "[$frame get wcs name $system] coordinates"
	    }
	}
    }
    set tt proj${id}${frame}
    $ap($tt,graph) xaxis configure -title $title
}

proc ProjectionXAxis {frame id w x} {
    set xc proj${id}${frame}xc
    set yc proj${id}${frame}yc

    global $xc $yc
    # sometimes, $x equals $xc length
    if {$x<[$xc length]} {
	set a [format "%6.3f" [expr "$$xc\($x\)"]]
	set b [format "%6.3f" [expr "$$yc\($x\)"]]

	return "$a\n$b"
    } else {
	return {}
    }
}

proc ProjectionDefaultDialog {} {
    global marker
    global ed

    set w ".projection"

    set ed(ok) 0
    set ed(thick) $marker(projection,thick)

    DialogCreate $w "Default Projection" -borderwidth 2
    frame $w.ed  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ed $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.ed.title -text "Thickness"
    entry $w.ed.thick -textvariable ed(thick) -width 10
    label $w.ed.unit -text "image" -relief groove -width 8
    
    grid $w.ed.title $w.ed.thick $w.ed.unit -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active \
	-command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.ed.thick select range 0 end
    DialogWait $w ed(ok) $w.ed.thick
    DialogDismiss $w

    if {$ed(ok)} {
	set marker(projection,thick) $ed(thick)
    }

    unset ed
}
