/*
 *  Copyright (C) 2000 heXoNet Support GmbH, D-66424 Homburg.
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "rfbServer.h"
#include "version.h"
#include "OXAbout.h"

#include "../icons/rfb_service_large.xpm"

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <X11/keysym.h>


#include <xclass/OXClient.h>
#include <xclass/OXWindow.h>
#include <xclass/OXFrame.h>
#include "OSocketFileHandler.h"
#include <xclass/OIdleHandler.h>
#include <xclass/OTimer.h>


#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#include <X11/extensions/XTest.h>

#include "OXServiceApplet.h"
#include "OXProperties.h"
#include "OXNewClient.h"
#include "XUpdateScanner.h"
#include "OXProperties.h"

#ifdef USE_ZLIB_WARREN
#include "ZlibConnection.h"
#endif // USE_ZLIB_WARREN

#include <map>
#include <list>

Properties properties;

OXClient *clientX;
const OXWindow *root;
Display *dpy;


namespace rfb {



class BaseServer;



class OXServerMainFrame: public OXMainFrame
{
  public:
    OXServerMainFrame( int w, int h, bool stdio = false );
    virtual ~OXServerMainFrame();
    
    void CreateFramebuffer();
    void DestroyFramebuffer();
    
    void ScanUpdates();
    
    virtual int HandleIdleEvent( OIdleHandler *_event );
    virtual int HandleTimer( OTimer *_timer );
    
    virtual int HandleFileEvent( OFileHandler *_fh, unsigned int _mask );
    
    void KillAllClients();
    void AddNewClient();

    void RegisterServer( BaseServer *server );
    void UnregisterServer( BaseServer *server );


    struct sockaddr_in name;
    OFileHandler *fhSocket;
    OIdleHandler *idle;
    OTimer *timer;

    XImage *framebufferImage;
    Framebuffer framebuffer;
    XUpdateScanner *scanner;
    
    OXServiceApplet *serviceApplet;
    
    map< OFileHandler*, BaseServer* > handleServerMap;
    list< BaseServer* > serverList;
};


class BaseServer: public Server
{
  public:
    BaseServer( int _fd );
    virtual ~BaseServer();
    
    virtual void handleKeyEvent( KeyEvent &keyEvent );
    virtual void handlePointerEvent( PointerEvent &pointerEvent );

    virtual void getServerInitialisation( ServerInitialisation &_serverInitialisation );

#ifdef USE_ZLIB_WARREN
    virtual void handleEnableZlib( CARD8 level );
    ZlibConnection *bufferedConnection;
#else
    BufferedConnection *bufferedConnection;
#endif // USE_ZLIB_WARREN

    int fd;
    OSocketFileHandler *fh;
    int buttonMask;
    
    int modifier;
    
    bool key_Mode_switch;
    bool key_Shift_L;
    bool key_Shift_R;
    bool key_Control_L;
    bool key_Control_R;
    bool key_Caps_Lock;
    bool key_Shift_Lock;
    bool key_Meta_L;
    bool key_Meta_R;
    bool key_Alt_L;
    bool key_Alt_R;
    bool key_Super_L;
    bool key_Super_R;
    bool key_Hyper_L;
    bool key_Hyper_R;
};



OXServerMainFrame *mainFrame;


OXServerMainFrame::OXServerMainFrame( int w, int h, bool stdio )
      : OXMainFrame( clientX->GetRoot(), w, h )
    {
      mainFrame = this;
      idle = NULL;
      timer = NULL;
      if ( stdio ) {
        serviceApplet = NULL;
	
      }
      else {
          char suffix[] = "/.x0rfbserver";
          char filename[strlen(getenv("HOME"))+strlen(suffix)+1];
          sprintf( filename, "%s%s", getenv("HOME"), suffix );
          struct stat statbuf;
          if ( stat( filename, &statbuf ) )
              new OXProperties( _client->GetRoot(),
                                this,
                                &properties );
          else {
	     readProperties( filename, properties );
	  }
	
        int s = socket( PF_INET, SOCK_STREAM, 0 );
        if ( s < 0 ) delete this;

        int one = 1;
        setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&one, sizeof(one));

        int port = 5900 + properties.displayNumber;
        if ( properties.autoDisplayNumber ) {
            port = atoi( strstr( DisplayString(clientX->GetDisplay()), ":" ) + 1 ) + 5899;
            do {
                port++;
                name.sin_family = AF_INET;
                name.sin_port = htons( port );
                name.sin_addr.s_addr = htonl (INADDR_ANY);
            } while ( bind (s, (struct sockaddr *) &name, sizeof (name) ) );
        }
        else {
            name.sin_family = AF_INET;
            name.sin_port = htons( port );
            name.sin_addr.s_addr = htonl (INADDR_ANY);
            if ( bind (s, (struct sockaddr *) &name, sizeof (name) ) ) {
                delete this;
                exit(1);
            }
        }
        if ( listen(s, 5) ) {
            delete this;
            exit(1);
        }
//        fhSocket = new OFileHandler( this, s, XCM_READABLE | XCM_EXCEPTION );
        fhSocket = new OFileHandler( this, s, XCM_READABLE );

        serviceApplet = new OXServiceApplet( clientX,
                                             (port < 6000)? port - 5900 : port );
      }
      clientX->Run();
    }


OXServerMainFrame::~OXServerMainFrame()
{
  KillAllClients();
}



void AddNewClient()
{
  mainFrame->AddNewClient();
}

void KillAllClients()
{
  mainFrame->KillAllClients();
}




int init_sockaddr (struct sockaddr_in *name, const char *hostname, uint16_t port)
{
  struct hostent *hostinfo;
  
  name->sin_family = AF_INET;
  name->sin_port = htons (port);
  hostinfo = gethostbyname (hostname);
  if (hostinfo == NULL)
    {
      return -1;
    }
  name->sin_addr = *(struct in_addr *) hostinfo->h_addr;
  return 0;
}


int doConnect( char *hostname, int port )
{
    struct sockaddr_in name;
    
    int s = socket( PF_INET, SOCK_STREAM, 0 );

    if ( init_sockaddr( &name, hostname, port ) )
      return -1;

    if ( connect( s, (struct sockaddr*) &name, sizeof( name ) ) ) {
        return -1;
    }

    int one = 1;
    setsockopt(s, IPPROTO_TCP, TCP_NODELAY, (char *)&one, sizeof(one));
    fcntl(s, F_SETFL, O_NONBLOCK);

    return s;
}


void OXServerMainFrame::AddNewClient()
{
  char buffer[1024];
  if ( getNewClient( buffer, sizeof(buffer), _client->GetRoot(), this ) ) {
    char *pos;
    if ( (pos = strchr( buffer, ':' )) ) {
      (*pos) = 0;
      pos++;
    }
    int s;
    if ( pos )
      s = doConnect( buffer, (atoi(pos)<100)? 5500+atoi(pos) : atoi(pos) );
    else
      s = doConnect( buffer, 5500 );
    char savePassword[ sizeof(properties.password) ];
    strncpy( savePassword, properties.password, sizeof(properties.password) );
    properties.password[0] = 0;
    if ( s >= 0 )
      new BaseServer( s );
    strncpy( properties.password, savePassword, sizeof(properties.password) );
  }
}


void OXServerMainFrame::KillAllClients()
{
  while ( !serverList.empty() )
    delete *(serverList.begin());
}


void OXServerMainFrame::RegisterServer( BaseServer *server )
{
  if ( serverList.empty() ) {
    if (serviceApplet) serviceApplet->SetIcon( true );
    XTestGrabControl( dpy, true );
    CreateFramebuffer();
  }
  serverList.push_back( server );
  handleServerMap[server->fh] = server;
}

void OXServerMainFrame::UnregisterServer( BaseServer *server )
{
  handleServerMap.erase(server->fh);
  serverList.remove( server );
  if ( serverList.empty() ) {
    if (idle) delete idle;
    idle = NULL;
    if (timer) delete timer;
    timer = NULL;
    DestroyFramebuffer();
    XTestDiscard( dpy );
    if (serviceApplet) serviceApplet->SetIcon( false );
  }
}



void OXServerMainFrame::CreateFramebuffer()
{
  cerr << "CreateFramebuffer() start" << endl;
  cerr << "dpy" << dpy << endl;
  cerr << "id " << root->GetId() << endl;
  cerr << "w  " << clientX->GetDisplayWidth() << endl;
  cerr << "h  " << clientX->GetDisplayHeight() << endl;
  framebufferImage = XGetImage( dpy,
                                root->GetId(),
				0,
				0,
				clientX->GetDisplayWidth(),
				clientX->GetDisplayHeight(),
				AllPlanes,
				ZPixmap );
  cerr << "FI " << framebufferImage << endl;
  framebuffer.width        = framebufferImage->width;
  framebuffer.height       = framebufferImage->height;
  framebuffer.bytesPerLine = framebufferImage->bytes_per_line;
  framebuffer.data         = (unsigned char*) framebufferImage->data;
  
  framebuffer.pixelFormat.bits_per_pixel   = framebufferImage->bits_per_pixel;
  framebuffer.pixelFormat.depth            = framebufferImage->depth;
  framebuffer.pixelFormat.big_endian_flag  = framebufferImage->bitmap_bit_order == MSBFirst;
  framebuffer.pixelFormat.true_colour_flag = true;

  if ( framebuffer.pixelFormat.bits_per_pixel == 8 ) {
    framebuffer.pixelFormat.red_shift = 0;
    framebuffer.pixelFormat.green_shift = 2;
    framebuffer.pixelFormat.blue_shift = 5;
    framebuffer.pixelFormat.red_max   = 3;
    framebuffer.pixelFormat.green_max = 7;
    framebuffer.pixelFormat.blue_max  = 3;
  } else {
    framebuffer.pixelFormat.red_shift = 0;
    if ( framebufferImage->red_mask )
      while ( ! ( framebufferImage->red_mask & (1 << framebuffer.pixelFormat.red_shift) ) )
        framebuffer.pixelFormat.red_shift++;
    framebuffer.pixelFormat.green_shift = 0;
    if ( framebufferImage->green_mask )
      while ( ! ( framebufferImage->green_mask & (1 << framebuffer.pixelFormat.green_shift) ) )
        framebuffer.pixelFormat.green_shift++;
    framebuffer.pixelFormat.blue_shift = 0;
    if ( framebufferImage->blue_mask )
      while ( ! ( framebufferImage->blue_mask & (1 << framebuffer.pixelFormat.blue_shift) ) )
      framebuffer.pixelFormat.blue_shift++;
    framebuffer.pixelFormat.red_max   = framebufferImage->red_mask   >> framebuffer.pixelFormat.red_shift;
    framebuffer.pixelFormat.green_max = framebufferImage->green_mask >> framebuffer.pixelFormat.green_shift;
    framebuffer.pixelFormat.blue_max  = framebufferImage->blue_mask  >> framebuffer.pixelFormat.blue_shift;
  }
  scanner = new XUpdateScanner( dpy, root->GetId(), &framebuffer );

  cerr << "fpf " << int(framebuffer.pixelFormat.bits_per_pixel) << endl;

  cerr << "CreateFramebuffer() end" << endl;
}



void OXServerMainFrame::DestroyFramebuffer()
{
  delete scanner;
  XDestroyImage( framebufferImage );
}


int OXServerMainFrame::HandleTimer( OTimer *_timer )
{
  if ( _timer == timer ) {
//      cerr << "TIMER" << endl;
      delete ( timer );
      timer = NULL;
      idle = new OIdleHandler( this );
      return True;
  }
  return False;
}


int OXServerMainFrame::HandleIdleEvent( OIdleHandler *_idle )
{
  if ( idle ) delete ( idle );
  idle = NULL;
  if ( timer ) delete ( timer );
  timer = NULL;
  if ( serverList.empty() ) return true;
  if ( properties.scanDelay ) {
      if (!timer) timer = new OTimer( this, properties.scanDelay );
  }
  list< BaseServer* >::iterator s;
  int wantsUpdate = 0;
  for ( s = serverList.begin(); s != serverList.end(); s++ )
      if ( (*s)->framebufferUpdateRequested 
          || (*s)->incrementalFramebufferUpdateRequested ) wantsUpdate++;
  if ( properties.scanOnRequestOnly ) {
      for ( s = serverList.begin(); s != serverList.end(); s++ )
          (*s)->framebufferUpdateRequested = 0;
  }
  if ( !wantsUpdate ) return true;
  ScanUpdates();
  for ( s = serverList.begin(); s != serverList.end(); s++ )
    (*s)->sendIncrementalFramebufferUpdate();
  for ( s = serverList.begin(); s != serverList.end(); s++ )
    (*s)->fh->SetEventMask(XCM_READABLE|XCM_WRITABLE);
  return true;
};


void OXServerMainFrame::ScanUpdates()
{
    list< BaseServer* >::iterator s;
//    cerr << "StartUpdates() start" << endl;
    list< Hint > hintList;

    scanner->searchUpdates( hintList );

    list< Hint >::iterator i;
    for ( i = hintList.begin(); i != hintList.end(); i++ )
        for ( s = serverList.begin(); s != serverList.end(); s++ )
            (*s)->handleHint( *i );
//    cerr << "StartUpdates() end" << endl;
};




int OXServerMainFrame::HandleFileEvent( OFileHandler *__fh, unsigned int _mask )
{
  OSocketFileHandler *_fh = (OSocketFileHandler*) __fh;
  if ( !idle && !serverList.empty() ) {
      if ( properties.scanDelay ) {
          if (!timer) timer = new OTimer( this, properties.scanDelay );
      }
      else {
          idle = new OIdleHandler( this );
      }
  }
  int fd = _fh->GetFd();

// Accept new incoming connection
  if ( _fh == fhSocket ) {
    if ( _mask & XCM_READABLE ) {
      struct sockaddr addr;
      size_t laddr = sizeof(addr);
      int fdClient = accept(fd, &addr, &laddr);
      int one = 1;
      setsockopt(fdClient, IPPROTO_TCP, TCP_NODELAY, (char *)&one, sizeof(one));
      fcntl(fdClient, F_SETFL, O_NONBLOCK);
      if ( properties.acceptSocketConnections )
          new BaseServer( fdClient );
      else
          close( fdClient );
    }
    return true;
  }

// Do server->client data transfer
  BaseServer *server = handleServerMap[_fh];
  BufferedConnection *connection = server->bufferedConnection;

  if ( _mask & XCM_READABLE ) {
    _fh->SetEventMask(XCM_READABLE|XCM_WRITABLE);

    int bytesRead = read( fd,
                          connection->receiverBuffer.data,
                          connection->receiverBuffer.size );
    if ( bytesRead > 0 )
      connection->receiverBuffer.end += bytesRead;
    else {
      delete server;
      return true;
    }
    while ( server->currentState && connection->hasReceiverBufferData() )
      server->update();
    connection->receiverBuffer.pos = 0;
    connection->receiverBuffer.end = 0;
  }


  if ( _mask & XCM_WRITABLE ) {
    int bytesWritten = write( fd,
                              connection->senderBuffer.data + connection->senderBuffer.pos,
                              connection->senderBuffer.end - connection->senderBuffer.pos );
    if ( bytesWritten >= 0 ) {
      connection->senderBuffer.pos += bytesWritten;
    }
    else {
//      delete server;
//      return true;
    }
    if ( bytesWritten == 0 ) {
//        cerr << "XCM_WRITABLE : 0" << endl;
        _fh->SetEventMask(XCM_READABLE);
    }
  }

  if (!server->currentState) {
//    delete server;
//    return true;
  }

  if ( _mask & XCM_EXCEPTION ) {
//     delete server;
//     return true;
  }

  return false;
}



void BaseServer::handleKeyEvent( KeyEvent &keyEvent )
{
    if ( keyEvent.key == XK_Mode_switch ) key_Mode_switch = keyEvent.down_flag;
    if ( keyEvent.key == XK_Shift_L )     key_Shift_L = keyEvent.down_flag;
    if ( keyEvent.key == XK_Shift_R )     key_Shift_R = keyEvent.down_flag;
    if ( keyEvent.key == XK_Control_L )   key_Control_L = keyEvent.down_flag;
    if ( keyEvent.key == XK_Control_R )   key_Control_R = keyEvent.down_flag;
    if ( keyEvent.key == XK_Caps_Lock )
        if (keyEvent.down_flag)           key_Caps_Lock = !key_Caps_Lock;
    if ( keyEvent.key == XK_Shift_Lock )
        if (keyEvent.down_flag)           key_Shift_Lock = !key_Shift_Lock;
    if ( keyEvent.key == XK_Meta_L )      key_Meta_L = keyEvent.down_flag;
    if ( keyEvent.key == XK_Meta_R )      key_Meta_R = keyEvent.down_flag;
    if ( keyEvent.key == XK_Alt_L )       key_Alt_L = keyEvent.down_flag;
    if ( keyEvent.key == XK_Alt_R )       key_Alt_R = keyEvent.down_flag;
    if ( keyEvent.key == XK_Super_L )     key_Super_L = keyEvent.down_flag;
    if ( keyEvent.key == XK_Super_R )     key_Super_R = keyEvent.down_flag;
    if ( keyEvent.key == XK_Hyper_L )     key_Hyper_L = keyEvent.down_flag;
    if ( keyEvent.key == XK_Hyper_R )     key_Hyper_R = keyEvent.down_flag;

    modifier = 0;
    if ( key_Shift_L     ) modifier &= ShiftMask;
    if ( key_Shift_R     ) modifier &= ShiftMask;
    if ( key_Shift_Lock  ) modifier &= ShiftMask;
    if ( key_Caps_Lock   ) modifier &= LockMask;
    if ( key_Control_L   ) modifier &= ControlMask;
    if ( key_Control_R   ) modifier &= ControlMask;
    if ( key_Mode_switch ) modifier &= Mod1Mask;
    if ( key_Meta_L      ) modifier &= Mod1Mask;
    if ( key_Meta_R      ) modifier &= Mod1Mask;

    if ( !properties.disableRemoteControl ) {

        if ( properties.useXTest ) {
            KeyCode kc = XKeysymToKeycode( dpy, keyEvent.key );
            if ( kc != NoSymbol )
                XTestFakeKeyEvent( dpy,
                                   XKeysymToKeycode( dpy, keyEvent.key ),
                                   keyEvent.down_flag,
                                   CurrentTime );
        }
        else {
            KeyCode kc = XKeysymToKeycode( dpy, keyEvent.key );
            Window dest;
            int revert;
            XGetInputFocus(dpy, &dest, &revert);
            XKeyEvent event;
            event.type = (keyEvent.down_flag)? KeyPress : KeyRelease;
            event.display = dpy;
            event.window = dest;
            event.root = root->GetId();
            event.subwindow = None;
            event.time = CurrentTime;
            event.x = 1;
            event.y = 1;
            event.x_root = 1;
            event.y_root = 1;
            event.same_screen = True;
            event.state = modifier;
            event.keycode = kc;
            cerr << "XSendEvent (XKeyEvent)" << endl;
            if ( kc != NoSymbol )
                XSendEvent( dpy,
                            dest,
                            True,
                            (keyEvent.down_flag)? KeyPressMask : KeyReleaseMask,
                            (XEvent *) &event );
        }
    }
}



// Stolen from http://www.cs.washington.edu/homes/gjb/patches/fvwm-SendButtonPress.patch

void
fill_x_button_event(XButtonEvent *evt, int type, int button, int modifier, 
                   int x, int y, int x_root, int y_root, Window child, Window sub_window)
{
    evt -> type = type;
    evt -> display = dpy;
    evt -> window = child;
    evt -> subwindow = sub_window;
    evt -> root = root->GetId();
    evt -> time = CurrentTime;
    evt -> x = x;
    evt -> y = y;
    evt -> x_root = x_root;
    evt -> y_root = y_root;
    evt -> same_screen = 1;
    evt -> button = button;
    evt -> state = modifier;
}

Window WindowGettingButtonEvent(Window w, int x, int y) {
    int x2, y2;
    Window child, w2 = w;
    XWindowAttributes wa;

 find_window:
    XTranslateCoordinates(dpy, w, w2, x, y, &x2, &y2, &child);
    if (child) {
       x = x2;
       y = y2;
       w = w2;
       w2 = child;
       goto find_window;
    }
    x = x2;
    y = y2;
    w = w2;

 find_listener:
    XGetWindowAttributes(dpy, w, &wa);
    if (!(wa.all_event_masks & (ButtonPressMask | ButtonReleaseMask))) {
       Window d1, *d3, parent;
       unsigned int d4;
       
       XQueryTree(dpy, w, &d1, &parent, &d3, &d4);
       if (d3) XFree(d3);
       if (parent) {
           w = parent;
           goto find_listener;
       }
    }
    return w;
}

void BaseServer::handlePointerEvent( PointerEvent &pointerEvent )
{
    if ( !properties.disableRemoteControl ) {
    
        if ( properties.useXTest ) {

            XTestFakeMotionEvent( dpy,
                                  0,
                                  pointerEvent.x_position,
                                  pointerEvent.y_position,
                                  CurrentTime );
            int i = 1;
            while (i <= 5) {
              if ( (buttonMask & (1 << (i-1))) != (pointerEvent.button_mask & (1 << (i-1))) )
                XTestFakeButtonEvent( dpy, i, (pointerEvent.button_mask & (1 << (i-1)))? True : False, CurrentTime );
              i++;
            }
        }
        else {
            cerr << "XWarpPointer" << endl;
            XWarpPointer( dpy,
                          None,
                          root->GetId(),
                          0, 0, 0, 0,
                          pointerEvent.x_position,
                          pointerEvent.y_position );

            int i = 1;
            while (i <= 5) {
                if ( (buttonMask & (1 << (i-1))) != (pointerEvent.button_mask & (1 << (i-1))) ) {
                    Window child;
                    XButtonEvent event;
                    int button = i;
                    int x, y, x2, y2;
		    int x_root, y_root;
                    unsigned int JunkMask;
                    Window JunkRoot;
                    Window pointer_win, dummy;
                    XQueryPointer( dpy, root->GetId(), &JunkRoot, &pointer_win,
                                   &x_root,&y_root,&x, &y, &JunkMask);

                    child = WindowGettingButtonEvent(pointer_win,x,y);
                    x2 = x; y2 = y;
                    XTranslateCoordinates(dpy, pointer_win, child, x2, y2,
                                          &x, &y, &dummy);
                    fill_x_button_event(&event,
                                        (pointerEvent.button_mask & (1 << (i-1)))? ButtonPress : ButtonRelease,
                                        button,
                                        modifier, 
                                        x, y, x_root, y_root, child, 0);
                    XSendEvent(dpy,
                               PointerWindow,
                               True,
                               (pointerEvent.button_mask & (1 << (i-1)))? ButtonPressMask : ButtonReleaseMask, 
                               (XEvent *) &event );
                }
                i++;
            }
        }
    }
    buttonMask = pointerEvent.button_mask;
}


#ifdef USE_ZLIB_WARREN
void BaseServer::handleEnableZlib( CARD8 level )
{
  cerr << "EnableZlib: level " << (int) level << endl;
  Server::handleEnableZlib( level );
  bufferedConnection->enableSenderDeflation( level );
  cerr << "Zlib enabled" << endl;
}
#endif // USE_ZLIB_WARREN


void BaseServer::getServerInitialisation( ServerInitialisation &_serverInit )
{
  Server::getServerInitialisation( _serverInit );
  if ( getenv("HOSTNAME") ) {
    _serverInit.name_length = strlen( getenv("HOSTNAME") );
    _serverInit.name_string = (CARD8 *) malloc( _serverInit.name_length + 1 );
    strcpy( (char*) _serverInit.name_string, getenv( "HOSTNAME" ) );
  }
  else if ( getenv("HOST") ) {
    _serverInit.name_length = strlen( getenv("HOST") );
    _serverInit.name_string = (CARD8 *) malloc( _serverInit.name_length + 1 );
    strcpy( (char*) _serverInit.name_string, getenv( "HOST" ) );
  }
  else {
    _serverInit.name_length = 0;
    _serverInit.name_string = (CARD8 *) malloc( 1 );
    _serverInit.name_string[0] = 0;
  }
}




BaseServer::BaseServer( int _fd )
  : Server()
  , fd( _fd )
  , buttonMask( 0 )
  , modifier( 0 )
  , key_Mode_switch( 0 )
  , key_Shift_L( 0 )
  , key_Shift_R( 0 )
  , key_Control_L( 0 )
  , key_Control_R( 0 )
  , key_Caps_Lock( 0 )
  , key_Shift_Lock( 0 )
  , key_Meta_L( 0 )
  , key_Meta_R( 0 )
  , key_Alt_L( 0 )
  , key_Alt_R( 0 )
  , key_Super_L( 0 )
  , key_Super_R( 0 )
  , key_Hyper_L( 0 )
  , key_Hyper_R( 0 )
{
  cerr << "BaseServer() start" << endl;
  memcpy( password, "\0\0\0\0\0\0\0\0", 8 );
  strncpy( password, properties.password, 8 );
#ifdef USE_ZLIB_WARREN
  connection = bufferedConnection = new ZlibConnection();
#else
  connection = bufferedConnection = new BufferedConnection( 32768, 32768 );
#endif // USE_ZLIB_WARREN

  framebuffer = &mainFrame->framebuffer;
  framebuffer->width = clientX->GetDisplayWidth();
  framebuffer->height = clientX->GetDisplayHeight();
  InitBlocks( 32, 32 );
//  fh = new OFileHandler( mainFrame, fd, XCM_READABLE | XCM_WRITABLE | XCM_EXCEPTION );
  fh = new OSocketFileHandler( mainFrame, fd, XCM_READABLE | XCM_WRITABLE );
  mainFrame->RegisterServer( this );
  connection->send( (unsigned char*) RFB_PROTOCOL_VERSION, 12 );
  cerr << "BaseServer() end" << endl;
}


BaseServer::~BaseServer()
{
  cerr << "~BaseServer() start" << endl;
  DeleteBlocks();
  mainFrame->UnregisterServer( this );
  delete bufferedConnection;
  delete fh;
  shutdown( fd, 2 );
  close( fd );
  cerr << "~BaseServer() end" << endl;
}


} // namespace rfb





void printVersion_x0rfbserver()
{
  cerr << "x0rfbserver:" << endl
       << endl << "heXoNet RFB server exporting a running X session"
       << endl << "Version " << VERSION_x0rfbserver
       << endl;
}

void printHelp_x0rfbserver()
{
  printVersion_x0rfbserver();
  cerr 
    << endl 
    << "usage: x0rfbserver [<options>]" << endl
    << "       x0rfbserver -stdio" << endl
    << "       x0rfbserver -about" << endl
    << "       x0rfbserver -help" << endl
    << endl
    << "<options>" << endl
    << "       -shared" << endl
    << "       -viewonly" << endl;
  ;
  exit( 1 );
}


void parseCommandLine_x0rfbserver( int argc, char **argv )
{
  enum { UNDEF, ABOUT, STDIO } mode;
  mode = UNDEF;

  int i = 1;
  while ( i < argc ) {

    if ( !strcmp( argv[i], "-help" ) ) {
      printHelp_x0rfbserver();
    } else
    
    if ( !strcmp( argv[i], "-about" ) ) {
      if ( mode == UNDEF ) mode = ABOUT;
      else printHelp_x0rfbserver();
      i++;
    } else
    
    if ( !strcmp( argv[i], "-stdio" ) ) {
      if ( mode == UNDEF ) mode = STDIO;
      else printHelp_x0rfbserver();
      i++;
    } else printHelp_x0rfbserver();
  }

  switch( mode ) {
    case UNDEF: {
      clientX = new OXClient;
      root = clientX->GetRoot();
      dpy = root->GetDisplay();
      new rfb::OXServerMainFrame( 48, 48 );
      exit( 0 );
    }
    break;

    case ABOUT: {
      clientX = new OXClient;
      OXAbout *about = new OXAbout( clientX->GetRoot(),
	                            new OXMainFrame( clientX->GetRoot(), 16, 16 ),
	                   	    "About: heXoNet x0rfbserver V" VERSION_x0rfbserver,
				    "x0rfbserver version " VERSION_x0rfbserver,
				    "x0rfbserver - Helpdesk from heXoNet",
				    clientX->GetPicture( "rfb_service_large", rfb_service_large_xpm )
				  );
      about->MapWindow();
      clientX->WaitFor( about );
      exit( 0 );
    }
    break;

    
      
    case STDIO: {
      clientX = new OXClient;
      root = clientX->GetRoot();
      dpy = root->GetDisplay();
      new rfb::OXServerMainFrame( 48, 48, true );
      exit( 0 );
    }
    break;
      
  }
}





int main( int argc, char **argv )
{
  parseCommandLine_x0rfbserver( argc, argv );
  
  return 0;
}


