getReposOption <- function() {
    out <- getOption("repositories2")
    out
}

getMangledName <- function(pkg, vers, mangSep=":v:") {
    if (is(vers,"VersionNumber"))
        vers <- stringRep(vers)
    return(paste(pkg,mangSep,vers,sep=""))
}

getReposEntry <- function(repURL, repFile="replisting",
                          rddFile="repdatadesc.rda",
                          rtFile="repThemes.rda", method="auto") {
    ## First check to see if repURL is symbolic
    optReps <- getReposOption()
    if (repURL %in% names(optReps))
        repURL <- as.character(optReps[repURL])

    ## Get the replisting from the remote repository
    repListing <- getReplisting(repURL, repFile, method=method)
    if (is.null(repListing)) {
        note(paste(repURL,
                   "does not seem to have a valid repository, skipping"))
        return(NULL)
    }
    repDf <- loadRepDD(repURL, rddFile, method=method)
    ## repDf is supposed to be able to be NULL, but need something of
    ## class 'data.frame' for the reposEntry object.
    if (is.null(repDf))
        repDf <- data.frame()
    repDf <- new("repdatadesc", repdatadesc=repDf)

    repTheme <- loadRepThemes(repURL, rtFile)
    if (is.null(repTheme))
        repTheme <- list()

    return(buildReposEntry(repListing, repDf, repTheme))
}

getReplisting <- function(repURL,repFile="replisting", method="auto") {
    ## Passed a repository URL, will retrieve the replisting (if any)
    fileURL <- paste(repURL, repFile, sep="/")
    options(show.error.messages = FALSE)
    on.exit(options(show.error.messages = TRUE))

    tmp <- tempfile()
    z <- try(download.file(fileURL, tmp, quiet=TRUE, method=method))
    if (is(z, "try-error") == TRUE) {
        warning(paste("Failed to read replisting at",
                      repURL))
        return(NULL)
    }

    tryRes <- try(read.dcf(tmp))
    options(show.error.messages=TRUE)
    if (is(tryRes,"try-error") == TRUE) {
        return(NULL)
    }
    if (length(tryRes) == 0) {
        warning("Empty replisting file at ", repURL)
        return(NULL)
    }

    repL <- new("replisting", replisting=tryRes)
    return(repL)
}

loadRepDD <- function(repURL, repDD="repdatadesc.rda", method) {
    ## Will load the remote repository's repdatadesc RDA file, if
    ## available
    ddURL <- paste(repURL,repDD,sep="/")

    if (missing(method))
        method <- "auto"

    options(show.error.messages=FALSE)
    retVal <- try(loadURL(ddURL, quiet=TRUE, mode="wb", method=method))
    options(show.error.messages=TRUE)

    if (is(retVal, "try-error")) {
        warning(paste("Failed to read repdatadesc at",
                      repURL))
        return(NULL)
    }

    return(eval(parse(text=retVal)))
}

    setClass("replisting", representation(replisting="matrix"))


    if (is.null(getGeneric("replisting")))
        setGeneric("replisting", function(object)
                   standardGeneric("replisting"))

    if (is.null(getGeneric("repURL")))
        setGeneric("repURL", function(object, pos)
                   standardGeneric("repURL"))

    setMethod("summary","replisting",function(object, ...) {
        repL <- replisting(object)
        rem <- nrow(repL)-1
        repL <- repL[1,,drop=FALSE]
        print.matrix(repL,quote=FALSE,collab=c("Repository Name   ",
                                      "Repository Type   ",
                                      "Repository Base URL   ",
                                      "Repository URL Path   ",
                                      "Repository Release Level"),
                     rowlab=paste(as.character(1:nrow(repL)),":",sep=""))
        if (rem > 1) {
            print.noquote(paste(rem,"other repositories known.  Use verbose=TRUE to see their information"))
        }
    })


if (is.null(getGeneric("repReleaseLevel")))
    setGeneric("repReleaseLevel", function(object)
               standardGeneric("repReleaseLevel"))

setMethod("repReleaseLevel", "replisting", function(object) {
    mtrx <- replisting(object)

    if ("reprellevel" %in% colnames(mtrx))
        mtrx[1,"reprellevel"]
    else
        "release"
})

setMethod("show","replisting",function(object) {
    rL <- replisting(object)
    nreps <- nrow(rL)
    cat("Repository Listing:\n")
    for(i in seq(along=nreps)) {
        cat("\tRepository: ", rL[i, "repname"], "\n")
        cat("\t\t Type: ", rL[i, "reptype"], "\n")
    }
    return(NULL)})


    setMethod("replisting", "replisting", function(object)
              object@replisting)

    setMethod("repURL","replisting",
              function(object, pos) {
                  if (missing(pos))
                      pos <- 1
                  base <- object@replisting[pos,"repaddrBase"]
                  path <- object@replisting[pos,"repaddrPath"]

                  reps <- getReposOption()
                  if (!is.na(reps[base]))
                      base <- as.character(reps[base])

                  url <- paste(base, path, sep="/")
                  return(url)
              })

    if (is.null(getGeneric("numSubReps")))
        setGeneric("numSubReps", function(object)
                   standardGeneric("numSubReps"))
    setMethod("numSubReps", "replisting", function(object)
              (nrow(object@replisting)-1))

    setClass("repdatadesc", representation(repdatadesc="data.frame"))

if (is.null(getGeneric("repdatadesc")))
    setGeneric("repdatadesc", function(object)
               standardGeneric("repdatadesc"))

    setMethod("repdatadesc", "repdatadesc", function(object)
              object@repdatadesc)
    setMethod("summary","repdatadesc",function(object, ...) {
            repD <- repdatadesc(object)
            print(noquote(paste(as.character(nrow(repD)),"packages available")))
        })

    if (is.null(getGeneric("depends")))
        setGeneric("depends", function(object, pkgInfo)
            standardGeneric("depends"))
    setMethod("depends","repdatadesc",function(object, pkgInfo){
        mangName <- getMangledName(pkgName(pkgInfo), pkgVersion(pkgInfo))
        if (mangName %in% rownames(object@repdatadesc)) {
            out <- object@repdatadesc[mangName,]$Depends[[1]]
            out
        }
        else
            return(NULL)
    })

    if (is.null(getGeneric("suggests")))
        setGeneric("suggests", function(object, pkgInfo)
            standardGeneric("suggests"))
    setMethod("suggests","repdatadesc",function(object, pkgInfo){
        mangName <- getMangledName(pkgName(pkgInfo), pkgVersion(pkgInfo))
        if (mangName %in% rownames(object@repdatadesc)) {
            out <- object@repdatadesc[mangName,]$Suggests[[1]]
            out
        }
        else
            return(NULL)
    })

    if (is.null(getGeneric("imports")))
        setGeneric("imports", function(object, pkgInfo)
            standardGeneric("imports"))
    setMethod("imports","repdatadesc",function(object, pkgInfo){
        mangName <- getMangledName(pkgName(pkgInfo), pkgVersion(pkgInfo))
        if (mangName %in% rownames(object@repdatadesc)) {
            out <- object@repdatadesc[mangName,]$Imports[[1]]
            out
        }
        else
            return(NULL)
    })

    if (is.null(getGeneric("keywords")))
        setGeneric("keywords", function(object, pkgInfo)
            standardGeneric("keywords"))
    setMethod("keywords","repdatadesc",function(object, pkgInfo){
        mangName <- getMangledName(pkgName(pkgInfo), pkgVersion(pkgInfo))
        if (mangName %in% rownames(object@repdatadesc)) {
            out <- object@repdatadesc[mangName,]$Keywords[[1]]
            out
        }
        else
            return(NULL)
    })

    setMethod("show","repdatadesc",function(object) {
        repD <- repdatadesc(object)
        if (inherits(repD,"Pkg")) {
            pkgs <- repD$Package
            vers <- unlist(lapply(repD$Version,as.character))
            OSspec <- repD$OSspecific
            types <- unlist(lapply(lapply(OSspec,names),paste,collapse=", "))
            mat <- t(rbind(pkgs,vers,types))
            print.matrix(mat,quote=FALSE,collab=c("Package   ",
                                         "Version    ","Types Available"),
                         rowlab=paste(as.character(1:nrow(repD)),":",sep=""))
        }
        else if (inherits(repD,"Vignette")) {
            vigs <- repD$VignetteIndexEntry
            vigMtrx <- matrix(vigs)
            print.matrix(vigMtrx,quote=FALSE,
                         collab=c("Vignette Name"),
                         rowlab=paste(as.character(1:nrow(repD)),":",sep=""))
        }
    })



    setClass("ReposList", representation(repList="list"))

    if (is.null(getGeneric("repList")))
        setGeneric("repList", function(object)
                   standardGeneric("repList"))
    setMethod("repList", "ReposList", function(object)
              object@repList)

    if (is.null(getGeneric("repList<-")))
        setGeneric("repList<-", function(object, value)
                   standardGeneric("repList<-"))
    setMethod("repList<-", "ReposList", function(object, value) {
        curURLs <- repURLs(object)
        if (is(value,"ReposList")) {
            newURLs <- repURLs(value)
            goodURLs <- which(!(newURLs %in% curURLs))
            if (length(goodURLs) > 0) {
                newList <- repList(value)
                object@repList <- c(object@repList,newList[goodURLs])
            }
        }
        else if (is(value,"ReposEntry")){
            if (!(repURL(value) %in% curURLs))
                object@repList[[length(object@repList)+1]] <- value
        }
        else if (is.list(value)){
            newURLs <- repURLs(value)
            goodURLs <- which(!(newURLs %in% curURLs))
            if (length(goodURLs) > 0) {
                object@repList <- c(object@repList,value[goodURLs])
            }
        }
        object
    })

    if (is.null(getGeneric("numReps")))
        setGeneric("numReps", function(object)
                   standardGeneric("numReps"))
    setMethod("numReps", "ReposList", function(object)
              length(object@repList))

    if (is.null(getGeneric("getRepEntry")))
        setGeneric("getRepEntry", function(object, pos)
                   standardGeneric("getRepEntry"))
    setMethod("getRepEntry", "ReposList", function(object, pos)
              object@repList[[pos]])

    if (is.null(getGeneric("repPkgs")))
        setGeneric("repPkgs", function(object)
                   standardGeneric("repPkgs"))
    setMethod("repPkgs", "ReposList", function(object) {
        unique(unlist(lapply(repList(object), repPkgs)))
    })

    if (is.null(getGeneric("repPkgInfoList")))
        setGeneric("repPkgInfoList", function(object, pkgs, type)
                   standardGeneric("repPkgInfoList"))

    setMethod("repPkgInfoList", "ReposList", function(object,pkgs,type) {
        if (missing(type))
            type <- NULL
        if (missing(pkgs))
            pkgs <- NULL
        x <- lapply(repList(object),repPkgInfos,pkgs,type)
        x
    })

    if (is.null(getGeneric("repNames")))
        setGeneric("repNames", function(object,which)
            standardGeneric("repNames"))
    setMethod("repNames", "ReposList", function(object,which) {
        x <- unlist(lapply(repList(object), repName))
        if (!missing(which))
            x <- x[which]
        x
    })

    if (is.null(getGeneric("repURLs")))
        setGeneric("repURLs", function(object,which)
            standardGeneric("repURLs"))
    setMethod("repURLs", "ReposList", function(object,which) {
        x <- unlist(lapply(repList(object), repURL))
        if (!missing(which))
            x <- x[which]
        x
    })

    setMethod("show", "ReposList", function(object)
              print(paste("A ReposList with",numReps(object),
                          "repositories")))

buildReposList <- function(reps, recurse=TRUE, method="auto") {
    repList <- getReposList(reps, recurse=TRUE, method=method)
    return(new("ReposList",repList=repList))
}

getReposList <- function(reps, recurse=TRUE, method="auto") {
    ## Will generate a ReposList object out of a set of reposiotires
    outList <- new("ReposList", repList=list())
    if (is(reps,"ReposList"))
        reps <- repList(reps)

    for (rep in reps) {
        if (hasFiles(rep))
            repList(outList) <- rep
        if (recurse == TRUE) {
            subs <- numSubReps(rep)
            if (subs > 0)
                repList(outList) <- getReposList(getSubRepList(rep, method=method),
                                                 recurse, method=method)
        }
    }
    return(outList)
}


    setClass("ReposEntry",representation(replisting="replisting",
                                         repdatadesc="repdatadesc",
                                         reposThemes="list"))

    if (is.null(getGeneric("replisting")))
        setGeneric("replisting", function(object)
                   standardGeneric("replisting"))

    if (is.null(getGeneric("repdatadesc")))
        setGeneric("repdatadesc", function(object)
                   standardGeneric("repdatadesc"))
    if (is.null(getGeneric("repURL")))
        setGeneric("repURL", function(object,pos)
            standardGeneric("repURL"))

    if (is.null(getGeneric("repType")))
        setGeneric("repType", function(object)
                   standardGeneric("repType"))
    if (is.null(getGeneric("repName")))
        setGeneric("repName", function(object)
                   standardGeneric("repName"))

if (is.null(getGeneric("reposThemes")))
    setGeneric("reposThemes", function(object)
               standardGeneric("reposThemes"))

################
## Basic slot gets, and info straight from replisting
################
setMethod("reposThemes", "ReposEntry", function(object)
          object@reposThemes)

    setMethod("repType", "ReposEntry", function(object)
              (object@replisting)@replisting[1,"reptype"])

    setMethod("replisting", "ReposEntry", function(object)
              object@replisting)
    setMethod("repdatadesc", "ReposEntry", function(object)
              object@repdatadesc)

    setMethod("repName", "ReposEntry", function(object)
              object@replisting@replisting[1,"repname"])

    setMethod("repURL", "ReposEntry", function(object,pos) {
        if (missing(pos))
            pos <- 1
        repURL(replisting(object),pos)
        })

    setGeneric("repdataframe", function(object)
         standardGeneric("repdataframe"))

    setMethod("repdataframe", "ReposEntry", function(object)
        repdatadesc(repdatadesc(object)))

    ####################
    ####################

    ############
    ## Subrepository management
    ############
    if(is.null(getGeneric("numSubReps")))
        setGeneric("numSubReps", function(object)
                   standardGeneric("numSubReps"))
    setMethod("numSubReps", "ReposEntry", function(object)
              numSubReps(replisting(object)))

    if(is.null(getGeneric("getSubRep")))
        setGeneric("getSubRep", function(object,pos,method)
                   standardGeneric("getSubRep"))
    setMethod("getSubRep", "ReposEntry", function(object,pos,method) {
        if (pos > numSubReps(object))
            return(NULL)
        else
            return(getReposEntry(repURL(object,pos+1), method=method))
        })
    if (is.null(getGeneric("getSubRepList")))
        setGeneric("getSubRepList", function(object, method)
                   standardGeneric("getSubRepList"))
    setMethod("getSubRepList", "ReposEntry", function(object, method) {
        num <- numSubReps(object)
        out <- list()
        if (num > 0) {
            for (i in 1:num) {
                out[[length(out)+1]] <-
                    getReposEntry(repURL(object,i+1), method=method)
            }
        }
        outObj <- new("ReposList",repList=out)
        return(outObj)
    })
    #####################
    #####################

    #####################
    ## Dataframe information
    ######################
        if (is.null(getGeneric("depends")))
        setGeneric("depends", function(object, pkgInfo)
            standardGeneric("depends"))
    setMethod("depends","ReposEntry",function(object, pkgInfo){
        out <- depends(object@repdatadesc, pkgInfo)
        out
    })

    if (is.null(getGeneric("suggests")))
        setGeneric("suggests", function(object, pkgInfo)
            standardGeneric("suggests"))
    setMethod("suggests","ReposEntry",function(object, pkgInfo){
        out <- suggests(object@repdatadesc, pkgInfo)
        out
    })

    if (is.null(getGeneric("imports")))
        setGeneric("imports", function(object, pkgInfo)
            standardGeneric("imports"))
    setMethod("imports","ReposEntry",function(object, pkgInfo){
         out <- imports(object@repdatadesc, pkgInfo)
        out
    })

    if (is.null(getGeneric("keywords")))
        setGeneric("keywords", function(object, pkgInfo)
            standardGeneric("keywords"))
    setMethod("keywords","ReposEntry",function(object, pkgInfo){
         out <- keywords(object@repdatadesc, pkgInfo)
         out
     })

    ######################
    ## Methods to for file manipulation & file information
    ######################
    if(is.null(getGeneric("hasFiles")))
        setGeneric("hasFiles", function(object)
                   standardGeneric("hasFiles"))
    ## hasFiles will determine if the repository has any files for download
    setMethod("hasFiles","ReposEntry", function(object) {
        if (nrow(repdataframe(object)) < 1)
            return(FALSE)
        else
            return(TRUE)
    })

    if (is.null(getGeneric("pkgRVersion")))
        setGeneric("pkgRVersion", function(object, pkg, vers, type)
                   standardGeneric("pkgRVersion"))
    setMethod("pkgRVersion", "ReposEntry", function(object, pkg, vers,
                                                    type)
          {
              z <- repdataframe(object)
              rowN <- getMangledName(pkg, vers)
              if (rowN %in% rownames(z)) {
                  os <- z[rowN,]$OSspecific[[1]][[type]]$Rvers
                  if ((is.null(os))||(os==""))
                      return(NULL)
                  else
                      return(buildVersionNumber(os))
              }
              else
                  return(NULL)
          })

    if (is.null(getGeneric("repPkgs")))
        setGeneric("repPkgs", function(object)
                   standardGeneric("repPkgs"))
    setMethod("repPkgs", "ReposEntry", function(object) {
        repD <- repdataframe(object)
        repD$Package
    })

    if( is.null(getGeneric("repObjects")))
        setGeneric("repObjects", function(object,pkgs,type,files)
            standardGeneric("repObjects"))
    setMethod("repObjects", "ReposEntry", function(object,pkgs=NULL,
                                                   type,files){
	repD <- repdataframe(object)
        if (nrow(repD) == 0)
            return(NULL)
        if (missing(pkgs))
            pkgs <- NULL
        if (!is.null(pkgs)) {
            whichPkgs <- which(repD$Package %in% pkgs)
            contPkgs <- unlist(lapply(repD$Contains,
                                                function(x,y) {
                                                    any(y %in% x)},
                                                pkgs))
            if (length(contPkgs) > 0)
                whichPkgs <- unique(c(whichPkgs, which(contPkgs)))
        }
        else
            whichPkgs <- 1:nrow(repD)


        if (length(whichPkgs) > 0) {
            repD <- repD[whichPkgs,,drop=FALSE]

            pkg <- repD$Package
            vers <- unlist(lapply(repD$Version,as.character))
            relLevel <- repD$ReleaseLevel
            defRepLevel <- repReleaseLevel(replisting(object))
            if (is.null(relLevel))
                relLevel <- rep(defRepLevel, length(pkg))
            else
                relLevel <- sapply(relLevel, function(x,y)
                    if (x == "NA") {y} else {x} , defRepLevel,
                                   USE.NAMES=FALSE)

            mat <- t(rbind(pkg,vers, relLevel))
            if ((!missing(type))&&(!is.null(type))) {
                OSspec <- repD$OSspecific
                types <-
                    unlist(lapply(lapply(OSspec,names),paste,collapse=", "))
                types <- strsplit(types,", ")
                rightType <- NULL
                for (i in 1:length(types)) {
                    if (type %in% types[[i]])
                        rightType <- c(rightType,i)
                }
                mat <- mat[rightType,,drop=FALSE]
                if (nrow(mat) == 0)
                    mat <- NULL
                else {
                    if (missing(files))
                        files <- FALSE
                    if (files == TRUE) {
                        ## Get file names
                        OSspec <- OSspec[rightType]
                        tmpMat <- matrix(nrow=nrow(mat),ncol=1)
                        for (i in 1:nrow(mat)) {
                            tmpMat[i,1] <-
                                basename(OSspec[[i]][type][[1]]$File)
                        }
                        mat <- cbind(mat, tmpMat)
                        colnames(mat)[4] <- "filenames"
                    }
                }
            }
            return(mat)
        }
        else return(NULL)
    })

    if (is.null(getGeneric("repPkgInfos")))
        setGeneric("repPkgInfos", function(object,pkgs,type)
                   standardGeneric("repPkgInfos"))
    setMethod("repPkgInfos", "ReposEntry", function(object,pkgs=NULL,type){
        if (missing(type))
            type <- NULL
        if (missing(pkgs))
            pkgs <- NULL

        ro <- repObjects(object,pkgs=pkgs,type=type)
        repD <- repdataframe(object)

        if (is.null(ro))
            return(ro)
        out <- list()
        if (nrow(ro) > 0) {
            for (i in 1:nrow(ro)) {
                ## FIXME: Currently need some backwards compatability
                pkgRow <- which(repD[,"Package"] == ro[i,"pkg"])
                if (length(pkgRow) > 0) {
                    conts <- character()
                    if ("Contains" %in% colnames(repD)) {
                        conts <- repD[pkgRow,"Contains"][[1]]
                    }

                    out[[i]] <- buildPkgInfo(ro[i,"pkg"],ro[i,"vers"],
                                             relLevel=ro[i,"relLevel"],
                                             contains=conts)
                }
            }
        }
        return(out)
    })

    if (is.null(getGeneric("downloadFile")))
        setGeneric("downloadFile", function(object, file, vers, type,
                                            dest=NULL,method="auto")
                   standardGeneric("downloadFile"))
    setMethod("downloadFile", "ReposEntry", function(object, file,
                                                     vers, type,
                                                     dest=NULL,
                                                     method="auto") {

        if (missing(file)) {
            warning("Can not continue without a filename to download")
            return(NULL)
        }
        if (missing(type)) {
            warning("Can not continue without a 'type' to download")
            return(NULL)
        }
        if ((missing(vers)) && (type != "vignette")) {
            warning("Can not continue without a version of the file")
            return(NULL)
        }
        errMsg <- paste("There is no",type,"version of package",
                        file,"available at",repURL(object))
        ## Look up pkg/vers in DF
        rd <- repdataframe(object)

        print(paste("Attempting to download",file,"from",repURL(object)))

        if (repType(object) == "package") {
            rowName <- getMangledName(file,vers)
            ## try() doesn't seem to stop it from failing when
            ## indexing the rowname, so check first
            if (! rowName %in% rownames(rd))
                return(NULL)
            row <- rd[rowName,]
            ## get type
            osS <- row$OSspecific[[1]]
            if (is.null(osS))
                return(NULL)
            curOS <- which(names(osS) %in% type)
            if (length(curOS) > 0) {
                filePath <- osS[[curOS]]$File
                if (is.null(filePath)) {
                    warning(errMsg)
                    return(NULL)
                }
            }
            else {
                warning(errMsg)
                return(NULL)
            }
        }
        else if (repType(object) == "vignette") {
            vigRow <- which(rd$VignetteIndexEntry %in% vig)[1]
            filePath <- rd[vigRow,]$PDFpath
            if ((is.na(filePath))||(is.null(filePath))) {
                warning(paste("Vignette",file,"not found at",
                        repURL(object)))
                return(NULL)
            }
        }
        ## download to dest (NULL is temp)
        fileURL <- paste(repURL(object),filePath, sep="/")
        if (is.null(dest))
            dest <- tempdir()

        destFile <- file.path(dest,basename(filePath))
        download.file(fileURL, destFile, mode="wb", quiet=TRUE,
                      method=method)
        print("Download complete.")
        ## return filename (or NULL on error)
        return(destFile)
    })



setMethod("repReleaseLevel", "ReposEntry", function(object) {
    repReleaseLevel(replisting(object))
})

setMethod("summary","ReposEntry",function(object, ...){
    cat("Repository Information:\n")
    summary(replisting(object))
    cat("\n")
    summary(repdatadesc(object))
})

setMethod("show","ReposEntry",function(object){
    cat("Repository Information:\n")
    show(replisting(object))
    cat("\n")
    show(repdatadesc(object))
    if (length(reposThemes(object)) == 0)
       cat("No themes registered with this repository\n")
   else {
        cat("\nThe following themes are registered with this repository:\n")
        for (i in 1:length(reposThemes(object)))
            show(reposThemes(object)[[i]])
    }
})

#############

is.ReposEntry <- function(x) {
    if (is(x, "ReposEntry"))
        return(TRUE)
    else
        return(FALSE)
}

buildReposEntry <- function(replisting, repdatadesc,
                            repTheme=list()) {
    a <- new("ReposEntry", replisting=replisting,
             repdatadesc=repdatadesc, reposThemes=repTheme)
    if (is.null(a))
        warning("NULL ReposEntry object.  Check URL")
    return(a)
}


    setClass("pkgListing", representation(pkgList="list",
                                          repList="ReposList"))

    if (is.null(getGeneric("pkgList")))
        setGeneric("pkgList", function(object)
                   standardGeneric("pkgList"))
    setMethod("pkgList", "pkgListing", function(object)
              object@pkgList)

    if (is.null(getGeneric("repList")))
        setGeneric("repList", function(object)
                   standardGeneric("repList"))
    setMethod("repList", "pkgListing", function(object)
              object@repList)

    if (is.null(getGeneric("repListing")))
        setGeneric("repListing", function(object)
                   standardGeneric("repListing"))
    setMethod("repListing", "pkgListing", function(object)
              repList(object@repList))

    if (is.null(getGeneric("getRepEntry")))
        setGeneric("getRepEntry", function(object, pos)
                   standardGeneric("getRepEntry"))
    setMethod("getRepEntry", "pkgListing", function(object, pos)
              getRepEntry(object@repList,pos))

if (is.null(getGeneric("getRepURL")))
    setGeneric("getRepURL", function(object, pos)
               standardGeneric("getRepURL"))
setMethod("getRepURL", "pkgListing", function(object, pos)
          repURLs(repList(object))[pos])



    if (is.null(getGeneric("packages")))
        setGeneric("packages", function(object)
                   standardGeneric("packages"))
    setMethod("packages", "pkgListing", function(object)
              names(object@pkgList))


    if (is.null(getGeneric("pkgVersionList")))
        setGeneric("pkgVersionList", function(object, pkg, names)
                   standardGeneric("pkgVersionList"))
    setMethod("pkgVersionList", "pkgListing", function(object, pkg,
                                                       names) {
        if (missing(names))
            names <- "index"

        pList <- object@pkgList[[pkg]]
        if (length(pList) == 0)
            return(pList)

        curNames <-  names(pList)
        names(pList) <- switch(names,
                              "names"=repNames(repList(object),as.numeric(curNames)),
                              "urls"=repURLs(repList(object),as.numeric(curNames)),
                              "index"=curNames,
                               stop(paste(names,
                                          "is not valid for option names:",
                                          "please choose between\n 'names',",
                                          "'urls', and 'index'."))
                               )
        pList
    })


    setMethod("summary","pkgListing",function(object, ...){
        cat("A package listing with the following packages:\n")
        print(packages(object))
        cat("\nFrom:\n")
        show(repList(object))
    })

    setMethod("show","pkgListing",function(object) {
        pL <- pkgList(object)
        if (length(pL) == 0)
            return(NULL)
        out <- matrix(ncol=3)
        pkgs <- names(pL)
        for (i in 1:length(pL)) {
            if (length(pL[[i]]) > 0) {
                whichREs <- as.numeric(names(pL[[i]]))
                outURLs <- repURLs(repList(object), whichREs)
                for (j in 1:length(pL[[i]])) {
                    if (length(pL[[i]][[j]]) > 0) {
                        for (k in 1:length(pL[[i]][[j]])) {
                            out <- rbind(out,c(pkgs[i],
                                               stringRep(pL[[i]][[j]][[k]]),
                                               outURLs[j]))
                        }
                    }
                }
            }
        }
        print.matrix(out[2:nrow(out),,drop=FALSE],quote=FALSE,
                     collab=c("Package   ", "Version   ","Repository URL"),
                     rowlab=paste(as.character(1:nrow(out)),":",sep=""))
    })

installPkgListing <- function(pkgList, lib, type, method="auto",
                              versForce=TRUE, force=FALSE,
                              searchOptions=TRUE, getAllDeps=FALSE,
                              getNewest=TRUE) {
    pkgs <- packages(pkgList)
    if (length(pkgs) == 0)
        return(NULL)

    if (missing(lib))
        stop("Argument lib is missing")
    if (missing(type))
        stop("Argument type is missing")

    pkgStats <- new("pkgStatusList",statusList=list())

    curPkgList <- pkgList(pkgList)
    for (i in 1:length(curPkgList)) {
        curPkg <- curPkgList[[i]]
        if (length(curPkg) == 0)
            next
        reps <- names(curPkg)

        gotPkgs <- list()
        for (j in 1:length(curPkg)) {
            ## Check dependencies, slightly different then
            ## in baseFileSelect because we need to only
            ## check off pkgInfos from resolve.depends that
            ## have the *Same* version as the ones specified
            curPkgInfo <- buildPkgInfo(pkgs[i], curPkg[[j]][[1]])
            ## No need to get this package if we already acquired
            ## it via dependency installs
            if (any(unlist(lapply(gotPkgs,"==",curPkgInfo))))
                next

            repEntry <- getRepEntry(pkgList, as.numeric(reps[j]))
            rd <- resolve.depends(curPkgInfo, repEntry, force,
                                  lib=lib,
                                  searchOptions=searchOptions,
                                  getAllDeps=getAllDeps,
                                  versForce=versForce,
                                  getNewest=getNewest)
            rdPkgs <- get("pkgList", rd)
            if (length(packages(rdPkgs)) > 0) {
                pIs <- getPkgStatusListInfos(rdPkgs)
                pIs <- pIs[updated(rdPkgs)]
                gotPkgs <- c(gotPkgs, pIs)
            }
            if (get("halt",rd) == FALSE) {
                ret <- downAndInPkg(pkgName(curPkgInfo), repEntry,
                                    stringRep(pkgVersion(curPkgInfo)),
                                    lib=lib, type=type, method=method,
                                    versForce=versForce)
            }
            else
                ret <- FALSE

            statusList(pkgStats) <- new("pkgStatus",
                                        package=pkgs[i],
                                        found=TRUE,
                                        updated=ret,
                                        url=repURL(repEntry),
                                        pkgVersion=curPkg[[j]][[1]])
        }
    }
    return(pkgStats)
}

buildPkgListing <- function(repList, pkgs=NULL, type=NULL,
                            develOK=FALSE,
                            method="auto") {
    if (!(is(repList,"ReposList")) && (is.list(repList)))
        repList <- buildReposList(repList, method=method)

    if (!is(repList,"ReposList"))
        return(NULL)

    ## pkgs==NULL implies "all packages"
    if (numReps(repList)<1)
        return(NULL)

    if (is.null(pkgs)) {
        pkgs <- repPkgs(repList)
        if (is.null(pkgs))
            return(NULL)
    }

    ## We can have either a vector of pkg names, a list of pkg names,
    ## or a list of pkgInfos.  In any case, make sure that we have a
    ## vector of pkg names
    hasPkgInfos <- FALSE
    if (is.list(pkgs)) {
        pkgVec <- unique(unlist(lapply(pkgs, function(x) {
            if (is(x,"pkgInfo")) {
                hasPkgInfos <<- TRUE
                return(pkgName(x))
            }
            else if (is.character(x))
                return(x)
            else
                stop("Malformed 'pkgs' parameter")
        })))
    }
    else
        pkgVec <- pkgs


    ## Now for each repository, get any appropriate pkg info
    repPIs <- repPkgInfoList(repList, pkgVec, type)

    ## Create skeleton for pkgList
    pkgList <- vector(mode="list",length=length(pkgVec))
    for (i in 1:length(pkgList))
        pkgList[[i]] <- list()


    ## !! There has to be a better way here then a double for loop
    for (j in 1:numReps(repList)) {
        if (!is.null(repPIs[[j]])) {
            if (hasPkgInfos) {
                badPack <- numeric()
                for (k in 1:length(repPIs[[j]])) {
                    ## Need to see if this pkgInfo is in the
                    ## list of what we're looking for
                    if (!any(unlist(lapply(pkgs, "==",
                                           repPIs[[j]][[k]]))))
                        badPack <- c(badPack, k)
                 }
                if (length(badPack) > 0)
                    repPIs[[j]] <- repPIs[[j]][-badPack]
                if (length(repPIs[[j]]) == 0)
                    next
            }

            pkgN <- unlist(lapply(repPIs[[j]],pkgName))
            pkgV <- lapply(repPIs[[j]],pkgVersion)

            for (i in seq(along=pkgVec)) {
                goodBundle <- unlist(lapply(repPIs[[j]], function(x,y)
                                      y %in% pkgContains(x),
                                      pkgVec[i]))
                if (any(goodBundle))
                    pkgVec[i] <- pkgName(repPIs[[j]][[which(goodBundle)]])
            }


            goodPkg <- which(pkgN %in% pkgVec)


            ## Try to find alternate packages

            develPkgs <- list()
            repPkgNames <- unlist(lapply(repPIs[[j]], pkgName))
            for (k in seq(along = goodPkg)) {
                if ((develOK == FALSE)&&
                    (pkgRelLevel(repPIs[[j]][[match(pkgN[k],repPkgNames)]]) == "devel")) {
                    develPkgs <- c(develPkgs,
                                   paste(pkgN[k], "version",
                                         pkgV[[k]]))
                }
                else {
                    pakLoc <- match(pkgN[k], pkgVec)
                    len <- length(pkgList[[pakLoc]])
                    pkgList[[pakLoc]][[len+1]] <- pkgV[k]
                    names(pkgList[[pakLoc]])[len+1] <- as.character(j)
                }
            }
            if ((numReps(repList) == 1)&&(length(develPkgs) > 0)) {
                note(paste("The following packages were found at " ,
                           repURLs(repList)[j], ", but were marked",
                           " as developmental software:\n",
                           paste(develPkgs,collapse="\n"),
                           "\nTo allow reposTools to access these packages,",
                           " use the 'develOK=TRUE' argument.\n",
                           sep=""))
            }
        }
    }

    names(pkgList) <- pkgVec
    pkgList <- pkgList[sapply(pkgList,
                              function(x){if (length(x) > 0){
                                  TRUE
                              }
                              else {
                                  FALSE
                              }})]

    return(new("pkgListing",pkgList=pkgList,repList=repList))
}
