/* editing.c - Routines for editing the renames.
 *
 * Copyright (C) 2001  Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif
/* C89 */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
/* POSIX/gnulib */
#include <stdbool.h>
/* gettext */
#include <gettext.h> /* will include <libintl.h> if ENABLE_NLS */
#define _(String) gettext(String)
/* common */
#include "common/memory.h"
#include "common/string-utils.h"
#include "common/io-utils.h"
#include "common/error.h"
#include "common/llist.h"
/* qmv */
#include "qmv.h"

static EditFormat *formats[] = {
	&destination_only_format,
	&dual_column_format,
	&single_column_format
};

/**
 * Find an edit-format by its long or short name.
 * @returns NULL if no format was found.
 */
EditFormat *
find_edit_format_by_name(const char *name)
{
	int c;

	for (c = 0; c < sizeof(formats)/sizeof(*formats); c++) {
		if (strcmp(name, formats[c]->name) == 0
				|| strcmp(name, formats[c]->short_name) == 0)
			return formats[c];
	}

	return NULL;
}

/**
 * A readline completion generator for the various edit
 * formats.
 */
char *
edit_format_generator(const char *text, int state)
{
	static int c;

	if (state == 0)
		c = 0;

	while (c < sizeof(formats)/sizeof(*formats)) {
		char *name = formats[c]->name;
		c++;
		if (starts_with(name, text))
			return xstrdup(name);
	}

	return NULL;
}

void
edit_command(char **args)
{
	if (llist_is_empty(renames)) {
		printf(_("no files to rename - use `list'\n"));
	} else {
		bool all = (args[1] != NULL && strcmp(args[1], "all") == 0);
		edit_renames(all);
	}
}

bool
edit_renames(bool all)
{
	FILE *file;
	char *cmd;
	int rc;
	LList *edit_renames = llist_new();
	Iterator *it;

	/* Prepare the edit-file */
	file = fopen(edit_filename, "w");
	if (file == NULL) {
		warn_errno(_("cannot create %s"), edit_filename);
		return false;
	}

	/* Determine what files we want to display to user */
	if (!all) {
		for (it = llist_iterator(renames); iterator_has_next(it); ) {
			FileSpec *spec = iterator_next(it);

			if (spec->renamed == RENAME_FAILED) {
				llist_add(edit_renames, spec);
				continue;
			}

			switch (spec->status) {
			case STATUS_NO_CHANGE:
			case STATUS_RENAME:
			case STATUS_CIRCULAR:
			case STATUS_COMPLETED:
				break;
			default:
				llist_add(edit_renames, spec);
			}
		}
		iterator_free(it);
	}
	if (llist_is_empty(edit_renames)) {
		for (it = llist_iterator(renames); iterator_has_next(it); ) {
			FileSpec *spec = iterator_next(it);
			if (spec->renamed != RENAME_COMPLETE)
				llist_add(edit_renames, spec);
		}
		iterator_free(it);
	}

	/* Write renames to the output file */
	format->output(file, edit_renames);
	if (fclose(file) != 0) {
		warn_errno(_("cannot close %s"), file);
		llist_free(edit_renames);
		return false;
	}

	/* Start the editor */
	cmd = xasprintf("%s %s", editor_program, edit_filename);
	/* Fortunately, edit_filename usually does not contain
	 * special characters and does not start with a dash. */
	rc = system(cmd);
	free(cmd);
	if (rc == -1) {
		warn_errno(_("cannot start editor `%s'"), editor_program);
		llist_free(edit_renames);
		return false;
	}
	if (rc != 0) {
		warn(_("`%s' exited with non-zero status"), editor_program);
		llist_free(edit_renames);
		return false;
	}

	/* Open the file */
	file = fopen(edit_filename, "r");
	if (file == NULL) {
		warn_errno(_("cannot open %s for reading"), edit_filename);
		llist_free(edit_renames);
		return false;
	}

	/* Read the file */
	if (!format->input(file, edit_renames)) {
		/* FIXME: we're in a weird state here, fix somehow */
		llist_free(edit_renames);
		return false;
	}
	if (fclose(file) != 0) {
		warn_errno(_("cannot close %s"), edit_filename);
		return false;
	}

	/* Reset status of *all* renames (except those already completed). */
	for (it = llist_iterator(renames); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		if (spec->renamed != RENAME_COMPLETE) {
			spec->status = STATUS_NONE;
			spec->renamed = RENAME_UNDONE;
		}
	}
	iterator_free(it);
	llist_free(edit_renames);

	if (plan != NULL)
		free_plan(plan);

	plan = make_plan(renames);
	display_plan(plan, false);
	return true;
}
