/*
 * CCameraPathInterpolator.h
 * $Id: 
 *
 * Copyright (C) 2001 Thomas Woerner, Michael Meissner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Class CCameraPathInterpolator
//  Purpose     : Provides funcionality



#ifndef CCAMERA_PATH_INTERPOLATOR_H
#define CCAMERA_PATH_INTERPOLATOR_H


// Own
////////
#include "CList.h"
#include "CCameraKeyPathPoint.h"



/** Calculates a camera path out of the given
  * key path. There a different possible types
  * of interpolation algorithms to use defined
  * as ShapeTypes.
  *
  * @author Thomas Woerner, Michael Meissner
  *
  * [tw] 2001/04/28 initial version
  * [tw] 2001/05/01 added Catmull-Rom interpolation
  */
class CCameraPathInterpolator {

   public:
    
      enum ShapeType { SHAPE_LINE = 0,
                       SHAPE_TCB = 1,
                       SHAPE_CATMULL_ROM = 2 };

      enum PathType { PATH_OPEN = 0,
                      PATH_CLOSED = 1 };

      /** Constructs new Interpolator and computes path. */
      CCameraPathInterpolator(CList<CCameraKeyPathPoint> path,
                              ShapeType nShapeType = SHAPE_LINE, 
                              PathType nPathType = PATH_OPEN)
         : keys(path) {
         m_nShapeType = nShapeType;
         m_nPathType  = nPathType;
         compute();
      }

      /** Destructs (default). */
      ~CCameraPathInterpolator() {}
    
      /** Returns total number of frames of computed path. */
      int getNumFrames() {
         return m_ShapeList.getNumObjects();
      }
    
       /** Returns frame of computed path.
        * frame in [1..getNumFrames()] */
       CCameraKeyPathPoint *getFrame(int nFrame) {
          if (nFrame >= 0 && nFrame < getNumFrames())
             return &m_ShapeList[nFrame];
	  else
	     return NULL;
       }
    
      /** Returns computed path. */
      CList<CCameraKeyPathPoint> getPath() {
         return m_ShapeList;
      }
    
   protected:
    
      /** Computes path. */
      void compute();
    
      /** Input path (sampling points). */
      CList<CCameraKeyPathPoint> keys;
    
      /** Shape type: linear / tcb. */
      ShapeType m_nShapeType;
    
      /** Path type: open / closed. */
      PathType m_nPathType;
    
      /** Computed output path. */
      CList<CCameraKeyPathPoint> m_ShapeList;

      /** Adds two CCameras.
       * Copies first CCamera, adds
       * (eye, ref point, view up, ratio, fovy, vpheight) */
      CCamera add(const CCamera &c1, const CCamera &c2) const {
         CCamera cam(c1);
         cam.setEyePos(c1.getEyePos() + c2.getEyePos());
         cam.setRefPoint(c1.getRefPoint() + c2.getRefPoint());
         cam.setViewUp(c1.getViewUp() + c2.getViewUp());
         cam.setRatio(c1.getRatio() + c2.getRatio());
         cam.setFovy(c1.getFovy() + c2.getFovy());
         cam.setVPHeight(c1.getVPHeight() + c2.getVPHeight());

         return cam;
      }

      /** Subtracts two CCameras.
       * Copies first CCamera, subtracts
       * (eye, ref point, view up, ratio, fovy, vpheight) */
      CCamera sub(const CCamera &c1, const CCamera &c2) const {
         CCamera cam(c1);
         cam.setEyePos(c1.getEyePos() + c2.getEyePos() * -1);
         cam.setRefPoint(c1.getRefPoint() + c2.getRefPoint() * -1);
         cam.setViewUp(c1.getViewUp() + c2.getViewUp() * -1);
         cam.setRatio(c1.getRatio() - c2.getRatio());
         cam.setFovy(c1.getFovy() - c2.getFovy());
         cam.setVPHeight(c1.getVPHeight() - c2.getVPHeight());

         return cam;
      }

      /** Multiplies CCamera with double value 
       * Copies CCamera, multiplies
       * (eye, ref point, view up, ratio, fovy, vpheight) */
      CCamera mul(const CCamera &c, const double d) const {
         CCamera cam(c);
         cam.setEyePos(c.getEyePos() * d);
         cam.setRefPoint(c.getRefPoint() * d);
         cam.setViewUp(c.getViewUp() * d);
         cam.setRatio(c.getRatio() * d);
         cam.setFovy(c.getFovy() * d);
         cam.setVPHeight((int)((double) c.getVPHeight() * d));

         return cam;
      }
};

#endif /* CCAMERA_PATH_INTERPOLATOR_H */
