/* primax_scan -- linux driver for Primax scanners

   Authors:
   Marco Foglia <Marco.Foglia@switzerland.org>
   Thomas Schano <schano@t-online.de>
   Christian Ordig <chr.ordig@gmx.net>

   Copyright (C) 1999 Marco Foglia, Thomas Schano, Christian Ordig

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <unistd.h>

#include "primax_scan.h"
#include "LM9811.h"
#include "lp.h"
#include "tables.h"

/*  The registers of the LM9811CVV represent the interface of the diffrent */
/*  CCDs. So we try to use it to detect the type of the scanner. This values */
/*  can not be used to get the number of pixels in one line! The registers */
/*  0x00 to 0x04 are specific to the CCD. The registers 0x05 to 0x07 are */
/*  used for calibration and error-correction. Datasheet at www.national.com */

static unsigned char adc_register[3][8] = {
    { 0x0A, 0xE8, 0xFE, 0x8E, 0x31, 0x70, 0x00, 0xF0}, /* Primax Colorado
                                                          Direct */
    { 0x1A, 0xE8, 0xFE, 0x8E, 0x31, 0x00, 0x00, 0x00}, /* Primax .... */
    { 0x1C, 0xEA, 0xFE, 0x82, 0x3D, 0x00, 0x00, 0x00}  /* Primax .... */
};


void calibrate_LM9811(scan_image *image);
int write_LM9811(unsigned char value, int register_nr);

static int read_LM9811(unsigned char *data, int register_nr );
static int init_LM9811(void);
// static void printdata(char *data, int height, int width);




/*-------------------------------------------------------------*/
/*
 * reads registers of the LM9811CCV
 * Datasheet at  http://www.national.com
 *
 * ( 0 <= register_nr <= 7 )
 */

static int read_LM9811(unsigned char *data, int register_nr )
{
    int status;
    
    status = epp_write_one(register_nr % 8, REGISTER_LM9811);
    status |= epp_read_one(data, REGISTER_LM9811);
    DBG(DBG_MEDIUM,"read_LM9811:   reg: %i  val: %02x\n", 
	register_nr , data[0]);
    return status;
}

/*-------------------------------------------------------------*/
/*
 * writes registers of the LM9811CCV
 *
 * ( 0 <= register_nr <= 7 )
 */
int write_LM9811(unsigned char value, int register_nr)
{
    int status;

    status = epp_write_one(register_nr % 8, REGISTER_LM9811);
    status |= epp_write_one(value, REGISTER_LM9811);
    DBG(DBG_MEDIUM, "write_LM9811:  reg: %i  val: %02x \n", 
	register_nr, value);
    return status;
}

/*-------------------------------------------------------------*/
/* 
 * Initializes the adc of the scanner
 *
 * values 1-4 seem to be static
 */

static int init_LM9811(void){
    
    int i,  status = NO_TIMEOUT;
    unsigned char reg[8];

    for (i = 0; i < 8; i++) 
	read_LM9811(&reg[i], i);
    
    for (i = 0; i < 8; i++) {
	/* FIX THIS IF MORE TYPES ARE SUPPORTED */
	if (adc_register[0][i] != reg[i])
	    DBG(DBG_MEDIUM, "init_adc:  reg: %i %02x -> %02x\n", 
		   i, reg[i], adc_register[0][i]);
	write_LM9811(adc_register[0][i], (char) i);
    }
    
    for (i = 0; i < 8; i++)
	status|=read_LM9811(&reg[i], i);
    DBG(DBG_MEDIUM, "init_adc: done\n");
    return status;
}

void printdata(char *data, int height, int width){
    int x, y;
    
    printf("new dataset\n");
    for (y = 0; y < height ; y++){
         printf("line %d\n",y);
        for (x = 0; x < width ; x++)
            printf("%2X ",0xff & data[ x + y * width ]);
        printf("\n");
    }
}



void calibrate_LM9811(scan_image *image) {

    int line_count, pix_count;
    int max_pix;

    unsigned char *line;
    unsigned long *avg;
    unsigned char *count;
    unsigned char *data;
    
    char *PGA, *VGA, *DAC;

    int V_adc1 = 0xff, V_adc2 = 0;
    int V_os1;

    char VGA_flag = 0, VGA_flag_old;
    long akku;
    
    scan_image *calibration_image;


    init_image( &calibration_image);
    init_LM9811();
    

    PGA = &(calibration_image->LM9811_reg_PGA);
    VGA = &(calibration_image->LM9811_reg_VGA);
    DAC = &(calibration_image->LM9811_reg_DAC);
    

    calibration_image->width = x_table_size;
    calibration_image->height = (int) Y_RESOLUTION * CALIBRATION_SCAN_SIZE;
    calibration_image->resolution = X_RESOLUTION;
    calibration_image->speed = image->speed;
    calibration_image->from_top =  CALIBRATION_BLACK_ARRAY_START;
    calibration_image->from_left = 0;
    calibration_image->num_colors = 3;
    calibration_image->colormode = RGB_MODE;
    max_pix =  calibration_image->width * calibration_image->num_colors;
    calibration_image->data = (char*) malloc(sizeof(char) *
                                             calibration_image->height *
                                             max_pix);
    line = malloc(sizeof(char) * max_pix);
    avg = malloc(sizeof(unsigned long) * max_pix);
    count = malloc(sizeof(unsigned char) * max_pix);
    data = calibration_image->data;

    // correction of offset (black-value)
    
    *PGA = 0x00;

    //PGA intern, offset-sign pos., operating, 2nd offset 0 LSB
    *VGA = 0x00 | 0x40;

    *DAC = 0xf0;
    
    scan(calibration_image,CALIBRATION);

    for (pix_count = 0; pix_count < max_pix; pix_count++)
        line[pix_count] = 0xff;
    for(pix_count = 0; pix_count < max_pix; pix_count++)
       for(line_count = 0; line_count < calibration_image->height;
           line_count++)
           if((data[line_count * max_pix + pix_count] & 0xff) <
              (line[pix_count] & 0xff))
               line[pix_count] = data[line_count * max_pix + pix_count];
    for (pix_count = 0; pix_count < max_pix; pix_count++)
       if (V_adc1 > (line[pix_count] & 0xFF))
           V_adc1 = (line[pix_count] & 0xFF) ;

    
    *PGA = 0xff;
   
    scan(calibration_image,CALIBRATION);

    for (pix_count = 0; pix_count < max_pix; pix_count++)
        line[pix_count] = 0xff;
    for (pix_count = 0; pix_count < max_pix; pix_count++)
       for (line_count = 0; line_count < calibration_image->height;
            line_count++)
           if ((data[line_count * max_pix + pix_count] & 0xff) <
               (line[pix_count] & 0xff))
               line[pix_count] = data[line_count * max_pix + pix_count];
    for (pix_count = 0; pix_count < max_pix; pix_count++)
       if (V_adc2 < (line[pix_count] & 0xFF))
           V_adc2 = (line[pix_count] & 0xFF) ;

    V_os1 = 15 + 0.3 * (V_adc1 - V_adc2);

    *DAC = (char) (V_os1 << 4) & 0xf0;


 //set offset-sign-bit*/
    if (V_os1 > 0x0f)
        *VGA = *VGA | 0x40;
    else
        *VGA = *VGA & 0xBF;
    
    if ( 3 * V_adc1 < V_adc2)
        *VGA = *VGA & 0xef;
    else
        *VGA = *VGA | 0x10;

//  tuning of gain (white-value)
    
    *PGA = 0x00; 
    calibration_image->from_top =  CALIBRATION_WHITE_ARRAY_START;


    do{
        VGA_flag_old = VGA_flag;

//        printf("\nPGA %x\n",*PGA & 0xff);

//        printf("\nVGA %x\n",*VGA & 0xff);

//        printf("\nDAC %x\n",*DAC & 0xff);

        scan(calibration_image,CALIBRATION);

        for (pix_count = 0; pix_count < max_pix; pix_count++){
            line[pix_count] = 0x01;
            count[pix_count] = 0;
            avg[pix_count] = 0;
        }
        for (pix_count = 0; pix_count < max_pix; pix_count++)
            for (line_count = 0; line_count < calibration_image->height;
                 line_count++)
                if ((data[line_count * max_pix + pix_count] & 0xff) >
                    (line[pix_count] & 0xff))
                    line[pix_count] = data[line_count * max_pix + pix_count];
//        printdata(line , 1 , max_pix);

        for (pix_count = 0; pix_count < max_pix; pix_count++)
            for (line_count = 0; line_count < calibration_image->height;
                 line_count++)
                if ((data[line_count * max_pix + pix_count] & 0xff) >
                    ((line[pix_count] & 0xff) * 0.75 )){
                    avg[pix_count] = avg[pix_count] +
                        (data[line_count * max_pix + pix_count] & 0xff);
                    count[pix_count]++;
                } 
//    printdata(count , 1 , max_pix);
        for (pix_count = 0; pix_count < max_pix; pix_count++){
            line[pix_count] = (avg[pix_count] / count[pix_count]) & 0xff;
            if( (0xff / 2.95) > (line[pix_count] & 0xff ))
            {/* printf("schleife\n");*/
/*                  printf("%d %x \n",pix_count, (line[pix_count] & 0xff )); */
                if ( (VGA_flag != 0) && (VGA_flag == VGA_flag_old) )
                    *VGA = (*VGA +1) & 0x0f; 
                else
                { /*printf("akku\n");*/
                    akku =  1600 *  0xff * 100 / (line[pix_count] & 0xff) /
                        181 / 295 - 1600 / 181 + 1;
                    if (akku > 0xf) akku = 0xf;
                    if (akku > ((*VGA) & 0xf))
                        (*VGA) = ((*VGA) & 0xf0) | (akku & 0x0f);
                }
                if  ((VGA_flag == VGA_flag_old) && (VGA_flag < 0xf ))
                    VGA_flag++;
                if  (VGA_flag >= 0xf )
                {
                    VGA_flag_old = VGA_flag;
                    printf("Calibration-error or wait for warmup the lamp\n");
                }
                
            }
        }
    }while (VGA_flag != VGA_flag_old );
// fine-tuning of gain (white-value)
//           printf ("Schleife ok flag = %d\n ", VGA_flag );
//    printdata( count , 1 , max_pix);

    for (pix_count = 0; pix_count < max_pix; pix_count++){
        akku =  (25600 *  0xff / (line[pix_count] & 0xff) /
                 195  - 25600 / 195 );
        image->LM9811_arr_PGA[pix_count] = akku & 0xff;
//            printf ("%d ",akku);
    }
    

//    printdata( count , 1 , max_pix);

    *VGA = *VGA | 0x80;
    
    image->LM9811_reg_PGA = *PGA;
    image->LM9811_reg_VGA = *VGA;
    image->LM9811_reg_DAC = *DAC;
    

//    printdata(line , 1 , max_pix);
//    printf("\nPGA %x\n",*PGA & 0xff);
//    printf("\nVGA %x\n",*VGA & 0xff);
//    printf("\nDAC %x\n",*DAC & 0xff);


    image->LM9811_reg_VGA = image->LM9811_reg_VGA | 0x90;
    
    free(calibration_image);
} 













