/*
 * picasm -- config.c
 *
 * Copyright 1995-2000 Timo Rossi, <trossi@iki.fi>
 * See the file LICENSE for license terms.
 *
 * The code in this file handles configuration fuse
 * setting with the CONFIG-directive
 *
 */

/*
 *XXXXX
 * 16BIT PM1/PM2 (operation mode) and WDTPS (watchdog prescaler)
 * need special handling (also code protection (in operation mode bits))
 *
 * Storing 16bit PIC config files to hex file? test with MPASM?
 */

#include <stdio.h>
#include <string.h>

#include "picasm.h"
#include "token.h"

/*
 * Check Yes/No (or Disable(d)/Enable(d) or On/Off) strings for CONFIG
 *
 * returns: 0=no, 1=yes, -1=error
 *
 */
static int
config_yes_no(char *s)
{
    if(strcasecmp(s, "yes") == 0 || strcasecmp(s, "on") == 0 ||
       strncasecmp(s, "enable", 6) == 0)
	return 1;
    else if(strcasecmp(s, "no") == 0 || strcasecmp(s, "off") == 0 ||
	    strncasecmp(s, "disable", 7) == 0)
	return 0;
    else
	return -1; /* error */
}

/*
 * selstr consists of NUL-terminated strings with two NULs in the end
 * If str matches one of the components of selstr, the index of
 * that component (starting from zero) is returned,
 * if no match is found, -1 is returned.
 */
static int
strsel(const char *selstr, const char *str)
{
    int sel = 0;

    while(*selstr != '\0')
    {
	if(strcasecmp(selstr, str) == 0)
	    return sel;
	
	selstr += strlen(selstr)+1;
	sel++;
    }

    return -1;
}

/*
 * parse the CONFIG directive
 * 
 * (the separate code protections of PIC14000 are not currently,
 *  and neither are the partial protections on some PICs)
 * 
 */
void
parse_config(void)
{
    static char symname[260];

    static const char osc_names[] =
	"LP\0XT\0HS\0RC\0IN\0INTRC\0INTRC_CLKOUT\0"
	"EXTRC\0EXTRC_CLKOUT\0LF\0EC\0";

    static const char config_names[] =
	"CP\0WDTE\0PWRTE\0MCLRE\0BODEN\0MPEEN\0"
	"LVP\0CPD\0WRT\0BKBUG\0";

    if(pic_type == NULL)
	fatal_error("PIC device type not set");

    for(;;)
    {
	if(token_type != TOK_IDENTIFIER)
	{
cfg_error:
	    error(1, "CONFIG syntax error");
	    return;
	}
	strcpy(symname, token_string);
	get_token();

	/* hmm... this is a little kludge, but as
	   the tokenizer now makes 'local id's from
	   the valid config strings, this must be used... */
	if(token_type != TOK_LOCAL_ID)
	{
	    if(token_type != TOK_EQUAL)
	    {
		error(1, "'=' expected");
		return;
	    }
	    get_token();
	    if(token_type != TOK_IDENTIFIER)
		goto cfg_error;
	}

	if(strcasecmp(symname, "OSC") == 0)
	{
	    int t;

	    if((t = strsel(osc_names, token_string)) < 0 ||
	       pic_type->osc[t] < 0)
		goto cfg_error;

	    if(config_fuses == INVALID_CONFIG)
		config_fuses = pic_type->config_def;
	    
	    config_fuses = (config_fuses & ~pic_type->osc_mask) | pic_type->osc[t];
	}
	else if(strcasecmp(symname, "MODE") == 0 || strcasecmp(symname, "PM") == 0)
	{
	    int t;

	    if(pic_type->instr_set != PIC16BIT)
		goto cfg_error;

	    if((t = strsel("PMC\0EMC\0MC\0MP\0", token_string)) < 0)
		goto cfg_error;

	    config_fuses = (config_fuses & 0xffaf) | ((t & 1) << 4) |
		((t & 2) << 5);
	    if(strcmp(pic_type->name, "17C42") != 0)
		config_fuses = (config_fuses & 0xff7f) | ((t != 0) << 15);

	}
	else if(strcasecmp(symname, "WDTPS") == 0)
	{
	    int t;

	    if(pic_type->instr_set != PIC16BIT)
		goto cfg_error;

	    if((t = strsel("OFF\064\0256\00\0", token_string)) < 0)
		goto cfg_error;

	    config_fuses = (config_fuses & 0xfff3) | (t << 2);
	    
	} else { /* OSC ... */
	    int t, fuse_val;

	    if((t = strsel(config_names, symname)) < 0)
	    {
		strcat(symname, "E");
		if((t = strsel(config_names, symname)) < 0)
		{
		    strcat(symname, "N");
		    if((t = strsel(config_names, symname)) < 0)
			goto cfg_error;
		}
	    }

	    if((fuse_val = pic_type->fuse[t]) == 0)
		goto cfg_error;

	    if(config_fuses == INVALID_CONFIG)
		config_fuses = pic_type->config_def;

	    if((t = config_yes_no(token_string)) < 0)
		goto cfg_error;

	    if(fuse_val < 0)
	    {
		fuse_val = -fuse_val;
		t = !t;
	    }

	    if(t)
		config_fuses |= fuse_val;
	    else
		config_fuses &= ~fuse_val;
	}
	
	get_token();
	if(token_type != TOK_COMMA)
	    break;
	
	get_token();
    }
}
