/*
 * mb-apppage.cc --
 *
 *      MediaBoard Application Page object
 *      manages the various mb drawing objects on a single page
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-apppage.cc,v 1.17 2002/02/03 03:16:30 lim Exp $
 */

#include "mb/mb-page.h"
#include "mb/mb-mgr.h"
#include "mb/mb-cmd.h"
#include "mb/mb-rcvr.h"

//
// Init static variable
//   Increase op array by this amount each time it is full
//
const PageId cNullPageId; /* the default constructors will initialize
			     everything to zero */

Page::Page(MBManager *pMgr, PageId pgid, const char *szCanvas)
	: MBPageObject(pMgr, pgid), pCanv_(NULL), isVisible_(FALSE),
	  phtCanvId_(NULL), phtTrashCan_(NULL),
	  lastExeSn_(0), isLocal_(FALSE)
{
	Tcl& tcl = Tcl::instance();
	MBBaseCanvas *pCanv = (MBBaseCanvas*) tcl.lookup(szCanvas);
	if (!pCanv) {
		SignalError(("Cannot find canvas %s!", szCanvas));
		return;
	} else {
		pCanv_ = pCanv;
	}

    	// these are the implicit assumptions that we made
	// if the size are different, the hashtable would become haywire
	assert(sizeof(CanvItemId) == sizeof(char*));
	assert( (sizeof(ClientData) == sizeof(n_long)) );
	phtCanvId_ = new Tcl_HashTable;
	Tcl_InitHashTable(phtCanvId_, TCL_ONE_WORD_KEYS);

	phtTrashCan_ = new Tcl_HashTable;
	Tcl_InitHashTable(phtTrashCan_, TCL_ONE_WORD_KEYS);

	return;
}

/*virtual*/
Page::~Page()
{
	Tcl_DeleteHashTable(phtCanvId_);
	delete phtCanvId_;

	emptyTrash();
	delete phtTrashCan_;
}


Bool Page::emptyTrash()
{
	Tcl_HashSearch hsearch;
	Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(phtTrashCan_, &hsearch);
	while (pEntry) {
		MBCanvItem *pItem = (MBCanvItem *)Tcl_GetHashValue(pEntry);
		delete pItem;
		pEntry=Tcl_NextHashEntry(&hsearch);
	}
	Tcl_DeleteHashTable(phtTrashCan_);
	return TRUE;
}


//
// Remember a creation of a new id on the canvas
//
void Page::addCanvItem(MBCmd* pCmd, ulong itemId, const CanvItemId& canvId)
{
	// we use the value zero as indication of failure
	// there should not be a id of zero from the canvas
	assert(canvId!=0);

	int created;

	MTrace(trcMB|trcExcessive,
	       ("adding item id: %lu <-> canv id: %lu", itemId, canvId));
	Tcl_HashEntry *newEntry =
		Tcl_CreateHashEntry(phtCanvId_, (char*)itemId, &created);
	if (!created)   // currently we don't change the canvid once created
		assert(FALSE);
	Tcl_SetHashValue(newEntry, (ClientData)canvId);

	if (pCanv_) {
		pCanv_->assocItem(canvId, pCmd);
	}
}


void Page::removeCanvItem(n_long itemId)
{
	Tcl_HashEntry *pEntry = Tcl_FindHashEntry(phtCanvId_, (char*)itemId);
	CanvItemId canvId=0;
	if (pEntry) {
		canvId = (CanvItemId) Tcl_GetHashValue(pEntry);
		Tcl_DeleteHashEntry(pEntry);
	}
	assert(!Tcl_FindHashEntry(phtCanvId_, (char*)itemId));

	if (pCanv_) {
		pCanv_->forgetItem(canvId);
	}
}


Bool Page::saveItemInTrash(n_long itemId, MBCanvItem* pItem)
{
	assert(pItem && "should not add null item");

	int created;
	Tcl_HashEntry *pEntry =
		Tcl_CreateHashEntry(phtTrashCan_, (char*) itemId, &created);
	if (!created) {
		MTrace(trcAll,
		       ("Weird: Delete an deleted item? Should not happen!" ));
		MBCanvItem* pItem = (MBCanvItem*)Tcl_GetHashValue(pEntry);
		delete pItem;
	}
	Tcl_SetHashValue(pEntry, pItem);
	return TRUE;
}


MBCanvItem* Page::removeItemFromTrash(const CanvItemId& itemId)
{
	Tcl_HashEntry *pEntry = Tcl_FindHashEntry(phtTrashCan_, (char*)itemId);
	if (!pEntry)
		return NULL;
	MBCanvItem* pItem = (MBCanvItem*) Tcl_GetHashValue(pEntry);
	assert(pItem!=0);
	return pItem;
}


#ifdef OLD
ulong Page::NearestItem(MBBaseRcvr* pRcvr, Coord x, Coord y, Coord dist)
{
	MBBaseCanvas* pCanvas = getCanvas();
	CanvItemId startcid =
		pCanvas->nextNearest(x, y, dist, 0);
	if (!startcid) return 0;         // implies no items are near enough

	// the canvas will return the closest item even if it is not within
	// dist, so we will have to check that it overlaps
	MBCmd* pCmd;
	if (pCanvas->overlap(x-dist, y-dist, x+dist, y+dist, startcid)) {
		pCmd = pCanvas->canvId2cmd(startcid);
		if (pCmd && pCmd->rcvr()==pRcvr) {
			return canvId2itemId(startcid);
		}
	} else { // the closest item is out of range
		return 0;
	}

	// canvas will loop thru all items, until startcid is returned again
	// all subsequent items returned should be within dist
	ulong nearestCid = 0;
	ulong nextCid=pCanvas->nextNearest(x,y,dist,startcid);
	while (nextCid!=startcid) {
		pCmd = pCanvas->canvId2cmd(nextCid);
		if (pCmd && pCmd->rcvr()==pRcvr) {
			nearestCid = nextCid;
			break;
		}
		nextCid=pCanvas->nextNearest(x, y, dist, nextCid);
	}
	MTrace(trcMB|trcVerbose,
	       ("Nearest Item returns (canvid)%ld",nearestCid));
	if (!nearestCid) return 0;
	else return canvId2itemId(nearestCid);
}


void Page::OverlappedItems(Coord x1, Coord y1, Coord x2, Coord y2,
                           int& countItems, n_long* &arItemIds)
{
	CanvItemId* arCanvItems=NULL;
	// initialize OUT parameters
	arItemIds=NULL;
	countItems=0;
	int count;
	getCanvas()->OverlappedItems(x1,y1,x2,y2,count,arCanvItems);
	MTrace(trcMB,("Canvas ov items: %d",count));
	if (!count) return;

	// filter off items not from this receiver
	countItems=count;
	arItemIds=NULL;
	CanvItemId* pCanvId=NULL;
	// pass 1: set all items we don't know about to zero
	for (pCanvId=arCanvItems; pCanvId<arCanvItems+count; pCanvId++) {
		if (!canvId2itemId(*pCanvId)) {
			MTrace(trcMB|trcVerbose,("Remove cid: %d",*pCanvId));
			*pCanvId=0;
			countItems--;
		}
	}
	if (!countItems) {
		delete arCanvItems;
		arCanvItems=NULL;
		return;
	}
	// pass 2: put all the valid items into one array
	arItemIds=new CanvItemId[countItems];
	int index=0;
	for (pCanvId=arCanvItems; pCanvId<arCanvItems+count; pCanvId++) {
		if (*pCanvId)
			arItemIds[index++]= canvId2itemId(*pCanvId);
	}
	assert( (index==countItems) && "check the above algorithm");
	return;
}
#endif


// Returns an canvId that is the most recent more recent than
// mostRecent but older than timestamp and update mostRecent;
// if cannot find a more recent one, return zero
//
// Criteria:
//  We scan thru the commands from the most recent, until we find the
//  item that is older than timestamp, and refer to and item.
//
//  REVIEW: Can do better? current: O(nM) on average, but const if append
//
CanvItemId Page::FindMostRecent(n_long& mostRecent, n_long timestamp)
{
	Bool done=FALSE;
	MBCmd *pCmd=NULL;
	n_long i=getMaxSeqno();

	// find the youngest one that refers to an (active) item
	n_long iid=0;
	CanvItemId cid = 0;
	// loop from most recent:
	//       - skip commands that are null or are older than timestamp
	//       - exit if we cannot find an item that fits citeria
	//         (see above)
	//       - skip if the command does not refer to an item or
	//         is not 'active'
	while (!done && i>=1) {
		pCmd = getCmd(i--);
		if (pCmd==NULL || pCmd->getTimeStamp()>=timestamp )
			continue;

		if (mostRecent >= pCmd->getTimeStamp())
			break;

		if ( 0==(iid=pCmd->getItemId())
		     ||
		     0==(cid=itemId2canvId(iid)) )
			continue;

		done = TRUE;
		break;
	}

	if (done==TRUE) {
		mostRecent = pCmd->getTimeStamp();
		assert(cid);
		return cid;
	}
	return 0;
}


// raises tgtId to just after cid
void Page::RaiseAfter(CanvItemId cid, n_long tgtId)
{
	CanvItemId afterId=itemId2canvId(tgtId);
	if (!afterId) {
		// this could happen e.g. when we are doing
		// reverse execution and taking away items
		//MTrace(trcMB|trcVerbose,("Could not find %lu",tgtId));
		//assert(FALSE);
		return;
	}
	// no need to raise if it is the same item
	if (afterId != cid)
		getCanvas()->raiseAfter(cid,afterId);
}


// raises tgtId to the front
void Page::RaiseToFront(n_long tgtId)
{
	CanvItemId cid=itemId2canvId(tgtId);
	if (!cid) {
		assert(FALSE);
		return;
	}
	getCanvas()->raise(cid);
}

/* time travel */
int Page::warpTime(MBReceiver* pRcvr, const MBTime& t)
{
	UpdateDeferred(t);
	if (targetTime() == t)
		return TCL_OK;

	if ((t != cMBTimeAny) &&
	    ((targetTime() == cMBTimeAny) || t < targetTime())) {
		return reverseWarp(pRcvr, t);
	}
	/* forward warp to time t
	 *   --  the range of commands to be executed is any commands
	 *       starting from
	 *		1) the first command if we have not execute any
	 *	     or 2) the next one since the last executed
	 *       to
	 *              1) the last command if target time is greater
	 *                 than the max time we have
	 *           or 2) the command just before one that has
	 *                 greater time than t
	 */
	assert(targetTime() != cMBTimeAny);
	ulong nextSn;
	nextSn = lastExeSn_ + 1;
	MTrace(trcMB|trcVerbose, ("warp: nextSn = %d", nextSn));
	MBCmd* pCmd;
	for (; nextSn <=  getMaxSeqno(); nextSn++) {
		pCmd = getCmd(nextSn);
		if (!pCmd) continue;
		/* the second clause is in fact redundant since
		 * cAnyTime = 0, but this makes the code clearer */
		if (!pCmd->activeAt(t))
			break;
		if (pCmd->Incomplete(this)) {
			Defer(pCmd);
			continue;
		}
		pRcvr->handleCmd(pCmd, this, targetTime(), t);
	}
	lastExeSn_ = nextSn - 1;
	MTrace(trcMB|trcVerbose, ("warp: set lastexeSn = %d", lastExeSn_));
	targetTime_ = t;
	return TCL_OK;
}

/* warp backwards in time, this is a private helper function for
 * warpTime */
/* FIXME: a better way is to use active objects, and ask the canvas to
 * iterate thru all items and call settime on each */
int Page::reverseWarp(MBReceiver* pRcvr, const MBTime& t)
{
	/* reverse warp to time t
	 *   --  the range of commands to be reversed is commands
	 *       starting from
	 *		1) the last command if current target is 'anytime'
	 *	     or 2) the command just before the last executed
	 *       to
	 *              1) the first command if target time is smaller
	 *                 than the min time we have
	 *           or 2) the command just after one that starts
	 *                 before t
	 */
	ulong nextSn;
	if (targetTime_ == cMBTimeAny) {
		for (nextSn=getMaxSeqno(); nextSn>=1; nextSn--) {
			if (getCmd(nextSn)) break;
		}
		if (nextSn == 0) {
			targetTime_ = t;
			return TCL_OK; /* no commands yet */
		}
	} else {
		nextSn = lastExeSn_;
	}
	MBCmd* pCmd;
	/* so that new commands later will cause activity */
	(DYN_CAST(MBManager*)(pRcvr->getMgr()))->resetTime();
	MTrace(trcMB|trcVerbose, ("revwarp: nextSn = %d", nextSn));
	for (; nextSn >= 1; nextSn--) {
		pCmd = getCmd(nextSn);
		if (!pCmd) continue;
		if (pCmd->startTime() <= t)
			break;
		if (!pCmd->Incomplete(this)) {
			// set last cmd and reverse command, but don't notify
			// FIXME: should we notify?
			pRcvr->handleCmd(pCmd, this, targetTime(), t);
		}
	}
	MTrace(trcMB|trcVerbose, ("revwarp: set lastexeSn = %d", lastExeSn_));
	lastExeSn_ = nextSn;
	targetTime_ = t;
	return TCL_OK;
}

int Page::timeRange(const PageId & /* pgId */, MBTime& start, MBTime& end)
{
	MBCmd* pCmd=NULL;
	ulong i;
	for (i=1; i <= getMaxSeqno(); i++)
		if (NULL != (pCmd = getCmd(i))) break;
	if (pCmd)
		start = pCmd->getTimeStamp();

	for (i=getMaxSeqno(); i>0; i--)
		if (NULL != (pCmd = getCmd(i)) ) break;
	if (pCmd)
		end = pCmd->getTimeStamp();

	return TCL_OK;
}

#ifdef MB_DEBUG
void Page::DumpCanvas(Tcl_Obj* pObj)
{
	PageId pgId=getId();
	char szTmp[200];
	sprintf(szTmp, "\n ======= CANVAS for Page: (%d%s):%lx ===========",
	      pgId.sid.ss_uid, intoa(pgId.sid.ss_addr), pgId.uid);
	Tcl_AppendToObj(pObj, szTmp, -1);

	Tcl_HashSearch hsearch;
	Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(phtCanvId_, &hsearch);
	ulong itemId;
	CanvItemId canvId;
	while (pEntry) {
		itemId = (ulong) Tcl_GetHashKey(phtCanvId_,pEntry);
		canvId = (CanvItemId)Tcl_GetHashValue(pEntry);
		sprintf(szTmp, "- itemid: %lu -> %lu",itemId,canvId);
		Tcl_AppendToObj(pObj, szTmp, -1);
		pEntry=Tcl_NextHashEntry(&hsearch);
	}
}
#endif  // MB_DEBUG
