/*
 * decoder.cc --
 *
 *      Video Decoder class
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/codec/decoder.cc,v 1.23 2002/02/03 03:13:33 lim Exp $";

#include <stdio.h>
#include <stdlib.h>
#ifndef WIN32
#include <sys/param.h>
#endif
#include "sys-time.h"
#include "inet.h"
#include "rtp.h"
#include "decoder.h"
#include "renderer.h"
#include "color-hist.h"
#include "postdct.h"

extern "C" {
#ifdef USE_SHM
#include <sys/ipc.h>
#include <sys/shm.h>
#if defined(sun) && !defined(__svr4__)
int shmget(key_t, int, int);
char *shmat(int, char*, int);
int shmdt(char*);
int shmctl(int, int, struct shmid_ds*);
#endif
#endif
}

Decoder::Decoder(int hdrlen) : PacketHandler(hdrlen),
	nstat_(0), color_(1), decimation_(422), inw_(0), inh_(0),
	engines_(0), rvts_(0), nblk_(0), ndblk_(0)
{
	/*FIXME*/
	now_ = 1;

	for (int i = 0; i < MAXSTAT; ++i)
		stat_[i].cnt = 0;
}


Decoder::~Decoder()
{
	if (rvts_) delete[] rvts_;
}

int Decoder::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
		if (strcmp(argv[1], "width") == 0) {
			tcl.resultf("%d", width());
			return (TCL_OK);
		}
		if (strcmp(argv[1], "height") == 0) {
			tcl.resultf("%d", height());
			return (TCL_OK);
		}
		if (strcmp(argv[1], "info") == 0) {
			char* bp = tcl.buffer();
			info(bp);
			tcl.result(bp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "stats") == 0) {
			char* bp = tcl.buffer();
			tcl.result(bp);
			stats(bp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "decimation") == 0) {
			tcl.resultf("%d", decimation());
			return (TCL_OK);
		}
		if (strcmp(argv[1], "redraw") == 0) {
			redraw();
			return (TCL_OK);
		}
		if (strcmp(argv[1], "playout") == 0) {
			/*FIXME*/
			tcl.result("0");
			return (TCL_OK);
		}
	} else if (argc == 3) {
		if (strcmp(argv[1], "color") == 0) {
			setcolor(atoi(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "attach") == 0) {
			Renderer* r = (Renderer*)TclObject::lookup(argv[2]);
			if (r == 0) {
				tcl.resultf("%s attach: no such renderer: %s",
					    argv[0], argv[2]);
				return (TCL_ERROR);
			}
			attach(r);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "detach") == 0) {
			Renderer* r = (Renderer*)TclObject::lookup(argv[2]);
			if (r == 0) {
				tcl.resultf("%s detach: no such target: %s",
					    argv[0], argv[2]);
				return (TCL_ERROR);
			}
			return detach(r);
		}
		if (strcmp(argv[1], "histogram") == 0) {
			ColorHist* ch = ColorHist::lookup(argv[2]);
			if (ch == 0) {
				tcl.resultf("%s histogram: no such histogram: %s",
					    argv[0], argv[2]);
				return (TCL_ERROR);
			}
			colorhist(ch->histogram());
			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}

void Decoder::info(char* wrk) const
{
	*wrk = 0;
}

void Decoder::stats(char* bp)
{
	if (nstat_ == 0) {
		*bp = 0;
		return;
	}
	for (int i = 0; i < nstat_; ++i) {
		sprintf(bp, "%s %d ", stat_[i].name, stat_[i].cnt);
		bp += strlen(bp);
	}
	bp[-1] = 0;
}

void Decoder::allocshm(dmabuf& d, int size, int flag)
{
#ifdef USE_SHM
	d.shmid = shmget(IPC_PRIVATE, size, IPC_CREAT|0777);
	if (d.shmid < 0) {
		perror("shmget");
		exit(1);
	}
	d.bp = (u_char *)shmat(d.shmid, 0, flag);
	if (d.bp == (u_char*)-1) {
		perror("shmat");
		exit(1);
	}
#endif
}

/*FIXME not used*/
void Decoder::freeshm(dmabuf& d)
{
#ifdef USE_SHM
	if (shmdt((char*)d.bp) < 0)
		perror("shmdt");
#endif
}

/*
 * Return time of day in microseconds.
 */
double Decoder::gettimeofday()
{
	timeval tv;
	::gettimeofday(&tv, 0);
	return (1e6 * double(tv.tv_sec) + double(tv.tv_usec));
}

void Decoder::attach(Renderer* r)
{
	r->next_ = engines_;
	engines_ = r;
	r->setcolor(color_);
	r->now(0);
	redraw();
}

void Decoder::redraw(const u_char* /* frm */)
{
	/* make sure all the renderer's are synced */
	now_ = 1;
	memset(rvts_, 1, nblk_);
	for (Renderer* p = engines_; p != 0; p = p->next_)
		p->now(0);
//	render_frame(frm);
}

int Decoder::detach(Renderer* r)
{
	Renderer** p;
	for (p = &engines_; *p != r; p = &(*p)->next_)
		if (*p == 0)
			return TCL_ERROR;
	*p = (*p)->next_;
	return TCL_OK;
}

void Decoder::colorhist_422_556(u_int* hist, const u_char* yp,
					const u_char* up, const u_char* vp,
					int width, int h) const
{
	while (--h >= 0) {
		for (int w = width; w > 0; w -= 2) {
			/* 5V:5U:6Y */
			int uv = vp[0] >> 3 << 11 | up[0] >> 3 << 6;
			++hist[uv | yp[0] >> 2];
			++hist[uv | yp[1] >> 2];
			yp += 2;
			up += 1;
			vp += 1;
		}
	}
}

void Decoder::colorhist_411_556(u_int* hist, const u_char* yp,
					const u_char* up, const u_char* vp,
					int width, int h) const
{
	for (; h > 0; h -= 2) {
		for (int w = width; w > 0; w -= 2) {
			/* 5V:5U:6Y */
			int uv = vp[0] >> 3 << 11 | up[0] >> 3 << 6;
			++hist[uv | yp[0] >> 2];
			++hist[uv | yp[1] >> 2];
			++hist[uv | yp[width] >> 2];
			++hist[uv | yp[width + 1] >> 2];
			yp += 2;
			up += 1;
			vp += 1;
		}
		yp += width;
	}
}

void Decoder::render_frame(const u_char* frm)
{
	render_frame(frm, CODEC_ANY, 0);
}

void Decoder::render_frame(const u_char* frm, int codec, int quality)
{
	//
	// rvts_
	//
	// rvts_ is the Replenishment Vector of "TimeStamps" buffer, and it's 
	//	composed of one byte per 8-pixel-wide, 8-pixel-high block. The 
	//	purpose of carrying a replenishment vector in the decoder is to 
	//	relieve the renderer of the task of refreshing the full video 
	//	window if only a small number of blocks were updated by the source. 
	//
	// The structure is created by the main decoder object (Decoder::resize()), 
	//	and is used by it, the particular decoder (e.g., H261Decoder and its 
	//	implementation P64Decoder, where it's called marks_), and the 
	//	Renderer. Every time the particular decoder decodes a block, it 
	//	marks up the corresponding byte in the rvts_ with a commonly-agreed 
	//	"timestamp" (check P64Decoder::decode_mb()). These timestamps have 
	//	nothing in common with RTP timestamps (they are for sure 1-byte 
	//	numbers).
	//
	// When the renderer receives the frame, it only renders those blocks 
	//	whose timestamp is the current one. To avoid old blocks being 
	//	rendered because they were updated 256 time ticks ago (and therefore 
	//	whose timestamp coincides  with the current timestamp), 
	//	Decoder::render_frame() takes care of adding one to those 
	//	timestamps before permitting the particular decoder accessing to 
	//	the structure. Therefore, the meaning of each table's value is the 
	//	following: 
	//
	//	(now_ - rvts_[i]) mod 256 = how long ago was this block updated (255 
	//	                            meaning 255 ticks or more)
	//
	// This decoder optimization only works for codecs where you can send 
	//	only some blocks of a frame. I.e., it doesn't make sense for MJPEG.
	//
    Renderer *p, *q;
	long int engine_footprint, new_engine_footprint;

	// get the initial renderer list footprint
	engine_footprint = 0;
	for (p = engines_; p != 0; p = p->next_) {
		engine_footprint = engine_footprint ^ (long int)p;
	}

	YuvFrame f(now_, (u_int8_t*)frm, rvts_, inw_, inh_);
	for (p = engines_; p != 0; p = p->next_) {
		p->recv(&f, codec, quality);

		// get the new renderer list footprint
		new_engine_footprint = 0;
		for (q = engines_; q != 0; q = q->next_) {
			new_engine_footprint = new_engine_footprint ^ (long int)q;
		}

		// compare both footprints and exit if there were changes
		if (new_engine_footprint != engine_footprint) {
			// the renderer list has changed
			break;
		}
	}

	// get the new timestamp
	now_ = (now_ + 1) & 0xff;

	// actualize the replenishment vector (add one with saturation to 
	//	the blocks whose timestamp coincides with the new timestamp)
	u_char* ts = rvts_;
	for (int k = nblk_; --k >= 0; ++ts) {
		if (*ts == now_) {
			*ts = (*ts + 1) & 0xff;
		}
	}
}

void Decoder::setcolor(int color)
{
	if (color != color_) {
		color_ = color;
		for (Renderer* p = engines_; p != 0; p = p->next_)
			p->setcolor(color_);
	}
}

void Decoder::resize(int width, int height)
{
	inw_ = width;
	inh_ = height;
	nblk_ = (width * height) / 64;
	delete[] rvts_;
	rvts_ = new u_char[nblk_];
	memset(rvts_, 0, nblk_);
	redraw();
}

/*
 * Dummy class for streams that we recognize but cannot decode (for lack
 * of software support).  We still want to put all the stats etc. --
 * especially the format -- but can't decode it.  FIXME We need to somehow
 * mark the stream in the user-interface so that the user knows that
 * vic cannot decode the stream (to avoid assuming something else is
 * wrong with the tranmission)
 */
class NullDecoder : public Decoder {
public:
	NullDecoder() : Decoder(0) {}
	int colorhist(u_int* /* histogram */) const { return (0); }
	void redraw() {}
	virtual void recv(pktbuf* pb) { pb->release();}
};

static class VideoNullDecoderClass : public TclClass {
public:
	VideoNullDecoderClass() : TclClass("Module/VideoDecoder/Null") {}
	TclObject* create(int /* argc */, const char*const* /* argv */) {
		return (new NullDecoder);
	}
} dm_null;

PlaneDecoder::PlaneDecoder(int hdrlen) : Decoder(hdrlen), frm_(0)
{
}

PlaneDecoder::~PlaneDecoder()
{
	delete[] frm_;
}

void PlaneDecoder::redraw()
{
	Decoder::redraw(frm_);
}

void PlaneDecoder::resize(int width, int height)
{
	delete[] frm_;
	int size = width * height;
	frm_ = new u_char[2 * size];
	/*
	 * Initialize image to gray.
	 */
	memset(frm_, 0x80, 2 * size);
	Decoder::resize(width, height);
}

int PlaneDecoder::colorhist(u_int* histogram) const
{
	int w = inw_;
	int h = inh_;
	int s = w * h;
	u_char* up = frm_ + s;
	colorhist_422_556(histogram, frm_, up, up + (s >> 1), w, h);
	return (1);
}
