# vcc3-lib.tcl --
#
#       Core library functions for talking to Canon Vcc3 camera through the
#       serial port.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

global serialFd_
global g_lastCommand

# this is used for request/response transactions, such as readPanSpeed; it
#   is also now used for ack and response synchronization
global g_sync

# FIXME - these camera state variables should be in a global array such as
#   g_camState so that functions only need to do global g_camState and they
#   can get everything (to avoid stupid errors)
# camera state variables
global g_panSpeed
global g_tiltSpeed
global g_panPosition
global g_tiltPosition
global g_zoomSpeed
global g_zoomPosition
global g_maxZoom
global g_focusSpeed
global g_aeRef g_exposureSpeed g_iris g_shutterSpeed g_gain
global g_fadeSpeed g_fadeGain g_fadeLevel

source ../../lib/tcl/sleep.tcl
source vcc3-pantilt.tcl
source vcc3-zoom.tcl
source vcc3-ccu.tcl


# this only supports one camera

#
# If no filename is specified, the serial port is used
#
proc vcc3_setupControl { {filename "/dev/cuaa0"} } {
    global serialFd_
    global g_sync g_lastCommand
    global g_panSpeed g_tiltSpeed g_panPosition g_tiltPosition
    global g_zoomSpeed g_zoomPosition g_maxZoom g_focusSpeed
    global g_aeRef g_exposureSpeed g_iris g_shutterSpeed g_gain

    set g_sync(ack) "NO_WAIT"
    set g_sync(response) "NO_WAIT"
    set g_lastCommand ""
    set g_panSpeed 0
    set g_tiltSpeed 0
    set g_panPosition 0
    set g_tiltPosition 0
    set g_zoomSpeed 0
    set g_zoomPosition 0
    set g_maxZoom 0
    set g_focusSpeed 0
    set g_focusPosition 0
    set g_aeRef 0
    set g_exposureSpeed 0
    set g_iris 0
    set g_shutterSpeed 0
    set g_gain 0
    set g_fadeSpeed 0
    set g_fadeGain 0
    set g_fadeLevel 0

    
    if {[catch {open "$filename" {RDWR NONBLOCK NOCTTY}} serialFd_]} {
	puts stderr "Fatal error:  cannot open serial port: $filename, error is $serialFd_"
	return 0
    } else {
	if {[catch {fconfigure $serialFd_ -blocking 0 -mode "9600,n,8,2" -buffering none -translation binary} result]} {
	    puts stderr "Fatal error: cannot configure serial port"
	    puts stderr $result
	    return 0
	}
    }
    
    # if we got here, we were able to open the serial port
    vcc3_setCallback "vcc3_receiveData"
    
    set retVal [vcc3_initCamera]

    if {!$retVal} {
	puts stderr "Init camera failed: ensure cable is connected and power is on"
	return 0
    }
    return 1
}

proc vcc3_closeControl {} {
    global serialFd_

    # restore remote control mode - no notify (power on default)
    set str [binary format "cccc" 0x08 0x17 0x01 0x01]
    vcc3_sendCommand $str
    # need to wait for the command to complete (ack to be received)
    sleep 1000
    
    catch {close $serialFd_}
}

# Internal function
proc vcc3_initCamera {} {
    global g_sync
    
    # set the vcc3 into host mode
    set result [vcc3_enablePCControl]
    if {!$result} {
	return 0
    }
    set result [vcc3_getMaxZoom]
    if {!$result} {
	return 0
    }
    set result [vcc3_home]
    return $result
}

proc vcc3_receiveData {} {
    set pktList [vcc3_getPacket]

#    puts stdout ""
#    puts stdout "vcc3_receiveData: got packet"
#    vcc3_printPacket $pktList

    array set pkt $pktList
    set type [expr $pkt(frameID) & 0x80]
#    set temp [format "pkt type is 0x%x, frameID is 0x%x" $type $pkt(frameID)]
#    puts stdout $temp
    if {$type != 0} {
#	puts stdout "vcc3_receiveData: got ack/nack frame"
	vcc3_processAck $pktList
    } else {
#	puts stdout "vcc3_receiveData: got command frame"
	vcc3_processCommand $pktList
    }
}

proc vcc3_processAck {pktList} {
    global g_sync g_lastCommand
    
    array set pkt $pktList

    set g_sync(ack) "NO_WAIT"
    
    set temp [expr $pkt(commandID) & 0x17]

    if {$temp == 0x00} {
	# normal ack
#	puts stdout "vcc3_processAck: ACK: g_sync(ack) is $g_sync(ack)"
    }
    if {$temp == 0x01} {
	puts stdout "vcc3_processAck: NACK: buffer full"
    }
    if {$temp == 0x02} {
	puts stdout "vcc3_processAck: NACK: length error"
    }
    if {$temp == 0x03} {
	puts stdout "vcc3_processAck: NACK: sequence error"
    }
    if {$temp == 0x04} {
	puts stdout "vcc3_processAck: NACK: communication error"
    }
    if {$temp == 0x10} {
	puts stdout "vcc3_processAck: NACK: checksum error"
    }
}

proc vcc3_processCommand {pktList} {
    global g_sync
    array set pkt $pktList

    # send the ACK
    vcc3_sendAck

    set temp [expr $pkt(commandID) & 0x60]

    if {$temp == 0x00} {
	set g_sync(response) "NO_WAIT"
#	puts stdout "vcc3_processCommand: got positive response"
#	vcc3_printPacket $pktList
	vcc3_processPositiveResponse $pktList
    }
    if {$temp == 0x40} {
	set g_sync(response) "NO_WAIT"
	# FIXME - pan/tilt and zoom limits cause negative responses, so
	#   I should filter those out or just not print anything on a
	#   negative response
	puts stdout "vcc3_processCommand: got negative response"
	vcc3_printPacket $pktList
	vcc3_processNegativeResponse $pktList
    }
    if {$temp == 0x20} {
#	puts stdout "vcc3_processCommand: got notification command"
	vcc3_processNotification $pktList
    }
}

proc vcc3_processPositiveResponse {pktList} {
    array set pkt $pktList

    set found 0
    set dev [expr $pkt(frameID) & 0xF]
    if {$dev == 0x1} {
	set found 1
#	puts stdout "vcc3_processPositiveResponse: Zoom positive response"
	vcc3_processZoomPositiveResponse $pktList
    }
    if {$dev == 0x5} {
	set found 1
#	puts stdout "vcc3_processPositiveResponse: Pan/Tilter positive response"
	vcc3_processPanTiltPositiveResponse $pktList
    }
    if {$dev == 0x8} {
	set found 1
#	puts stdout "vcc3_processPositiveResponse: CCU positive response"
	vcc3_processCCUPositiveResponse $pktList
    }
    if {!$found} {
	puts stdout "vcc3_processPositiveResponse: unrecognized device"
	vcc3_printPacket $pktList
    }
}

proc vcc3_processNegativeResponse {pktList} {
    array set pkt $pktList

    set found 0
    set dev [expr $pkt(frameID) & 0xF]
    if {$dev == 0x1} {
	set found 1
#	puts stdout "vcc3_processNegativeResponse: Zoom negative response"
#	vcc3_processZoomNegativeResponse $pktList
    }
    if {$dev == 0x5} {
	set found 1
#	puts stdout "vcc3_processNegativeResponse: Pan/Tilter negative response"
	vcc3_processPanTiltNegativeResponse $pktList
    }
    if {$dev == 0x8} {
	set found 1
#	puts stdout "vcc3_processNegativeResponse: CCU negative response"
#	vcc3_processCCUNegativeResponse $pktList
    }
    if {!$found} {
	puts stdout "vcc3_processNegativeResponse: unrecognized device"
	vcc3_printPacket $pktList
    }
}

proc vcc3_processNotification {pktList} {
    array set pkt $pktList

    set found 0
    set dev [expr $pkt(frameID) & 0xF]
    if {$dev == 0x1} {
	set found 1
#	puts stdout "vcc3_processNotification: Zoom notification"
	vcc3_processZoomNotification $pktList
    }
    if {$dev == 0x5} {
	set found 1
#	puts stdout "vcc3_processNotification: Pan/Tilt notification"
	vcc3_processPanTiltNotification $pktList
    }
    if {$dev == 0x8} {
	set found 1
#	puts stdout "vcc3_processNotification: CCU notification"
    }
    if {!$found} {
	puts stdout "vcc3_processNotification: unrecognized device"
	vcc3_printPacket $pktList
    }
}
    
proc vcc3_printPacket {pktList} {
    # tcl doesn't allow passing of arrays, so use a list as an intermediate
    array set pkt $pktList
    
    puts stdout "\tlength = $pkt(length)"
    set temp [format "\tframeID = 0x%x" $pkt(frameID)]
    puts stdout $temp
    set temp [format "\tcommandID = 0x%x" $pkt(commandID)]
    puts stdout $temp
    set num $pkt(parameter,num)
    for {set x 0} {$x < $num} {incr x 1} {
  	puts stdout "\tparam $x = $pkt(parameter,$x)"
    }
    set temp [format " \tcheckSum = 0x%x" $pkt(checkSum)]
    puts stdout $temp
}

# Internal Function
proc vcc3_getPacket {} {
    set pkt(length) [getByte]
#    puts stdout "vcc3_receiveData: length is $pkt(length)"
    
    set pkt(frameID) [getByte]
#    set temp [format "vcc3_receiveData: frameID is 0x%x" $frameID]
#    puts stdout $temp

    set pkt(commandID) [getByte]
#    set temp [format "vcc3_receiveData: commandID is 0x%x" $commandID]
#    puts stdout $temp

    set numLeft [expr $pkt(length) - 3]
    set pkt(parameter,num) $numLeft
    for {set x 0} {$x < $numLeft} {incr x 1} {
	set pkt(parameter,$x) [getByte]
    }
    
    set pkt(checkSum) [getByte]
#    set temp [format "vcc3_receiveData: checkSum is 0x%x" $checkSum]
#    puts stdout $temp

    # tcl doesn't allow arrays to be passed around, so use a list
    set pktList [array get pkt]
    
    return $pktList
}

proc vcc3_sendAck {} {
    set str [binary format "cc" 0x88 0x00]
    vcc3_sendString $str
}

proc vcc3_sendCommand {str} {
    global g_sync g_lastCommand

#    puts stdout "vcc3_sendCommand called"
    
    # FIXME - do I need to wait for response?
    # commands are serialized, this is inefficient in some cases,
    #   but easier to implement
    # although not all need to be serialized, it is easier than checking
    #   for "duplex exceptions"
    #
    # this makes presets not as quick to respond because you need to pan/tilt,
    #   then zoom, then adjust brightness; if you didn't force serialization,
    #   you could be zooming as you pan/tilt
    #
    # I didn't implement this because I didn't have time; however, some of the
    #   api functions have support for this, since they use g_sync variables
    #   which aren't needed with the sync on response
    #
    # Another way to do this might be to look at the negative responses, and
    #   retry after some delay; in this way the camera takes care of sniffing
    #   out duplex exceptions; I think this is the most promising for ensuring
    #   parallel execution of commands when possible

    # should I wait forever for a response?  or just set a large timeout value
    #  the problem occurs when a button is pushed in the middle of a long
    #  operation such as fade, it will fail instead of looping until the fade
    #  is done, what should the behavior be?

    #puts stdout "sending $str"
    #vcc3_waitSync "response" forever

    # for now, I just return 0 if another operation is in progress, this
    #   puts the burden of retry on the caller (who may have his own thread
    #   making it easier)

    if {$g_sync(response) == "WAIT"} {
	# just fail here
#	puts stdout "vcc3_sendCommand: wait is true so returning 0"
	return 0
    }
    set g_sync(ack) "WAIT"
    set g_sync(response) "WAIT"
    set g_lastCommand $str
    vcc3_sendString $str
    set result [vcc3_waitSync "ack"]
    if {!$result} {
	return 0
    }

    # waiting here ensures that all commands have completed before returning;
    #   this allows programs to write stuff such as:
    #        vcc3_setPanPosition 400
    #        vcc3_setTiltPosition -150
    #        vcc3_setFocusSpeed 7
    #        vcc3_setAERef 72
    #
    #   and have it work correctly; note that this makes the individual syncs
    #   such as startMove redundant....
    return [vcc3_waitSync "response"]

}

# Internal function
#
# the str passed to this function should not include the length field or
#   the checksum field
# 
proc vcc3_sendString { str } {
    global serialFd_

    set len [string length $str]
    # add 1 for the checksum still to be calculated
    set len [expr $len + 1]
    set str [binary format "ca*" $len $str]
    
    set checkSum [computeChecksum $str]
    set str [binary format "a*c" $str $checkSum]

    if {[catch {puts -nonewline $serialFd_ $str} result]} {
	puts stderr $result
	return -code error "Error: unable to write to VCC3"
    }
    if {[catch {flush $serialFd_} result]} {
	puts stderr $result
	return -code error "Error: unable to write to VCC3"
    }
}

# Internal Function (?)
proc vcc3_setCallback {procName} {
    global serialFd_

    if {[catch {fileevent $serialFd_ readable "$procName"} result]} {
	return -code error "Error: unable to register VCC3 callback: $result"
    }
}

# FIXME - I'm not sure what a good timeout val is, because if the pan speed is
#   low, it can take a long time to get to the place, without error
#
# For now, put a really high value and assume there will be no errors (haven't
#   had any yet)
proc vcc3_waitSync {val {timeout 40000}} {
    global g_sync

    set sleepTime 50
    set timeSlept 0

#    puts "vcc3_waitSync: g_sync is [array get g_sync]"
#    puts "vcc3_waitSync: val is $val"

    if {$timeout == "forever"} {
	while {$g_sync($val) == "WAIT"} {
	    sleep $sleepTime
	}
	return 1
    }
    
    while {($g_sync($val) == "WAIT") && ($timeSlept < $timeout)} {
	sleep $sleepTime
	set timeSlept [expr $timeSlept + $sleepTime]
#	puts stdout "vcc3_waitSync: timeSlept is $timeSlept; timeout is $timeout"
    }
    if {$g_sync($val) == "WAIT"} {
	# we timed out
	puts stdout "vcc3_waitSync: timed out on $val"
	return 0
    } else {
	return 1
    }
}



############################generic functions############################

# returns an 8-bit integer in the range 0-255, or "" if nothing is available
#   from serialFd_
#
# this routine blocks until a byte is read!
proc getByte {} {
    set retVal [getCharBlocking]
    set ascVal [charToInt $retVal]
    return $ascVal
}

# this takes a string and converts the first character in it to an integer
#   in the range 0-255
#
# if the string is empty, returns an empty string
#
proc charToInt { ch } {
    set ascVal ""
    set numConv [binary scan $ch "c" ascVal]
#    puts stdout "num Conv = $numConv; ascVal = $ascVal"
    if {$numConv == 1} {
	if {$ascVal < 0} {
	    # this is necessary because Tcl can't handle unsigned chars
	    set ascVal [expr $ascVal + 256]
	}
    }
    return "$ascVal"
}

# by default, reads one character and returns it as a string
#
# can specify more characters to read
#
# this is non-blocking, so if the buffer is empty, will return ""
proc getChar { {numToRead 1} } {
    global serialFd_

    if {[catch {read $serialFd_ $numToRead} retVal]} {
	return ""
    } else {
	set len [string length $retVal]
#	puts stdout "read $len, expected $numToRead:     $retVal"
	return "$retVal"
    }
}

proc getCharBlocking { {numToRead 1} } {
    global serialFd_

    set retVal ""

    while {$numToRead} {
	if {[catch {read $serialFd_ $numToRead} readStr]} {
	    return ""
	} else {
	    set len [string length $readStr]
#puts stdout "len=$len, numToRead=$numToRead"
	    set numToRead [expr $numToRead - $len]
	    append retVal $readStr
	    if {$numToRead == 0} {
#		puts stdout "getCharBlocking: returning $retVal"
		return $retVal
	    }		
	}
#	puts stdout "getCharBlocking: waiting for available character"
	after 100
    }
#    puts stdout "getCharBlocking: returning $retVal"
    return $retVal
}

# takes a string and computes the checksum
#
# the checksum is 255 - (the sum of all the ascii character values, (0-255),
#    mod 255)
#
proc computeChecksum {str} {
    set size [string length $str]
    set sum 0
    for {set x 0} {$x < $size} {incr x 1} {
	set tmp [charToInt [string index $str $x]]
	set sum [expr $sum + $tmp]
    }
    set sum [expr $sum % 256]
    set sum [expr 256 - $sum]
    return $sum
}

