/*
   debug.c: a handy utility to view registers & memory for NetWinder
   Copyright 1998  woody@corelcomputer.com

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include <unistd.h>

#define	DEB_BYTE	0
#define	DEB_WORD	1
#define	DEB_DWORD	2

#define	CMD_DUMP	'd'
#define	CMD_EDIT	'e'
#define	CMD_INPUT	'i'
#define	CMD_OUTPUT	'o'
#define	CMD_SUPER_IN	"si"
#define	CMD_SUPER_OUT	"so"
#define	CMD_CPLD_READ	"cr"
#define	CMD_CPLD_WRITE	"cw"

#define VERSION		"1.1"
#define outb(p, v) \
  tmpSuperIO = v; \
  lseek(f_mem, p+0xE0000000, SEEK_SET); \
  write(f_mem, & tmpSuperIO, 1);

#define inb(p) \
  (lseek(f_mem, p+0xE0000000, SEEK_SET), \
  read(f_mem, & tmpSuperIO, 1), \
  tmpSuperIO)

#define UNLOCK    outb(0x370, 0x87); outb(0x370, 0x87);
#define LOCK      outb(0x370, 0xAA);

extern int errno;

int tmpSuperIO;

void 
db (int f_mem, char *bdata, int bcount, char *buff)
{
  int bposition;
  int i;
  char btemp;

  if (!bcount)
    return;

  bposition = (int) bdata & 0x0F;

  for (i = 0; i < 77; i++)
    *(buff + i) = ' ';

  *(buff + 77) = 0;

  *(buff + 59) = '[';
  *(buff + 76) = ']';

  i = sprintf (buff, "%08X", (unsigned int) bdata & 0xFFFFFFF0);
  *(buff + i) = ' ';

  i = lseek (f_mem, (unsigned int) bdata, SEEK_SET);
  if (i != (unsigned int) bdata)
    printf ("Memory access error!\n");

  while (bcount--)
    {
      i = read (f_mem, &btemp, 1);
      bdata++;

      if (0 >= i)
	btemp = '?';

      sprintf (buff + 10 + 3 * bposition, "%02X", btemp);
      *(buff + 10 + 3 * bposition + 2) = ' ';

      if ((btemp < 0x20) || (btemp >= 0x7F))
	btemp = '.';
      *(buff + 60 + bposition) = btemp;

      bposition++;
      if (bposition == 16)
	{
	  *(buff + 9 + 4 * 3) = '.';
	  *(buff + 9 + 4 * 3 + 4 * 3) = '-';
	  *(buff + 9 + 4 * 3 + 4 * 3 + 4 * 3) = '.';
	  printf ("%s\n", buff);

	  for (i = 0; i < 77; i++)
	    *(buff + i) = ' ';

	  *(buff + 77) = 0;

	  *(buff + 59) = '[';
	  *(buff + 76) = ']';

	  i = sprintf (buff, "%08X", (unsigned int) bdata & 0xFFFFFFF0);
	  *(buff + i) = ' ';
	  bposition = 0;
	}
    }

  if (bposition)
    {
      *(buff + 9 + 4 * 3) = '.';
      *(buff + 9 + 4 * 3 + 4 * 3) = '-';
      *(buff + 9 + 4 * 3 + 4 * 3 + 4 * 3) = '.';
      printf ("%s\n", buff);
    }
}


void 
dd (int f_mem, unsigned int *ddata, int dcount, char *buff)
{
  int dposition;
  int i;
  char btemp;
  unsigned int dtemp;

  if (!dcount)
    return;

  if ((unsigned int) ddata & 3)	//align to a dword boundry, if necessary

    (unsigned int) ddata &= 0xFFFFFFFC;

  dposition = ((unsigned int) ddata >> 2) & 0x03;

  for (i = 0; i < 77; i++)
    *(buff + i) = ' ';

  *(buff + 77) = 0;

  *(buff + 59) = '[';
  *(buff + 76) = ']';

  i = sprintf (buff, "%08X", (unsigned int) ddata & 0xFFFFFFF0);
  *(buff + i) = ' ';


  i = lseek (f_mem, (unsigned int) ddata, SEEK_SET);
  if (i != (unsigned int) ddata)
    printf ("Memory access error!\n");

  while (dcount--)
    {
//          dtemp = *ddata++;
      i = read (f_mem, &dtemp, 4);
      ddata++;

      sprintf (buff + 11 + 12 * dposition, "%08X", dtemp);
      *(buff + 11 + 12 * dposition + 8) = ' ';

      //print byte chars inside assuming Little Endian
      btemp = dtemp & 0xFF;
      if ((btemp < 0x20) || (btemp >= 0x7F))
	btemp = '.';
      *(buff + 60 + dposition * 4) = btemp;

      dtemp >>= 8;
      btemp = dtemp & 0xFF;
      if ((btemp < 0x20) || (btemp >= 0x7F))
	btemp = '.';
      *(buff + 60 + dposition * 4 + 1) = btemp;

      dtemp >>= 8;
      btemp = dtemp & 0xFF;
      if ((btemp < 0x20) || (btemp >= 0x7F))
	btemp = '.';
      *(buff + 60 + dposition * 4 + 2) = btemp;

      dtemp >>= 8;
      btemp = dtemp & 0xFF;
      if ((btemp < 0x20) || (btemp >= 0x7F))
	btemp = '.';
      *(buff + 60 + dposition * 4 + 3) = btemp;

      dposition++;
      if (dposition == 4)
	{
	  *(buff + 9 + 4 * 3) = '.';
	  *(buff + 9 + 4 * 3 + 4 * 3) = '.';
	  *(buff + 9 + 4 * 3 + 4 * 3 + 4 * 3) = '.';
	  printf ("%s\n", buff);

	  for (i = 0; i < 77; i++)
	    *(buff + i) = ' ';

	  *(buff + 77) = 0;

	  *(buff + 59) = '[';
	  *(buff + 76) = ']';

	  i = sprintf (buff, "%08X", (unsigned int) ddata & 0xFFFFFFF0);
	  *(buff + i) = ' ';
	  dposition = 0;
	}
    }

  if (dposition)
    {
      *(buff + 9 + 4 * 3) = '.';
      *(buff + 9 + 4 * 3 + 4 * 3) = '.';
      *(buff + 9 + 4 * 3 + 4 * 3 + 4 * 3) = '.';
      printf ("%s\n", buff);
    }
}

int
main (int argc, char **argv)
{
  char textline[256];
  char kbuff[128];
  char *ptext;
  void *deb_addr = &textline[0];
  int f_mem;
  int deb_mode = DEB_BYTE;
  int i;
  int last_cmd = 'd';
  int temp;
  int in_edit = 0;
  int dtemp;
  int ldev;
  int j;
  int btemp;

  printf ("\nNetwinder Low Level Debug Tool V.%s, Woody, Corel Computer 1998.\n", VERSION);

  if (getuid() != 0)
    {
      printf ("This program must be run as root!\n\n");
      return EXIT_FAILURE;
    }

  f_mem = open ("/dev/mem", O_RDWR);
  ptext = &textline[0];
  UNLOCK;

  while (1)
    {
      fflush (stdin);
      fprintf (stdout, "\n->");
      fgets (kbuff, 127, stdin);
#if 0
      i = 0;
      do
	{
	  btemp = getchar ();
	  kbuff[i++] = (char) btemp;
	  if

	}
      while ((char) btemp != '\n');
      kbuff[i] = 0;

      printf ("kbuff=%02X %02X %02X %02X %02X %02X.\n",
	      kbuff[0],
	      kbuff[1],
	      kbuff[2],
	      kbuff[3],
	      kbuff[4],
	      kbuff[5]);

#endif

      for (i = 0; i < 127; i++)
	if (kbuff[i] != 0x20)
	  break;

      if (kbuff[i] == '\n')
	{
	  *(int *) &kbuff[i] = last_cmd;
	}

      switch (kbuff[i])
	{
	case 'e':
	case 'E':
	  i++;
	  last_cmd = CMD_EDIT;

	  if (tolower (kbuff[i]) == 'd')
	    {
	      deb_mode = DEB_DWORD;
	      i++;
	    }
	  else if (tolower (kbuff[i]) == 'w')
	    {
	      deb_mode = DEB_DWORD;
	      i++;
	    }
	  else if (tolower (kbuff[i]) == 'b')
	    {
	      deb_mode = DEB_BYTE;
	      i++;
	    }

	  i = sscanf (&kbuff[i], "%p", &deb_addr);

//          printf("edit mode %d, addr 0x%X...\n", deb_mode, deb_addr);
	  in_edit = 1;

	  while (in_edit)
	    {
	      i = lseek (f_mem, (unsigned int) deb_addr, SEEK_SET);
	      if (i != (unsigned int) deb_addr)
		printf ("Memory access error!\n");

	      if (deb_mode == DEB_BYTE)
		{
		  i = read (f_mem, &btemp, 1);

		  printf ("Byte %p: %02X -> ", deb_addr, btemp & 0xFF);

		  fgets (kbuff, 127, stdin);
		  i = sscanf (kbuff, "%02X", &temp);
		  if (i == -1)
		    {
		      in_edit = 0;
		      break;
		    }

		  i = lseek (f_mem, (unsigned int) deb_addr, SEEK_SET);
		  if (i != (unsigned int) deb_addr)
		    printf ("Memory access error!\n");

		  i = write (f_mem, &temp, 1);
		  deb_addr++;
		}
	      else if (deb_mode == DEB_DWORD)
		{
		  i = read (f_mem, &dtemp, 4);

		  printf ("Dword %p: %08X -> ", deb_addr, dtemp);
		  fgets (kbuff, 127, stdin);
		  i = sscanf (kbuff, "%08X", &temp);
		  if (i == -1)
		    {
		      in_edit = 0;
		      break;
		    }

		  i = lseek (f_mem, (unsigned int) deb_addr, SEEK_SET);
		  if (i != (unsigned int) deb_addr)
		    printf ("Memory access error!\n");

		  i = write (f_mem, &temp, 4);
		  deb_addr += 4;

		}
	    }
	  break;

	case 'd':
	case 'D':
	  i++;
	  last_cmd = CMD_DUMP;

	  if (tolower (kbuff[i]) == 'd')
	    {
	      deb_mode = DEB_DWORD;
	      i++;
	    }
	  else if (tolower (kbuff[i]) == 'w')
	    {
	      deb_mode = DEB_DWORD;
	      i++;
	    }
	  else if (tolower (kbuff[i]) == 'b')
	    {
	      deb_mode = DEB_BYTE;
	      i++;
	    }

	  i = sscanf (&kbuff[i], "%p", &deb_addr);

	  if (i == -1)
	    deb_addr += 160;

	  if (deb_addr)
	    {
	      if (deb_mode == DEB_BYTE)
		db (f_mem, (char *) deb_addr, 160, ptext);
	      else if (deb_mode == DEB_WORD)
		dd (f_mem, (unsigned int *) deb_addr, 160 / 4, ptext);
	      else if (deb_mode == DEB_DWORD)
		dd (f_mem, (unsigned int *) deb_addr, 160 / 4, ptext);
	    }
	  break;

	case 'i':
	case 'I':
	  i++;
	  last_cmd = CMD_INPUT;

	  i = sscanf (&kbuff[i], "%p", &deb_addr);

//          if (i == -1)        //if just i - keep using this port
	  //              deb_addr ++;

	  i = lseek (f_mem, (unsigned int) deb_addr + 0xE0000000, SEEK_SET);
	  if (i != (unsigned int) deb_addr + 0xE0000000)
	    printf ("Memory access error!\n");

	  i = read (f_mem, &btemp, 1);

	  i = sprintf (textline, "Port %04X", (unsigned int) deb_addr & 0xFFFF);
	  *(textline + i) = ' ';

	  sprintf (&textline[i + 1], " : %02X.\n", (unsigned char) btemp);
	  printf (textline);

	  break;

	case 'o':
	case 'O':
	  i++;
	  last_cmd = CMD_OUTPUT;

	  i = sscanf (&kbuff[i], "%04p %02X", &deb_addr, &btemp);

	  if (i != -1)
	    {
//              printf("out port %04X val %02X.\n", deb_addr, btemp);
	      i = lseek (f_mem, (unsigned int) deb_addr + 0xE0000000, SEEK_SET);

	      if (i != (unsigned int) deb_addr + 0xE0000000)
		printf ("Port access error (%04X != %04X)!\n", i & 0xFFFF, (unsigned int)deb_addr);
	      else
		{
		  i = write (f_mem, &btemp, 1);
		}
	    }
	  break;

	case 's':
	case 'S':
	  i++;
	  if (kbuff[i] == 'i' || kbuff[i] == 'I')
	    {
	      i++;
	      last_cmd = (int) CMD_SUPER_IN;

	      i = sscanf (&kbuff[i], "%02X %04p", &ldev, &deb_addr);

	      UNLOCK;
	      outb (0x370, 0x07);
	      outb (0x371, ldev);
	      outb (0x370, (unsigned int) deb_addr);
	      btemp = inb (0x371);

	      printf ("Logical dev. %02X, Reg %02X : %02X.\n",
		      ldev, (unsigned int) deb_addr & 0xFFFF, btemp);

	      break;
	    }
	  else if (kbuff[i] == 'o' || kbuff[i] == 'O')
	    {
	      i++;
	      last_cmd = (int) CMD_SUPER_OUT;

	      i = sscanf (&kbuff[i], " %X %04p %02X", &ldev, &deb_addr, &btemp);

	      printf ("Logical dev. %02X, Reg %02X : %02X.\n",
		      ldev, (unsigned int) deb_addr, btemp);

	      UNLOCK;
	      outb (0x370, 0x07);
	      outb (0x371, ldev);
	      outb (0x370, (int) deb_addr);
	      outb (0x371, (char) btemp);

	      break;
	    }
	  else
	    break;


	case 'c':
	case 'C':
	  i++;
	  if (kbuff[i] == 'r' || kbuff[i] == 'R')
	    {
	      i++;
	      last_cmd = (int) CMD_CPLD_READ;

	      ldev = open ("/dev/nwdebug", O_RDONLY);
	      if (ldev < 0)
		{
		  printf ("Error %d opening /dev/nwdebug\n", ldev);
		  break;
		}
	      else
		{
		  ioctl (ldev, 0x5D, &btemp);	//CMD_READ_CPLD

		  if (!errno)
		    printf ("CPLD: 0x%X (DS1620 rst:%s, 7111:%s, Mute:%s, Flash_wrt:N/A).\n",
			    btemp, btemp & 0x08 ? "OFF" : "ON", btemp & 0x04 ? "disabled" : "enabled", btemp & 0x02 ? "OFF" : "ON");
		  close (ldev);
		}
	      break;
	    }
	  else if (kbuff[i] == 'w' || kbuff[i] == 'W')
	    {
	      i++;
	      last_cmd = (int) CMD_CPLD_WRITE;

	      i = sscanf (&kbuff[i], " %02X", &btemp);

	      if (i != -1)
		{
		  btemp &= 0xEF;

		  ldev = open ("/dev/nwdebug", O_RDWR);
		  if (ldev < 0)
		    {
		      printf ("Error %d opening /dev/nwdebug\n", ldev);
		      break;
		    }
		  else
		    {
		      ioctl (ldev, 0x5D, &j);	//CMD_READ_CPLD

		      ioctl (ldev, 0x5E, &btemp);	//CMD_WRITE_CPLD

		      if (!errno)
			{
			  ioctl (ldev, 0x5D, &btemp);	//CMD_READ_CPLD

			  printf ("CPLD: old 0x%X, new 0x%X (DS1620 rst:%s, 7111:%s, Mute:%s, Flash_wrt:N/A).\n",
				  j, btemp, btemp & 0x08 ? "OFF" : "ON", btemp & 0x04 ? "disabled" : "enabled", btemp & 0x02 ? "OFF" : "ON");
			}
		    }
		  close (ldev);
		}
	      break;
	    }
	  else
	    break;


	case 'q':
	case 'Q':
	  LOCK;
	  close (f_mem);

	  exit (0);
	  break;

	case 'm':
	case 'M':
//printf("               ...MEMORY MAP...\n");
	  printf ("                        Debuger\n");
	  printf ("Phys Start  End         Start(2)   PCI Start(1) Contents\n");
	  printf ("\n");
	  printf ("0000.0000   01FF.FFFF   0                       32 meg SDRAM\n");
	  printf ("4000.0000   4000.FFFF                           DRAM mode registers\n");
	  printf ("4100.0000   410F.FFFF   E18x.xxxx               1 meg flash ROM\n");
	  printf ("4200.0000   420F.FFFF   E10x.xxxx               CSR\n");
	  printf ("5000.0000   5000.0FFF   E130.0xxx               StrongARM cache flush\n");
	  printf ("7800.0000   7800.0FFF   E120.0xxx               PCI write flush\n");
	  printf ("7900.0000   7900.0003   E110.0000               PCI IACK address\n");
	  printf ("7B00.0000   7BFF.FFFF   E0nn.xxxx               PCI config\n");
	  printf ("            VGA:E018..Ether100:E020,..SCSI:E030..IDE:E050..Ether10:E090..\n");
	  printf ("7C00.0000   7C00.0FFF   E000.0xxx               PCI I/O space\n");
	  printf ("7C00.xxxx               D080.xxxx               VGA config regs\n");
	  printf ("8000.0000   80FF.FFFF   D0xx.xxxx   08xx.xxxx   VGA\n");
//	  printf ("9000.0000   9000.FFFF   D100.xxxx   1000.xxxx   modem codec (optional)\n");
	  printf ("A000.0000   A000.00FF   E140.00xx   2000.00xx   Ethernet 100\n");
	  printf ("A001.0000   A001.00FF   E141.00xx   2001.00xx   SCSI\n");
	  printf ("\n");
	  printf ("(1) PCI bus masters see these addresses.\n");
	  printf ("(2) These are the addresses used by Linux code, after the memory management\n");
	  printf ("unit has been enabled. Note that the SDRAM is owned by the Linux kernel and\n");
	  printf ("is managed at address C000.0000. When memory is dished out to user code, it\n");
	  printf ("appears duplicated at 0-BFFF.FFFF in the user space.\n");

	  break;


	case '?':
	case 'h':
	case 'H':
	  printf ("Debug commands (all values in hex):\n");
	  printf ("dd addr          - dump dword\n");
	  printf ("dw addr          - dump word\n");
	  printf ("db addr          - dump byte\n");
	  printf ("ed addr          - edit dword\n");
	  printf ("ew addr          - edit word\n");
	  printf ("eb addr          - edit byte\n");
	  printf ("i  addr          - in port (byte)\n");
	  printf ("o  addr val      - out port (byte)\n");
	  printf ("si ldev reg      - superIO ldev reg in (byte)\n");
	  printf ("so ldev reg val  - superIO ldev reg out (byte)\n");
	  printf ("cr               - read CPLD (3 bits)\n");
	  printf ("cw val           - write CPLD (3 bits)\n");
	  printf ("m                - display memory map\n");

	  break;


	default:
	  break;
	}
    }

}
