/* $Id: diagnostic.c,v 1.32 2004/11/12 02:04:37 graziano Exp $ */

#include "config_portability.h"

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "diagnostic.h"
#include "osutil.h"
#include "strutil.h"

#define MAXRECORDS 2500

static FILE *diagDestinations[] =
	{DIAGSUPPRESS, DIAGSUPPRESS, DIAGSUPPRESS,
	DIAGSUPPRESS, DIAGSUPPRESS, DIAGSUPPRESS};
static long recordCounts[] =
	{0, 0, 0, 0, 0, 0};
static void *lock = NULL;			/* mutex for this modules */

/* we look quite extensively because I'm not quite sure what fprintf will
 * do if the FILE* changes on the way */
static void
PrintDiagnostic(DiagLevels level,
                const char *message,
                va_list arguments) {

	static const char *level_tags[] = {"", "", "Warning: ", "Error: ", "Fatal: ", "Debug: " };

	/* we need a lock because we write recordCounts and we use
	 * diagDestinations[] */
	if (GetNWSLock(&lock) == 0) {
		fprintf(stderr, "PrintDiagnostic: Error: Couldn't obtain the lock\n");
	}
	if(diagDestinations[level] != DIAGSUPPRESS) {
		if( (recordCounts[level]++ >= MAXRECORDS) &&
				(diagDestinations[level] != stdout) &&
				(diagDestinations[level] != stderr) ) {
		/* We want to avoid filling up the disk space when the
		 * system is running for weeks at a time.  It might be
		 * nice to save the old file under another name (maybe in
		 * /tmp), then reopen.  That requires changing the
		 * DirectDiagnostics() interface to take a file name
		 * instead of a FILE *.  */
			rewind(diagDestinations[level]);
			recordCounts[level] = 0;
		}

		fprintf(diagDestinations[level], "%.0f %d ", CurrentTime(), (int)getpid());
		fprintf(diagDestinations[level], "%s", level_tags[level]);
		vfprintf(diagDestinations[level], message, arguments);
		fflush(diagDestinations[level]);
	}
	if (ReleaseNWSLock(&lock) == 0) {
		fprintf(stderr, "PrintDiagnostic: Error: Couldn't release the lock\n");
	}
}

static void
BasicToggle (DiagLevels level) {
	static FILE *old[] =
		{DIAGSUPPRESS, DIAGSUPPRESS, DIAGSUPPRESS,
		 DIAGSUPPRESS, DIAGSUPPRESS, DIAGSUPPRESS};
	FILE *pretoggle;

	pretoggle = DiagnosticsDirection(level);
	if ((old[level] != stderr) && (old[level] != stdout) &&
				(old[level] != DIAGSUPPRESS)) {
		rewind(old[level]);
		(void)ftruncate(fileno(old[level]), 0);
	}
	DirectDiagnostics(level, old[level]);
	old[level] = pretoggle;
}



void
ToggleDiagnostics(	int toggleErrors,
			int toggleLogs) {
	if (toggleErrors) {
		BasicToggle(DIAGWARN);
		BasicToggle(DIAGERROR);
		BasicToggle(DIAGFATAL);
	}
	if (toggleLogs) {
		BasicToggle(DIAGDEBUG);
		BasicToggle(DIAGINFO);
		BasicToggle(DIAGLOG);
	}
}

void
DirectDiagnostics(DiagLevels level,
                  FILE *whereTo) {

#ifdef HAVE_FILENO
	int f = 0;

	/* just an extra check */
	if (whereTo != NULL) {
		f = fileno(whereTo);
	} 
	if (f < 0) {
		fprintf(stderr, "DirectDiagnostic: fileno failed\n");
		return;
	}
#endif
	GetNWSLock(&lock);
	diagDestinations[level] = whereTo;
	ReleaseNWSLock(&lock);
}

void
SetDiagnosticLevel(	int verbose,
			FILE *errFD,
			FILE *logFD) {
	FILE *tmpErr, *tmpLog;

	if (errFD != NULL) {
		tmpErr = errFD;
	} else {
		tmpErr = stderr;
	}
	if (logFD != NULL) {
		tmpLog = logFD;
	} else {
		tmpLog = stdout;
	}

	/* fatal errors are always on */
	DirectDiagnostics(DIAGFATAL, tmpErr);
	DirectDiagnostics(DIAGDEBUG, DIAGSUPPRESS);
	DirectDiagnostics(DIAGINFO, DIAGSUPPRESS);
	DirectDiagnostics(DIAGLOG, DIAGSUPPRESS);
	DirectDiagnostics(DIAGWARN, DIAGSUPPRESS);
	DirectDiagnostics(DIAGERROR, DIAGSUPPRESS);
	if (verbose > 5) {
		printf("Verbose level too high: set to 5\n");
		verbose = 5;
	}
	switch (verbose) {
	case 5:
		DirectDiagnostics(DIAGDEBUG, tmpLog);
	case 4:
		DirectDiagnostics(DIAGINFO, tmpLog);
	case 3:
		DirectDiagnostics(DIAGLOG, tmpLog);
	case 2:
		DirectDiagnostics(DIAGWARN, tmpErr);
	case 1:
		DirectDiagnostics(DIAGERROR, tmpErr);
	}
}


FILE *
DiagnosticsDirection(DiagLevels level) {
	return diagDestinations[level];
}


void
PositionedDiagnostic(DiagLevels level,
                     const char *fileName, 
                     int line,
                     const char *message,
                     ...) {

	va_list arguments;
	char *extendedMessage;

	/* we assume that NWS lines won't be past 10 digits */
	extendedMessage = (char *)MALLOC(strnlen(fileName, MAX_FILENAME_LENGTH)
				+ strnlen(message, MAX_MESSAGE_LENGTH) + 11);

	/* we need to va_start here because alpha is very unhappy to have
	 * NULL as va_list */
	va_start(arguments, message);
	if (extendedMessage == NULL) {
		/* out of memory */
		PrintDiagnostic(DIAGERROR, "PositionedDiagnostic: out of memory", arguments);
	} else {
		sprintf(extendedMessage, "%s:%d %s", fileName, line, message);
		PrintDiagnostic(level, extendedMessage, arguments);
		free(extendedMessage);
	}
	va_end(arguments);
}


void
Diagnostic(DiagLevels level,
           const char *message,
           ...) {
	va_list arguments;

	va_start(arguments, message);
	PrintDiagnostic(level, message, arguments);
	va_end(arguments);
}
