/* $Id: nws_ctrl.c,v 1.42 2004/12/06 20:05:44 graziano Exp $ */

#include "config_nws.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#if TIME_WITH_SYS_TIME
#	include <sys/time.h>
#	include <time.h>
#else
#	if HAVE_SYS_TIME_H
#		include <sys/time.h>
#	else
#		include <time.h>
#	endif
#endif

#include "diagnostic.h"
#include "messages.h"
#include "host_protocol.h"
#include "protocol.h"
#include "strutil.h"
#include "osutil.h"
#include "skills.h"
#include "clique_protocol.h"
#include "register.h"

#define NWSAPI_SHORTNAMES
#include "nws_api.h" 

typedef enum {DEAD, HEALTHY, SICK, UNRESPONSIVE} HostStatii;

/* local function that prints the status of a host (in cookie) */
static void
PrintStatus(	struct host_cookie *cookie,
		HostStatii stat,
		HostInfo *info) {
	const char *TYPES[] = {"forecaster", "memory", "name_server", "sensor"};
	const char *STATII[] = {"dead", "healthy", "sick", "unresponsive"};

	if (cookie == NULL) {
		return;
	}

	printf("%s:", HostCImage(cookie));
	/* if we have the info, we don't need stat */
	if (info != NULL) {
		printf(" %s ", TYPES[info->hostType]);
		printf("(believes to be %s) ", info->registrationName);
		printf("registered with %s ", info->nameServer);
	}
	printf(" %s\n", STATII[stat]);
}

static int
FillCookie(	struct host_cookie *cookie,
		const char *host,
		HostInfo *info,
		double timeout) {
	/* sanity check */
	if (cookie == NULL || host == NULL || info == NULL) {
		ABORT("FillCookie: NULL parameter\n");
	}

	Host2Cookie(host, NWSAPI_GetDefaultPort(NWSAPI_SENSOR_HOST), cookie);
	if (!ConnectToHost(cookie, NULL)) {
		PrintStatus(cookie, DEAD, NULL);
		return 0;
	}
	if (!GetHostInfo(cookie, info, timeout)) {
		PrintStatus(cookie, UNRESPONSIVE, NULL);
		return 0;
	}

	return 1;
}

static void
BurpXML(	FILE *f,
		const char *src,
		const char *dst,
		const char *res,
		const char *opts,
		NWSAPI_ForecastCollection *col) {

	/* sanity check */
	if (src == NULL || res == NULL || col == NULL) {
		ERROR("BurpXML: NULL parameters!\n");
		return;
	}

	/* let's print the seriesForecast stuff */
	fprintf(f, "   <seriesForecast>\n");
	fprintf(f, "\t<source>%s</source>\n", src);
	if (dst != NULL) {
		fprintf(f, "\t<destination>%s</destination>\n", dst);
	}
	fprintf(f, "\t<resource>%s</resource>\n", res);
	if (opts != NULL) {
		fprintf(f, "\t<options>%s</options>\n", opts);
	}

	/* if timestamp == 0 we don't have resonable values */
	if (col->measurement.timeStamp == 0) {
		fprintf(f, "   </seriesForecast>\n");
		return;
	}

	/* now let's print the forecastCollection */
	fprintf(f, "\t<forecastCollection>\n");
	fprintf(f, "\t\t<measurement>\n");
	fprintf(f, "\t\t\t<timestamp>%.0f</timestamp>\n", col->measurement.timeStamp);
	fprintf(f, "\t\t\t<value>%.9f</value>\n", col->measurement.measurement);
	fprintf(f, "\t\t</measurement>\n");
	fprintf(f, "\t\t<forecast minimize=\"absolute\">\n");
	fprintf(f, "\t\t\t<value>%.9f</value>\n", col->forecasts[NWSAPI_MAE_FORECAST].forecast);
	fprintf(f, "\t\t\t<error>%.9f</error>\n", col->forecasts[NWSAPI_MAE_FORECAST].error);
	fprintf(f, "\t\t\t<method>%s</method>\n", NWSAPI_MethodName(col->forecasts[NWSAPI_MAE_FORECAST].methodUsed));
	fprintf(f, "\t\t</forecast>\n");
	fprintf(f, "\t\t<forecast minimize=\"square\">\n");
	fprintf(f, "\t\t\t<value>%.9f</value>\n", col->forecasts[NWSAPI_MSE_FORECAST].forecast);
	fprintf(f, "\t\t\t<error>%.9f</error>\n", col->forecasts[NWSAPI_MSE_FORECAST].error);
	fprintf(f, "\t\t\t<method>%s</method>\n", NWSAPI_MethodName(col->forecasts[NWSAPI_MSE_FORECAST].methodUsed));
	fprintf(f, "\t\t</forecast>\n");
	fprintf(f, "\t</forecastCollection>\n");
	fprintf(f, "   </seriesForecast>\n");
}

static void
PrintForecast(	FILE *f,
		const char *src,
		const char *dst,
		const char *res,
		const char *opts,
		NWSAPI_ForecastCollection *col,
		int title) {
	/* let's see if we need to just write the title */
	if (title) {
		fprintf(f, "%-15s ", "Resource");
		fprintf(f, "%-15s ", "Source");
		if (dst != NULL) {
			fprintf(f, "%-15s ", "Dest");
		}
		fprintf(f, "%-12s ", "Time");
		fprintf(f, "%-12s ", "Measurement");
		fprintf(f, "%-14s ", "Forecast (MAE)");
		fprintf(f, "%-11s ", "Error (MAE)");
		fprintf(f, "%-30s ", "Methor (MAE)");
		fprintf(f, "%-14s ", "Forecast (MSE)");
		fprintf(f, "%-11s ", "Error (MSE)");
		fprintf(f, "%-30s\n", "Method (MSE)");
		return;
	}

	/* print the enciladas */
	fprintf(f, "%-15s ", res);
	fprintf(f, "%-15s ", src);
	if (dst != NULL) {
		fprintf(f, "%-15s ", dst);
	}
	if (opts != NULL) {
		fprintf(f, "%s", opts);
	}

	/* if timestamp == 0 we don't have resonable values */
	if (col->measurement.timeStamp == 0) {
		fprintf(f, "N/A\n");
		return;
	}

	/* now let's print the forecastCollection */
	fprintf(f, "%-12.0f ", col->measurement.timeStamp);
	fprintf(f, "%12.2f ", col->measurement.measurement);
	fprintf(f, "%14.2f ", col->forecasts[NWSAPI_MAE_FORECAST].forecast);
	fprintf(f, "%11.2f ", col->forecasts[NWSAPI_MAE_FORECAST].error);
	fprintf(f, "%-30s ", NWSAPI_MethodName(col->forecasts[NWSAPI_MAE_FORECAST].methodUsed));
	fprintf(f, "%14.2f ", col->forecasts[NWSAPI_MSE_FORECAST].forecast);
	fprintf(f, "%11.2f ", col->forecasts[NWSAPI_MSE_FORECAST].error);
	fprintf(f, "%-30s\n", NWSAPI_MethodName(col->forecasts[NWSAPI_MSE_FORECAST].methodUsed));
}

#define SWITCHES "s:P:t:Vv:f:XF"

static void
usage() {
	printf("\nUsage: nws_ctrl [OPTIONS] command host [command-options]\n");
	printf("Control the hosts for the Network Weather Service\n");
	printf("\nOPTIONS can be:\n");
	printf("\t-t timeout          specify a timeout\n");
	printf("\t-X                  print result from proxy cmd in XML\n");
	printf("\t-P password         the password of the remote nwsHost\n");
	printf("\t-F                  force an operation\n");
	printf("\t-f file             read host and command options from #file#\n");
	printf("\t-s size,buffer,msg  instruct ping to use these values\n");
	printf("\t-v level            verbose level (up to 5)\n");
	printf("\t-V                  print version\n");
	printf("\nand commands can be:\n");
	printf("\ttest                query #host#: accepts multiple hosts as options\n");
	printf("\tregister            #host# will change nameserver to #command-options#\n");
	printf("\tmemory              #host# will change memory to #command-options#\n");
	printf("\thalt                halt #host#\n");
	printf("\tlog                 toggle logs on #host#: accept optional error and log\n");
	printf("\tskill               ask #host# to run a remote skill (just once)\n");
	printf("\tadd                 ask #host# to add a sensor to a clique\n");
	printf("\tremove              ask #host# to remove a sensor from a clique\n");
	printf("\tping                probe band & lat to the #host#s\n");
	printf("\tstart               start an activity on #host#: you need to specify name:<activity> controlName:<control> skillName:<skill>. Others attributes are optional.\n");
	printf("\tstop                stop an <activityName> on #host#\n");
	printf("\ttime                translate seconds from epoch to friendlier time\n");
	printf("\tproxy               #host# is the proxy and the first options is the resource, the reminder are the hosts\n");
	printf("\tproxyFetch          #host# is the proxy and the first options is the resource, the reminder are the hosts\n");
	printf("Report bugs to <nws@nws.cs.ucsb.edu>.\n\n");
}

int
main(		int argc,
		char *argv[]) {
	const char *command; 
	char tmp[255], *commandLine, *ptr;
	const char *password;
	struct host_cookie cookie;
	int c, sizes[3], z, inXML, force;
	double tout;
	unsigned short verbose;
	extern char *optarg;
	extern int optind;
	HostInfo info;
	FILE *optionsFile;

	/* deafult values */
	password = NULL;
	commandLine = strdup("");;
	optionsFile = NULL;
	tout = -1;
	verbose = force = 0;
	sizes[0] = 256;		/* experiment size */
	sizes[1] = 32;		/* buffer size */
	sizes[2] = 16;		/* message size */
	inXML = 0;

	while((c = getopt(argc, argv, SWITCHES)) != EOF) {
		switch(c) {
		case 'P':
			password = optarg;
			break;

		case 'F':
			force = 1;
			break;

		case 'X':
			inXML = 1;
			break;

		case 't':
			tout = strtod(optarg, NULL);
			break;

		case 'f':
			optionsFile = fopen(optarg, "r");
			if (optionsFile == NULL) {
				printf("Cannot open file %s\n", optarg);
				exit(1);
			}
			break;

		case 's':
			/* Parse the comma-delimited size list, allowing
			 * each element to default if unspecified (e.g.
			 * "32", "10,,5" and ",,8" are all legal). */
			command = optarg;
			for (z = 0; z < 3 && *command != '\0'; z++) {
				if (*command != ',') {
					if (GETTOK(tmp, command, ",", &command)) {
						sizes[z] = (int)strtol(tmp, NULL, 10);
					}
				}
				if (*command != '\0') {
					command++;
				}
			}
			break;

		case 'v':
			verbose = (unsigned short)atol(optarg);
			break;

		case 'V':
			printf("nws_ctrl for NWS version %s", VERSION);
#ifdef HAVE_PTHREAD_H
			printf(", with thread support");
#endif
#ifdef WITH_DEBUG
			printf(", with debug support");
#endif
			printf("\n\n");
			exit(0);
			break;

		case '?':
			if (optopt == 'v') {
				/* using basic level */
				verbose = 2;
				break;
			}
			/* non recognized options: printing help */

		default:
			usage();
			exit(1);
			break;
		}
	}

	/* set the verbose level */
	SetDiagnosticLevel(verbose, stderr, stderr);

	/* now the next word should be the command */
	if (optind >= argc) {
		usage();
		exit(1);
	} 

	/* this is the command */
	command = strdup(argv[optind++]);
	if (command == NULL) {
		ABORT("out of memory\n");
	}

	/* now let's collect all the host/comamnds from command line/file */
	for (; optind < argc; optind++) {
		commandLine = REALLOC(commandLine, strlen(commandLine) + strlen(argv[optind]) + 2);
		if (commandLine == NULL) {
			ABORT("Out of memory\n");
			exit(1);
		}
		strcat(commandLine, argv[optind]);
		strcat(commandLine, "\n");
	}
	if (optionsFile != NULL) {
		/* read the file now */
		while (fgets(tmp, sizeof(tmp), optionsFile) != NULL) {
			/* remove empty lines */
			if (strlen(tmp) <= 1) {
				continue;
			}
			commandLine = REALLOC(commandLine, strlen(commandLine) + strlen(tmp) + 2);
			if (commandLine == NULL) {
				ABORT("Out of memory\n");
				exit(1);
			}
			strcat(commandLine, tmp);
			strcat(commandLine, "\n");
		}
		fclose(optionsFile);
	}
	if (commandLine[0] == '\0') {
		ABORT("There is no host/options specified\n");
		exit(1);
	}
	ptr = commandLine;

	if (strcmp(command, "test") == 0) {
		/* run a test command */
		for (; GETWORD(tmp, ptr, (const char **)&ptr); ) {
			/* try to connect to the host */
			if (FillCookie(&cookie, tmp, &info, tout)) {
				if (info.healthy) {
					PrintStatus(&cookie, HEALTHY, &info);
				} else {
					PrintStatus(&cookie, SICK, &info);
				}
				DisconnectHost(&cookie);
			}
		}
	} else if (strcmp(command, "testSeries") == 0) {
		NWSAPI_HostSpec nsSpec;
		Object seriesobject;
		ObjectSet seriesset, activityset;
		int optcount, filterLen;
		char *filter, *value, *name;

		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			ABORT("Must specify nameserver and seriesname\n");
		} else {
			nsSpec = *NWSAPI_MakeHostSpec(tmp, NWSAPI_GetDefaultPort(NWSAPI_NAME_SERVER_HOST));

			if(!NWSAPI_UseNameServer(&nsSpec)) {
				ABORT2("Unable to contact name server %s:%d\n", nsSpec.machineName, nsSpec.machinePort);
			}
		}

		for (optcount = 0; GETWORD(tmp, ptr, (const char **)&ptr); ) {
			optcount++;
			filterLen = 40 + strlen(tmp);
			filter = malloc(sizeof(char) * filterLen);
			sprintf(filter, "(&(objectclass=nwsSeries)(name=%s*))", tmp);
			/* let's get the series */
			if(!NWSAPI_GetObjectsTimeout(filter, &seriesset, tout)) {
				ABORT("error getting objects\n");
			}
			FREE(filter);

			/* nothing back: we don't know about this series */
			if (strlen(seriesset) <= 1) {
				FreeObjectSet(&seriesset);
				printf("%s: unknown\n", tmp);
				continue;
			} 

			/* let's gp through all the series we found */
			NWSAPI_ForEachObject(seriesset, seriesobject) {
				name = NwsAttributeValue_r(FindNwsAttribute(seriesobject, "name"));
				value = NwsAttributeValue_r(FindNwsAttribute(seriesobject, "activity"));
				if (value == NULL) {
					printf("%s: stale\n", name);
					FREE(name);
					continue;
				}

				filterLen = strlen(value) + 44;
				filter = malloc(sizeof(char) * filterLen);
				sprintf(filter, "(&(objectclass=nwsActivity)(name=%s))", value);
		 
				if (!NWSAPI_GetObjectsTimeout(filter, &activityset, tout)) {
					ABORT("error getting objects\n");
				}
				FREE(filter);

				if (strlen(activityset) > 1) {
					printf("%s: current\n", name);
				} else {
					printf("%s: orphan\n", name);
				}
				FREE(name);
				FREE(value);
				FreeObjectSet(&activityset);
			}
		}
		if (!optcount) {
			ABORT("Need to specify seriesname as option\n");
		}
	} else if (strcmp(command, "register") == 0) {
	        if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a nameserver\n");
			exit(1);
		}
		if (!HostChangeNameServer(&cookie, tmp, tout)) {
			printf("failed to change nameserver\n");
			exit(1);
		}
		DisconnectHost(&cookie);
	} else if (strcmp(command, "memory") == 0) {
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a memory\n");
			exit(1);
		}
		if (!HostChangeMemory(&cookie, tmp, tout)) {
			printf("failed to change memory\n");
			exit(1);
		}
		DisconnectHost(&cookie);
	} else if (strcmp(command, "halt") == 0) {
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a memory\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}
		if (!HostHalt(&cookie, password, tout)) {
			printf("failed to halt the host\n");
			exit(1);
		}
		DisconnectHost(&cookie);
	} else if (strcmp(command, "log") == 0) {
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}
		if (GETWORD(tmp, ptr, (const char **)&ptr)) {
			if (strcmp("error", tmp) == 0) {
				c = ALL_ERRORS;
			} else if (strcmp("log", tmp) == 0) {
				c = ALL_LOGS;
			}
		} else {
			c = ALL_DIAGNOSTICS;
		}
		if (!HostChangeDiagnostic(&cookie, c, tout)) {
			printf("failed to change diagnostics on the host\n");
			exit(1);
		}
		DisconnectHost(&cookie);
	} else if (strcmp(command, "skill") == 0) {
		char *opts;
		char **objs;
		int len;
		double d[2];

		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}
		/* turns whatever into options */
		opts = NULL;
		len = 0;
		for (; GETWORD(tmp, ptr, (const char **)&ptr); ) {
			opts = (char*)REALLOC(opts, len + strlen(tmp) + 2);
			if (opts == NULL) {
				ABORT("out of memory\n");
			}
			if (len != 0) {
				strcat(opts, "\t");
				len++;
			} else {
				opts[0] = '\0';
			}
			strcat(opts, tmp);
			len += strlen(tmp);
		}
		if (len == 0) {
			printf("I need at least a skill\n");
			exit(1);
		}

		/* simple check: skillName gotta be here */
		if (strstr(opts, "skillName") == NULL) {
			printf("no attribute \"skillName\" found!\n");
			exit(2);
		}

		if (!RemoteUseSkill(&cookie, opts, tout, &objs, d, &len)) {
			printf("failed to use skill on %s\n", HostCImage(&cookie));
			exit(2);
		}
		for (len--; len >= 0; len--) {
			printf("%s %s: %.2f\n", HostCImage(&cookie), objs[len], d[len]);
			free(objs[len]);
		}
		free(opts);
		DisconnectHost(&cookie);
	} else if (strcmp(command, "add") == 0 || strcmp(command, "remove") == 0) {
		char *name, *host;

		/* ask a sensor to add a new member to a clique */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}

		/* find the clique name */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a clique name\n");
			exit(1);
		}
		name = strdup(tmp);
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a sensor\n");
			exit(1);
		}
		host = strdup(tmp);
		if (name == NULL || host == NULL) {
			printf("out of memory\n");
			exit(2);
		}

		if (strcmp(command, "add") == 0) {
			/* ask the sensor to add a new member */
			if (!CliqueJoin(&cookie, name, host, tout)) {
				printf("failed to add sensor to clique\n");
				exit(2);
			}
		} else {
			/* ask the sensor to add a new member */
			if (!CliqueRemove(&cookie, name, host, tout)) {
				printf("failed to remove sensor from clique\n");
				exit(2);
			}
		}
		free(name);
		free(host);
		DisconnectHost(&cookie);
	} else if (strcmp(command, "start") == 0) {
		/* let's start a new activity */
		NWSAPI_HostSpec s;
		NWSAPI_ObjectSet objs;
		char *attr, *name, *control;

		printf("not yet implemented\n");
		exit(2);

		/* let's get the host */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host \n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}

		/* let's get the controlName */
		control = NWSAPI_NwsAttributeValue_r(NWSAPI_FindNwsAttribute(ptr, "controlName"));
		if (control == NULL) {
			printf("I need a controlName\n");
			exit(1);
		}

		/* let's get the activity name */
		name = NWSAPI_NwsAttributeValue_r(NWSAPI_FindNwsAttribute(ptr, "controlName"));
		if (name == NULL) {
			printf("I need an activity name\n");
			exit(1);
		}

		/* let's get the skillName */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a skill\n");
			exit(1);
		}

		/* let's build the list of options we understand: there
		 * are control and skill options */

		/* let's check if the activity is already registered */


	} else if (strcmp(command, "stop") == 0) {
		NWSAPI_HostSpec s;
		NWSAPI_ObjectSet objs;
		char filter[255 + 1];

		/* let's create a NWSAPI_HostSpec for the sensor */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host \n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}
		/* but first let's set the nameserver */
		s = *NWSAPI_MakeHostSpec(info.nameServer, NWSAPI_GetDefaultPort(NWSAPI_NAME_SERVER_HOST));
		if (!NWSAPI_UseNameServer(&s)) {
			printf("failed to set nameserver\n");
		}

		/* let's get the activity */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need an activity \n");
			exit(1);
		}

		/* let's see if we have such an activity registered */
		snprintf(filter, 256, "(name=%s)", tmp);
		objs = NWSAPI_NewObjectSet();

		if (!NWSAPI_GetObjectsTimeout(filter, &objs, tout) ||
				NWSAPI_FirstObject(objs) == NULL) {
			printf("failed to find activity on the nameserver: trying to stop it anyway\n");
		}

		/* is host a nameserver? */
		if (info.hostType == NAME_SERVER_HOST) {
			NWSAPI_Object obj;
			char *hostName;

			obj = NWSAPI_FirstObject(objs);
			hostName = NWSAPI_NwsAttributeValue_r(NWSAPI_FindNwsAttribute(obj, "host"));
			if (hostName == NULL) {
				printf("failed to find sensor\n");
				exit (2);
			}
			s = *NWSAPI_MakeHostSpec(hostName, NWSAPI_GetDefaultPort(NWSAPI_SENSOR_HOST));
			free(hostName);
		} else {
			s = *NWSAPI_MakeHostSpec(cookie.name, cookie.port);
		}

		NWSAPI_FreeObjectSet(&objs);

		/* let's stop it */
		if (!NWSAPI_HaltActivity(&s, tmp)) {
			printf("failed to stop activty %s\n", tmp);
			exit(2);
		}
	} else if (strcmp(command, "ping") == 0) {
		/* let's try to ping an host */
		SkillResult *res = NULL;
		int len, j;
		char opts[255];

		for (; GETWORD(tmp, ptr, (const char **)&ptr); ) {
			/* let's fill in the options: target ... */
			sprintf(opts, "target:%s", tmp);
			/* size, buffer and message sizes in kB */
			strcat(opts, "\tsize:");
			sprintf(opts + strlen(opts), "%d", sizes[0]);
			strcat(opts, "\tbuffer:");
			sprintf(opts + strlen(opts), "%d", sizes[1]);
			strcat(opts, "\tmessage:");
			sprintf(opts + strlen(opts), "%d", sizes[2]);

			/* let's print the sizes and the target */
			printf("(%dk,%dk,%dk) ", sizes[0], sizes[1], sizes[2]);
			printf("to %s: ", tmp);

			UseSkill(tcpMessageMonitor, opts, tout, &res, &len);
			for(j = 0; j < len; j++) {
				printf("%s: ", ResourceName(res[j].resource));
				if (!res[j].succeeded) {
			              printf("failed ");
				      continue;
				}
				printf("%.4f %s ", res[j].measurement, ResourceLabel(res[j].resource));
			} 
			printf("\n");
			FreeSkillResults(len, &res);
		}
	} else if (strcmp(command, "time") == 0) {
		time_t t;
		char s[256];

		for (; GETWORD(tmp, ptr, (const char **)&ptr); ) {
			t = (int)strtol(tmp, (char **)NULL, 10);
			strftime(s, 256, "%d-%m-%Y/%H:%M:%S", localtime(&t));
			printf("%s\n", s);
		}
		fflush(stdout);
	} else if (!strcmp(command, "proxy") || !strcmp(command, "proxyFetch")) {
		NWSAPI_HostSpec proxy;
		char **hosts, *res;
		int howMany, ind, j, interMachine;
		NWSAPI_ForecastCollection *col;

		/* first argument is the proxy */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			ABORT("I need a proxy\n");
		}
		proxy = *NWSAPI_MakeHostSpec(tmp, NWSAPI_GetDefaultPort(NWSAPI_PROXY_HOST));

		/* second argument is the resource */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			ABORT("I need a resource\n");
		}
		res = strdup(tmp);
		if (res == NULL) {
			ABORT("out of memory\n");
		}
		if (NWSAPI_IsResourceValid(res) == -1) {
			ABORT1("Unknown resource (%s)\n", res);
		}

		/* turns the hosts into char ** */
		hosts = NULL;
		for (c = 0; GETWORD(tmp, ptr, (const char **)&ptr); c++) {
			hosts = (char **)realloc(hosts, sizeof(char *)*(c + 2));
			if (hosts == NULL) {
				ABORT("out of memory\n");
			}
			hosts[c] = (char *)strdup(tmp);
			if (hosts[c] == NULL) {
				ABORT("out of memory\n");
			}
			hosts[c + 1] = NULL;
		}
		if (c == 0) {
			ABORT("Need to specify at least one target host!\n");
		}
		howMany = c;

		/* now we do different things if we need to retrieve data
		 * or just tells the proxy to start fetching */
		if (!strcmp(command, "proxy")) {
			/* let's call the proxy */
			if (NWSAPI_GetAllForecasts(&proxy, res, NULL, hosts, &col, tout) == 0) {
				ABORT("Error in GetAllForecasts\n");
			}
			interMachine = NWSAPI_IntermachineResource(res);
			if (inXML) {
				fprintf(stdout, "<seriesCollection>\n");
			} else {
				/* let's print title */
				PrintForecast(stdout, 
						NULL, 
						(char *)interMachine, 
						NULL,
						NULL,
						NULL,
						1);
			}
			for (ind = 0, c = 0; c < howMany; c++) {
				for (j = 0; j < howMany; j++) {
					char *dst;
	
					/* only interMachine resources (band/lat)
					 * returns matrices */
					if (!interMachine && j > 0) {
						break;
					}
					if (!interMachine) {
						dst = NULL;
					} else {
						dst = hosts[j];
					}
					/* skip the same hosts */
					if (c != j || !interMachine) {
						if (inXML) {
							BurpXML(stdout, 
								hosts[c],
								dst,
								res,
								NULL,
								&col[ind]);
						} else {
							PrintForecast(stdout, 
								hosts[c],
								dst,
								res,
								NULL,
								&col[ind],
								0);
						}
					}
					ind++;
				}
			}
			if (inXML) {
				fprintf(stdout, "</seriesCollection>\n");
			} else {
				fprintf(stdout, "\n");
			}
			FREE(col);
			FREE(res);
		} else {
			if (!NWSAPI_StartFetching(&proxy, res, NULL, hosts, tout)) {
				ERROR("failed to start fetching\n");
			}
		}
/*		while(howMany > 0) {
			FREE(hosts[--howMany]);
		}*/
		/* FREE(hosts) */
	} else {
		/* unknown command */
		usage();
		exit(1);
	}

	return(0);
}
