/*
 * Copyright (c) 1996 University College London
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Computer Science
 *      Department at University College London
 * 4. Neither the name of the University nor of the Department may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "prototypes.h"

#ifndef MAXINT
#define MAXINT 0x8fffffff
#endif

static struct timeval offset;	/* offset used to store offset between actual internal clock */
										/* and the clock being used to synchronise with other hosts. */

void init_time()
{
    offset.tv_sec = 0;
    offset.tv_usec = 0;
}

void get_cur_time(struct timeval *t)
{
    struct timezone tzp;
    struct timeval tmp;
    gettimeofday(&tmp, &tzp);
    add_time(&offset, &tmp, t);
    debug("offset: %d\n", offset.tv_sec);
}

void clock_sync(struct timeval *t)
{
    struct timeval mytime;
    get_cur_time(&mytime);
    if (newer(t, &mytime)) {
	struct timeval tdiff;

	diff_time(t, &mytime, &tdiff);	/* tdiff now holds the time interval t - mytime */

	if ((tdiff.tv_sec == 0) && (tdiff.tv_usec < 10000))
	    return;

	debug("Syncing clock: was %u,%6.6u, now %u,%6.6u\n",
	      mytime.tv_sec, mytime.tv_usec, t->tv_sec, t->tv_usec);

	debug("Offset was: %u,%6.6u\n", offset.tv_sec, offset.tv_usec);
	debug("Increasing time by %u,%6.6u\n", tdiff.tv_sec, tdiff.tv_usec);

	add_time(&tdiff, &offset, &offset);

	debug("Offset now: %u,%6.6u\n", offset.tv_sec, offset.tv_usec);
	if (offset.tv_sec > 100) {
		debug("Offset is too large... how should we fix this?\n");
	}
    }
}

int newer(struct timeval *t1, struct timeval *t2)
{
    return (((t1->tv_sec) > (t2->tv_sec)) ||
    (((t1->tv_sec) == (t2->tv_sec)) && ((t1->tv_usec) > (t2->tv_usec))));
}

void time_copy(struct timeval *t1, struct timeval *t2)
{
    t2->tv_sec = t1->tv_sec;
    t2->tv_usec = t1->tv_usec;
    return;
}

void htontime(struct timeval *t1, struct timeval *t2)
{
    t2->tv_sec = htonl(t1->tv_sec);
    t2->tv_usec = htonl(t1->tv_usec);
    return;
}

void ntohtime(struct timeval *t1, struct timeval *t2)
{
    t2->tv_sec = ntohl(t1->tv_sec);
    t2->tv_usec = ntohl(t1->tv_usec);
    return;
}

unsigned int subtract_time(struct timeval *t1, struct timeval *t2)
{
    unsigned int msdiff, sdiff, diff, max_sdiff, max_int;
    if ((t2->tv_usec) > (t1->tv_usec)) {
	/*we need to carry a second */
	sdiff = t1->tv_sec - t2->tv_sec - 1;
	msdiff = ((1000000 - (t2->tv_usec)) + (t1->tv_usec)) / 1000;
    } else {
	msdiff = (((t1->tv_usec) - (t2->tv_usec)) / 1000);
	sdiff = ((t1->tv_sec) - (t2->tv_sec));
    }

/* commented out to try something else to cure compiler warning
   if((sdiff)>(MAXINT/1000)) 
   { 
   sdiff=MAXINT/1000;
   } */

    /* this value replaces the constant MAXINT defined in nt.h */
    max_int = 0x8fffffff;

    max_sdiff = max_int / 1000;

    if (sdiff > max_sdiff) {
	sdiff = max_sdiff;
    }
    /* end of new version - jim. */

    diff = ((sdiff * 1000) + msdiff);
    return diff;
}

void add_time(struct timeval *t1, struct timeval *t2,
	      struct timeval *tr)
{
    tr->tv_usec = t1->tv_usec + t2->tv_usec;
    if (tr->tv_usec >= 1000000) {
	tr->tv_usec -= 1000000;
	tr->tv_sec = t1->tv_sec + t2->tv_sec + 1;
    } else {
	tr->tv_sec = t1->tv_sec + t2->tv_sec;
    }
}

void diff_time(struct timeval *t1, struct timeval *t2,
	       struct timeval *tr)
{
    if ((t2->tv_usec) > (t1->tv_usec)) {
	/*we need to carry a second */
	tr->tv_sec = t1->tv_sec - t2->tv_sec - 1;
	tr->tv_usec = (1000000 - (t2->tv_usec)) + (t1->tv_usec);
    } else {
	tr->tv_usec = (t1->tv_usec) - (t2->tv_usec);
	tr->tv_sec = t1->tv_sec - t2->tv_sec;
    }
}
