/* Support for calibrated display output
 */

#include "ip.h"

/* Default VIPS monitor model ... 6 is my Ultra2.
 */
#define MDEFAULT (6)

/* Our base calibrate struct. Copy this to get per-display calibrate
 * structs.
 */
CalibrateInfo *calibrate_preferences;

/* Init calibrate. Can't do this with a structure init, sadly. Called from
 * main.c. May be overridden later.
 */
void
init_calibrate( void )
{
	if( !(calibrate_preferences = IM_NEW( NULL, CalibrateInfo )) )
		error( our_resources.bad_memory );

	calibrate_preferences->ref = 1;
	calibrate_preferences->table = NULL;
	if( !(calibrate_preferences->dummy = im_open( "dummy1", "p" )) )
		error( our_resources.bad_memory );

	if( !our_resources.icc_profile || 
		strcmp( our_resources.icc_profile, "" ) == 0 )
		/* Fallback to this.
		 */
		calibrate_preferences->disp = *im_col_displays[ MDEFAULT ];
	else {
		if( call_string( (call_string_fn) im_ICC2display,
			our_resources.icc_profile, 
			&calibrate_preferences->disp, NULL ) ) {
			verrors( "Unable to load ICC profile \"%s\"\n"
				"Using default profile",
					our_resources.icc_profile );
			box_alert( NULL );
			calibrate_preferences->disp = 
				*im_col_displays[MDEFAULT];
		}
	}
}

/* Take a copy of a calibrate_info, ready for editing.
 */
static CalibrateInfo *
dup_cal_info( CalibrateInfo *old )
{
	CalibrateInfo *cal;

	if( !(cal = IM_NEW( NULL, CalibrateInfo )) ) {
		errors( our_resources.bad_memory );
		return( NULL );
	}

	cal->ref = 1;
	cal->disp = old->disp;
	if( !(cal->dummy = im_open( "dummy", "p" )) ) {
		errors( our_resources.bad_memory );
		FREE( cal );
		return( NULL );
	}
	cal->table = NULL;

	return( cal );
}

/* Copy a calibrate_info.
 */
CalibrateInfo *
copy_cal_info( CalibrateInfo *cal )
{
	cal->ref++;
	return( cal );
}

/* Free a calibrate_info.
 */
void
free_cal_info( CalibrateInfo *cal )
{
	cal->ref--;
	if( cal->ref <= 0 ) {
		if( cal->dummy ) {
			im_close( cal->dummy );
			cal->dummy = NULL;
			cal->table = NULL;
		}
		FREE( cal );
	}
}

/* Build Lab->disp tables, if necessary.
 */
Bool
build_lab_tables( CalibrateInfo *cal )
{
	/* Build VIPS tables.
	 */
	if( !cal->table ) {
		set_hourglass();

		cal->table = (void *) im_LabQ2disp_build_table( cal->dummy, 
			&cal->disp );
		if( !cal->table ) {
			set_pointer();
			errors( our_resources.bad_memory );
			return( False );
		}

		set_pointer();
	}

	return( True );
}

/* Structure we hold calibrate preferences widgets in.
 */
typedef struct calibrate_widgets {
	DialogInfo *dia;		/* Handle for dialog */

	CalibrateInfo *cal;		/* Prefs we are editing */
	struct display *scr;		/* Display we affect, if any */
	CalibrateInfo **pt;		/* Pointer to store new prefs to */

	/* Fields in im_col_display we edit 
	 */
	Widget YC[3];			
	Widget name;
	Widget Y0[3];
	Widget Vrw[3];
	Widget gamma[3];
	Widget CW[3];
	Widget mat[3][3];
	Widget type;			/* OptionMenu */
	Widget items[2];		/* Items on menu */
} CalibratePrefInfo;

/* Help button hit.
 */
/*ARGSUSED*/
static void
cal_pref_help_cb( DialogInfo *dia, void *client, notify_fn nfn, void *sys )
{
	/* Pop help.
	 */
	box_help( dia_shell( dia ), "calpref" );

	nfn( sys, DIALOG_OK );
}

/* Fill the fields of a calibration preferences editor.
 */
static void
fill_out_cal_prefs( CalibratePrefInfo *st )
{	
	int i, j;

	for( i = 0; i < 3; i++ )
		for( j = 0; j < 3; j++ ) 
			set_text_double( st->mat[j][i], 
				st->cal->disp.d_mat[j][i] );

	set_texts( st->name, st->cal->disp.d_name );
	set_optionset( st->type, st->items, st->cal->disp.d_type );

	set_text_double( st->YC[0], st->cal->disp.d_YCR );
	set_text_double( st->YC[1], st->cal->disp.d_YCG );
	set_text_double( st->YC[2], st->cal->disp.d_YCB );

	set_text_double( st->Y0[0], st->cal->disp.d_Y0R );
	set_text_double( st->Y0[1], st->cal->disp.d_Y0G );
	set_text_double( st->Y0[2], st->cal->disp.d_Y0B );

	set_text_int( st->Vrw[0], st->cal->disp.d_Vrwr );
	set_text_int( st->Vrw[1], st->cal->disp.d_Vrwg );
	set_text_int( st->Vrw[2], st->cal->disp.d_Vrwb );

	set_text_double( st->gamma[0], st->cal->disp.d_gammaR );
	set_text_double( st->gamma[1], st->cal->disp.d_gammaG );
	set_text_double( st->gamma[2], st->cal->disp.d_gammaB );

	set_text_double( st->CW[0], st->cal->disp.d_xCW );
	set_text_double( st->CW[1], st->cal->disp.d_yCW );
	set_text_double( st->CW[2], st->cal->disp.d_YCW );
}

/* Done button hit.
 */
/*ARGSUSED*/
static void
cal_pref_done_cb( DialogInfo *dia, void *client, notify_fn nfn, void *sys )
{
	decl( CalibratePrefInfo *, st, client );
	CalibrateInfo *cal = dup_cal_info( st->cal );
	int i, j;

	/* Write new prefs to our copy.
	 */
	for( i = 0; i < 3; i++ )
		for( j = 0; j < 3; j++ ) 
			if( !get_text_float( st->mat[j][i], 
				&cal->disp.d_mat[j][i] ) ) {
				nfn( sys, DIALOG_ERROR );
				return;
			}

	cal->disp.d_name = XmTextGetString( st->name );
	cal->disp.d_type = get_optionset( st->type, st->items, 2 );

	if( !get_text_float( st->YC[0], &cal->disp.d_YCR ) ||
		!get_text_float( st->YC[1], &cal->disp.d_YCG ) ||
		!get_text_float( st->YC[2], &cal->disp.d_YCB ) ) {
		errors( "YC: not a float" );
		nfn( sys, DIALOG_ERROR );
		return;
	}

	if( !get_text_float( st->Y0[0], &cal->disp.d_Y0R ) ||
		!get_text_float( st->Y0[1], &cal->disp.d_Y0G ) ||
		!get_text_float( st->Y0[2], &cal->disp.d_Y0B ) ) {
		errors( "Y0: not a float" );
		nfn( sys, DIALOG_ERROR );
		return;
	}

	if( !get_text_float( st->gamma[0], &cal->disp.d_gammaR ) ||
		!get_text_float( st->gamma[1], &cal->disp.d_gammaG ) ||
		!get_text_float( st->gamma[2], &cal->disp.d_gammaB ) ) {
		errors( "gamma: not a float" );
		nfn( sys, DIALOG_ERROR );
		return;
	}

	if( !get_text_float( st->CW[0], &cal->disp.d_xCW ) ||
		!get_text_float( st->CW[1], &cal->disp.d_yCW ) ||
		!get_text_float( st->CW[2], &cal->disp.d_YCW ) ) {
		errors( "CW: not a float" );
		nfn( sys, DIALOG_ERROR );
		return;
	}

	if( !get_text_int( st->Vrw[0], &cal->disp.d_Vrwr ) ||
		!get_text_int( st->Vrw[1], &cal->disp.d_Vrwg ) ||
		!get_text_int( st->Vrw[2], &cal->disp.d_Vrwb ) ) {
		errors( "Vrw: not a float" );
		nfn( sys, DIALOG_ERROR );
		return;
	}

	/* New prefs set up ok. Free old prefs, install new prefs. Rebuild
	 * tables.
	 */
	free_cal_info( st->cal );
	*st->pt = cal;

	/* If this calibrate_info is associated with a display, redraw it.
	 * Have to remake the display conversion too, to make sure we pick up
	 * the new tables.
	 */
	if( st->scr ) {
		if( remake_display_conversion( st->scr ) ) {
			verrors( "unable to remake display conversion" );
			nfn( sys, DIALOG_ERROR );
			return;
		}
		/*
		note_pending_total_repaint( st->scr );
		 */
	}

	nfn( sys, DIALOG_OK );
}

/* Make a 3x3 matrix of test widgets.
 */
static void
build_mat( Widget par, char *title, Widget mat[3][3] )
{
	Widget form2, rc;
	int i, j;
	Arg args[ MAX_ARGS ];
	int n;

	n = 0;
	form2 = build_labeledframe( par, title, args, n );
	n = 0;
	XtSetArg( args[n], XmNpacking, XmPACK_COLUMN ); n++;
	XtSetArg( args[n], XmNorientation, XmVERTICAL ); n++;
	XtSetArg( args[n], XmNnumColumns, 3 ); n++;
	XtSetArg( args[n], XmNadjustLast, False ); n++;
	rc = XmCreateRowColumn( form2, "mat_rc", args, n );
	XtManageChild( rc );

	for( i = 0; i < 3; i++ )
		for( j = 0; j < 3; j++ ) {
			n = 0;
			XtSetArg( args[n], 
				XmNeditMode, XmSINGLE_LINE_EDIT ); n++;
			XtSetArg( args[n], XmNcolumns, 8 ); n++;
			mat[j][i] = XmCreateText( rc, "mat_text", args, n );
			XtManageChild( mat[j][i] );
		}
}

/* Make three text input widgets, with labels as given, and save their 
 * widget ids in the array we are given.
 */
static void
build_rgb( Widget par, char *title, char *names[3], Widget *rgb )
{
	Widget label, form, text;
	Arg args[ MAX_ARGS ];
	int n;

	/* Label and frame for rgb widgets.
	 */
	n = 0;
	form = build_labeledframe( par, title, args, n );

	n = 0;
	XtSetArg( args[n], XmNleftAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[n], XmNrightPosition, 16 ); n++;
	XtSetArg( args[n], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNbottomAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNalignment, XmALIGNMENT_END ); n++;
	label = XmCreateLabelGadget( form, names[0], args, n );
	XtManageChild( label );
	n = 0;
	XtSetArg( args[n], XmNleftAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[n], XmNleftWidget, label ); n++;
	XtSetArg( args[n], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[n], XmNrightPosition, 32 ); n++;
	XtSetArg( args[n], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNbottomAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNeditMode, XmSINGLE_LINE_EDIT ); n++;
	XtSetArg( args[n], XmNcolumns, 5 ); n++;
	rgb[0] = text = XmCreateText( form, "red_text", args, n );
	XtManageChild( rgb[0] );

	n = 0;
	XtSetArg( args[n], XmNleftAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[n], XmNleftWidget, text ); n++;
	XtSetArg( args[n], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[n], XmNrightPosition, 48 ); n++;
	XtSetArg( args[n], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNbottomAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNalignment, XmALIGNMENT_END ); n++;
	label = XmCreateLabelGadget( form, names[1], args, n );
	XtManageChild( label );
	n = 0;
	XtSetArg( args[n], XmNleftAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[n], XmNleftWidget, label ); n++;
	XtSetArg( args[n], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[n], XmNrightPosition, 66 ); n++;
	XtSetArg( args[n], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNbottomAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNeditMode, XmSINGLE_LINE_EDIT ); n++;
	XtSetArg( args[n], XmNcolumns, 5 ); n++;
	rgb[1] = text = XmCreateText( form, "green_text", args, n );
	XtManageChild( rgb[1] );

	n = 0;
	XtSetArg( args[n], XmNleftAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[n], XmNleftWidget, text ); n++;
	XtSetArg( args[n], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[n], XmNrightPosition, 83 ); n++;
	XtSetArg( args[n], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNbottomAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNalignment, XmALIGNMENT_END ); n++;
	label = XmCreateLabelGadget( form, names[2], args, n );
	XtManageChild( label );
	n = 0;
	XtSetArg( args[n], XmNleftAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[n], XmNleftWidget, label ); n++;
	XtSetArg( args[n], XmNrightAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNbottomAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNeditMode, XmSINGLE_LINE_EDIT ); n++;
	XtSetArg( args[n], XmNcolumns, 5 ); n++;
	rgb[2] = text = XmCreateText( form, "blue_text", args, n );
	XtManageChild( rgb[2] );
}

/* Build the insides of a calibrate preferences editor.
 */
static void
build_cal_pref_edit( DialogInfo *dia, Widget form, CalibratePrefInfo *st )
{	
	/* Widget names we build.
	 */
	static char *rgb[3] = { "red_label", "green_label", "blue_label" };
	static char *xyY[3] = { "x_label", "y_label", "Y_label" };

	Widget label, rc, text, om, rc2;
	Arg args[ MAX_ARGS ];
	int n;

	/* Name.
	 */
	n = 0;
	XtSetArg( args[n], XmNleftAttachment, XmATTACH_NONE ); n++;
	XtSetArg( args[n], XmNrightAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNbottomAttachment, XmATTACH_NONE ); n++;
	XtSetArg( args[n], XmNeditMode, XmSINGLE_LINE_EDIT ); n++;
	st->name = text = XmCreateText( form, "name_text", args, n );
	XtManageChild( text );

	n = 0;
	XtSetArg( args[n], XmNleftAttachment, XmATTACH_NONE ); n++;
	XtSetArg( args[n], XmNrightAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[n], XmNrightWidget, text ); n++;
	XtSetArg( args[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET ); n++;
	XtSetArg( args[n], XmNtopWidget, text ); n++;
	XtSetArg( args[n], XmNbottomAttachment, 
		XmATTACH_OPPOSITE_WIDGET ); n++;
	XtSetArg( args[n], XmNbottomWidget, text ); n++;
	label = XmCreateLabelGadget( form, "display_name", args, n );
	XtManageChild( label );

	/* Type.
	 */
	n = 0;
	rc2 = build_menupane( form, "type_pd", args, n, XmMENU_PULLDOWN );
	st->items[0] = XmCreatePushButtonGadget( rc2, "disp_barco", NULL, 0 );
	XtManageChild( st->items[0] );
	st->items[1] = XmCreatePushButtonGadget( rc2, "disp_dumb", NULL, 0 );
	XtManageChild( st->items[1] );

	n = 0;
	XtSetArg( args[n], XmNleftAttachment, XmATTACH_NONE ); n++;
	XtSetArg( args[n], XmNrightAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNtopAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[n], XmNtopWidget, text ); n++;
	XtSetArg( args[n], XmNbottomAttachment, XmATTACH_NONE ); n++;
	XtSetArg( args[n], XmNsubMenuId, rc2 ); n++;
	XtSetArg( args[n], XmNcolormap, our_colormap ); n++;
	XtSetArg( args[n], XmNvisual, our_visual ); n++;
	XtSetArg( args[n], XmNdepth, our_depth ); n++;
	st->type = om = XmCreateOptionMenu( form, "type_om", args, n );
	XtManageChild( om );

	n = 0;
	XtSetArg( args[n], XmNleftAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNrightAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[n], XmNtopAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[n], XmNtopWidget, om ); n++;
	XtSetArg( args[n], XmNbottomAttachment, XmATTACH_FORM ); n++;
	rc = XmCreateRowColumn( form, "main_rc", args, n );
	XtManageChild( rc );

	/* Make easy parts.
	 */
	build_rgb( rc, "white_colour", xyY, st->CW );
	build_rgb( rc, "rwhiteop_label", rgb, st->YC );
	build_rgb( rc, "rblackop_label", rgb, st->Y0 );
	build_rgb( rc, "pwhite_label", rgb, st->Vrw );
	build_rgb( rc, "gamma_label", rgb, st->gamma );
	build_mat( rc, "matrix_label", st->mat );

	/* Make contents.
	 */
	fill_out_cal_prefs( st );
}

/* Pop up a calibrate preferences editor. If scr is NULL, then we are setting
 * the defaults.
 */
void
pop_cal_pref( Widget par, CalibrateInfo **pt, struct display *scr )
{	
	CalibratePrefInfo *st = IM_NEW( NULL, CalibratePrefInfo );
	char buf[ 256 ];

	/* Make dialog.
	 */
	st->cal = *pt;
	st->pt = pt;
	st->scr = scr;
	if( scr )
		im_snprintf( buf, 256, our_resources.tit_conversion, 
			MODEL( scr->sym )->name );
	else
		im_snprintf( buf, 256, our_resources.tit_default_conversion );

	st->dia = run_dialog( par, buf, False, False, DIALOG_MESSAGE,
		cal_pref_done_cb, dia_true_cb, cal_pref_help_cb, NULL, st, 
		(build_fn) build_cal_pref_edit, st, NULL, NULL,
		null_notify_fn, NULL );
}
