/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>

#include <netdude/nd_debug.h>
#include <netdude/nd_macros.h>
#include <netdude/nd_globals.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_protocol.h>
#include <netdude/nd_raw_protocol.h>
#include <netdude/nd_protocol_registry.h>

typedef struct nd_proto_registry
{
  GHashTable           *hash;
} ND_ProtoRegistry;

typedef struct nd_proto_reg_callback_data
{
  ND_ProtocolFunc       callback;
  void                 *user_data;
} ND_ProtoRegCallbackData;

typedef struct nd_proto_key
{
  guint32               magic;
  ND_ProtocolLayer      layer;
  gboolean              is_duplicate;
} ND_ProtoKey;

/* I need only a single one right now, but this'll
   make life easier in case I ever need more... */
static ND_ProtoRegistry  _registry;
static ND_ProtoRegistry  *registry = &_registry;

/*
static gint
proto_registry_strcasecmp(gconstpointer p1, gconstpointer p2)
{
  return g_strcasecmp((const gchar*)p1, (const gchar *)p2) == 0;
}*/

static gint
proto_registry_cmp(gconstpointer p1, gconstpointer p2)
{
  ND_ProtoKey *key1, *key2;

  key1 = (ND_ProtoKey *) p1;
  key2 = (ND_ProtoKey *) p2;

  return (key1->magic == key2->magic && (key1->layer & key2->layer));
}


static guint
proto_registry_hash(gconstpointer p1)
{
  ND_ProtoKey *key = (ND_ProtoKey *) p1;

  return key->magic;
}


void           
nd_proto_registry_init(void)
{
  registry->hash = g_hash_table_new(proto_registry_hash, proto_registry_cmp);
}


gboolean       
nd_proto_registry_register(ND_Protocol *proto)
{
  int i;
  ND_ProtoKey *key;


  D_ASSERT_PTR(proto);
  
  if (!proto)
    return FALSE;

  for (i = 0; i < ND_PROTO_MAGICS; i++)
    {
      if (proto->magic[i] > 0)
	{
	  if (nd_proto_registry_find(proto->layer, proto->magic[i])->id !=
	      nd_raw_proto_get()->id)
	    return FALSE;
	  
	  D(("Registering protocol '%s' for layer %i with magic 0x%.4x.\n",
	     proto->name, proto->layer, (guint) proto->magic[i]));

	  key = g_new0(ND_ProtoKey, 1);
	  key->magic = proto->magic[i];
	  key->layer = proto->layer;
	  key->is_duplicate = (i != 0);
	  
	  g_hash_table_insert(registry->hash, key, proto);
	}
    }


  nd_gui_proto_menu_register(proto);

  return TRUE;
}


void
nd_proto_registry_unregister(ND_Protocol *proto)
{
  ND_ProtoKey key;
  int i;

  if (!proto)
    return;
  
  for (i = 0; i < ND_PROTO_MAGICS; i++)
    {
      if (proto->magic[i] > 0) 
	{
	  key.magic = proto->magic[i];
	  key.layer = proto->layer;
	  g_hash_table_remove(registry->hash, &key);
	}
    }
}


ND_Protocol   *
nd_proto_registry_find(ND_ProtocolLayer layer, guint64 magic)
{
  ND_ProtoKey  key;
  ND_Protocol *proto = NULL;

  key.magic = magic;
  key.layer = layer;

  proto = (ND_Protocol *) g_hash_table_lookup(registry->hash, &key);
  if (proto)
    return proto;
  
  D(("Protocol 0x%.4x at layer %i not found, returning raw impl.\n",
     (int) magic, layer));
  proto = nd_raw_proto_get();

  return proto;
}


guint          
nd_proto_registry_size(void)
{
  return g_hash_table_size(registry->hash);
}


static void
proto_registry_foreach_cb (gpointer	key,
			   gpointer	value,
			   gpointer	user_data)
{
  ND_ProtoKey *pkey = (ND_ProtoKey *) key;
  ND_ProtoRegCallbackData *data = (ND_ProtoRegCallbackData *) user_data;
  ND_Protocol *proto = (ND_Protocol *) value;

  if (!pkey || (pkey && !pkey->is_duplicate))
    data->callback(proto, data->user_data);
  
  return;
  TOUCH(key);
}


void           
nd_proto_registry_foreach_proto(ND_ProtocolFunc callback,
				void *user_data)
{
  ND_ProtoRegCallbackData data;

  if (!callback)
    return;

  data.callback = callback;
  data.user_data = user_data;

  g_hash_table_foreach(registry->hash,
		       proto_registry_foreach_cb,
		       &data);

  proto_registry_foreach_cb(NULL, nd_raw_proto_get(), &data);
}

