///////////////////////////////////////////////////////////////////////////////
// $Id: mainwindow.cpp,v 1.11 2004/10/03 19:30:45 krake Exp $
//
// Package:   MOAGG Edit - Level Editor for MOAGG
// Copyright: Kevin Krammer, 2003
//
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
///////////////////////////////////////////////////////////////////////////////

/*! \file    mainwindow.cpp
    \author  Kevin Krammer, kevin.krammer@gmx.at
    \brief   Implementation of the main window class
*/

// Qt includes
#include <qaction.h>
#include <qdir.h>
#include <qfiledialog.h>
#include <qfileinfo.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qsettings.h>
#include <qtabwidget.h>

// local includes
#include "mainwindow.h"

// editors
#include "mapeditor.h"

///////////////////////////////////////////////////////////////////////////////

const QString MainWindow::BASE_CAPTION = "MOAGG Editor";

///////////////////////////////////////////////////////////////////////////////

MainWindow::MainWindow(QWidget* parent, const char* name)
	: MainWindowBase(parent, name)
{
	setCaption(BASE_CAPTION);
	m_settings = new QSettings();
	m_settings->beginGroup("/moaggedit");

	QObject::connect(m_tabs, SIGNAL(currentChanged(QWidget*)),
	                 this, SLOT(slotTabChanged()));

	QObject::connect(m_btnMapPath, SIGNAL(clicked()),
	                 this, SLOT(slotBrowseMapDirectory()));
	QObject::connect(m_btnTilesPath, SIGNAL(clicked()),
	                 this, SLOT(slotBrowseTilesDirectory()));

	QObject::connect(m_actGrid, SIGNAL(toggled(bool)), m_tabs, SLOT(update()));

	readSettings();
}

///////////////////////////////////////////////////////////////////////////////

MainWindow::~MainWindow()
{
	writeSettings();
	delete m_settings;
}

///////////////////////////////////////////////////////////////////////////////

QString MainWindow::mapDirectory() const
{
	return m_edMapPath->text();
}

///////////////////////////////////////////////////////////////////////////////

QString MainWindow::tilesDirectory() const
{
	return m_edTilesPath->text();
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotNew()
{
	MapEditor* editor = new MapEditor(this, m_tabs);
	if (editor->newDocument())
	{
		editor->showGrid(m_actGrid->isOn());
		m_editors.append(editor);
		m_tabs->addTab(editor, tr("untitled"));
		m_tabs->showPage(editor);
		setCaption(BASE_CAPTION + " - " + tr("untitled"));
	}
	else
	{
		delete editor;
	}
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotOpen()
{
	QString filter = tr("MOAGG mapfile (*.map);;All files (*)");
	QString filename = QFileDialog::getOpenFileName(mapDirectory(), filter, this);
	if (filename.isEmpty()) return;

	MapEditor* editor = new MapEditor(this, m_tabs);
	if (editor->openDocument(filename))
	{
		editor->showGrid(m_actGrid->isOn());
		m_editors.append(editor);
		m_tabs->addTab(editor, tabCaption(filename));
		m_tabs->showPage(editor);
		setCaption(BASE_CAPTION + " - " + filename);
	}
	else
	{
		QMessageBox::critical(this, tr("Error"),
		                      QString(tr("Could not open file %0")).arg(filename),
		                      QMessageBox::Ok, 0);
		delete editor;
	}
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotClose()
{
	if (m_editors.count() == 0) return;
	if (m_tabs->currentPageIndex() < 1) return;

	Editor* editor = m_editors.at(m_tabs->currentPageIndex()-1);
	if (editor == 0) return;

	bool discard = false;
	while (editor->isModified() && !discard)
	{
		QString message = tr("Closing editor window which has unsaved data");
		int button = QMessageBox::warning(this, tr("Unsaved changes"),
		                                  message,
		                                  tr("Save"), tr("Discard"), tr("Cancel"),
		                                  0, 2);
		switch (button)
		{
			case 0:
				slotSave();
				break;

			case 1:
				discard = true;
				break;

			case 2: return; // cancel

			default:
				break;
		}
	}

	m_tabs->removePage(m_tabs->currentPage());
	m_editors.removeRef(editor);

	delete editor;
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotSave()
{
	if (m_editors.count() == 0) return;
	if (m_tabs->currentPageIndex() < 1) return;

	Editor* editor = m_editors.at(m_tabs->currentPageIndex()-1);
	if (editor == 0) return;

	if (editor->fileName().isEmpty())
	{
		slotSaveAs();
	}
	else if (!editor->saveDocument())
	{
		QString message = tr("Could not save file %0");
		QMessageBox::critical(this, tr("Error"),
		                      message.arg(editor->fileName()),
		                      QMessageBox::Ok, 0);
	}
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotSaveAs()
{
	if (m_editors.count() == 0) return;
	if (m_tabs->currentPageIndex() < 1) return;

	Editor* editor = m_editors.at(m_tabs->currentPageIndex()-1);
	if (editor == 0) return;
	QString filter = editor->fileFilter();

	QString filename = QFileDialog::getSaveFileName(mapDirectory(), filter, this);
	if (filename.isEmpty()) return;

	if (!editor->saveDocumentAs(filename))
	{
		QString message = tr("Could not save file %0");
		QMessageBox::critical(this, tr("Error"),
		                      message.arg(editor->fileName()),
		                      QMessageBox::Ok, 0);
	}
	else
	{
		QWidget* page = m_tabs->currentPage();
		m_tabs->setTabLabel(page, tabCaption(filename));
		setCaption(BASE_CAPTION + " - " + filename);
	}
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotSelectAll()
{
	if (m_editors.count() == 0) return;
	if (m_tabs->currentPageIndex() < 1) return;

	Editor* editor = m_editors.at(m_tabs->currentPageIndex()-1);
	if (editor == 0) return;

	editor->selectAll();
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotResize()
{
	if (m_editors.count() == 0) return;
	if (m_tabs->currentPageIndex() < 1) return;

	Editor* editor = m_editors.at(m_tabs->currentPageIndex()-1);
	if (editor == 0) return;

	editor->resize();
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotCut()
{
	if (m_editors.count() == 0) return;
	if (m_tabs->currentPageIndex() < 1) return;

	Editor* editor = m_editors.at(m_tabs->currentPageIndex()-1);
	if (editor == 0) return;

	editor->cut();
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotCopy()
{
	if (m_editors.count() == 0) return;
	if (m_tabs->currentPageIndex() < 1) return;

	Editor* editor = m_editors.at(m_tabs->currentPageIndex()-1);
	if (editor == 0) return;

	editor->copy();
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotPaste()
{
	if (m_editors.count() == 0) return;
	if (m_tabs->currentPageIndex() < 1) return;

	Editor* editor = m_editors.at(m_tabs->currentPageIndex()-1);
	if (editor == 0) return;

	editor->paste();
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotShowGrid(bool show)
{
	if (m_editors.count() == 0) return;
	if (m_tabs->currentPageIndex() < 1) return;

	Editor* editor = m_editors.at(m_tabs->currentPageIndex()-1);
	if (editor == 0) return;

	editor->showGrid(show);
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::closeEvent(QCloseEvent* event)
{
	if (queryClose())
		MainWindowBase::closeEvent(event);
	else
		event->ignore();
}

///////////////////////////////////////////////////////////////////////////////

QString MainWindow::tabCaption(const QString& filename)
{
	QFileInfo fileInfo(filename);

	return fileInfo.fileName();
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::readSettings()
{
	QString mapDir = m_settings->readEntry("/paths/maps", QDir::homeDirPath());
	m_edMapPath->setText(mapDir);

	QString tilesDir = m_settings->readEntry("/paths/tiles", QDir::homeDirPath());
	m_edTilesPath->setText(tilesDir);
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::writeSettings()
{
	m_settings->writeEntry("/paths/maps", mapDirectory());
	m_settings->writeEntry("/paths/tiles", tilesDirectory());
}

///////////////////////////////////////////////////////////////////////////////

bool MainWindow::queryClose()
{
	while (m_editors.count() > 0)
	{
		uint count = m_editors.count();

		m_tabs->setCurrentPage(1);
		slotClose();

		// if number of editors remains unchanged,
		// the tab could not be closed (unsaved changed)
		if (count == m_editors.count())
			return false;
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotBrowseMapDirectory()
{
	QString currentDir = mapDirectory();

	QFileInfo fileInfo(currentDir);
	if (!(fileInfo.exists() && fileInfo.isDir()))
		currentDir = QDir::homeDirPath();

	QString newDir =
		QFileDialog::getExistingDirectory(currentDir, this, 0,
		                                  tr("Find map directory"));
	if (!newDir.isEmpty())
	{
		m_edMapPath->setText(newDir);
	}
	else
	{
		m_edMapPath->setText(currentDir);
	}
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotBrowseTilesDirectory()
{
	QString currentDir = tilesDirectory();

	QFileInfo fileInfo(currentDir);
	if (!(fileInfo.exists() && fileInfo.isDir()))
		currentDir = QDir::homeDirPath();

	QString newDir =
		QFileDialog::getExistingDirectory(currentDir, this, 0,
		                                  tr("Find MOAGG tiles directory"));
	if (!newDir.isEmpty())
	{
		m_edTilesPath->setText(newDir);
	}
	else
	{
		m_edTilesPath->setText(currentDir);
	}
}

///////////////////////////////////////////////////////////////////////////////

void MainWindow::slotTabChanged()
{
	int pageIndex = m_tabs->currentPageIndex();

	if (pageIndex < 1)
	{
		setCaption(BASE_CAPTION);
		m_actSave->setEnabled(false);
		m_actSaveAs->setEnabled(false);
		m_actClose->setEnabled(false);
		m_actGrid->setOn(true);
	}
	else
	{
		// save actions are always enabled when an editor is active to
		// allow overwriting changes made outside this applications
		m_actSave->setEnabled(true);
		m_actSaveAs->setEnabled(true);
		m_actClose->setEnabled(true);

		Editor* editor = m_editors.at(pageIndex-1);
		if (editor != 0)
		{
			m_actGrid->setOn(editor->showsGrid());
		}

		if (editor == 0 || editor->fileName().isEmpty())
		{
			setCaption(BASE_CAPTION);
		}
		else
		{
			setCaption(BASE_CAPTION + " - " + editor->fileName());
		}
	}
}

// End of file
