#include <iostream>

#include <cppunit/TestFixture.h>
#include <cppunit/extensions/HelperMacros.h>

#include "Configuration.h"
#include "StringTools.h"
#include "StringTokenizer.h"
#include "XMLVisitors.h"
#include "Tiles.h"

#include "AllObjects.h"
#include "ObjectRepository.h"

#include "GameControlBase.h"
#include "LevelReader.h"
#include "GameBridge.h"
#include "PlayerConfiguration.h"


//----------------------------------------------------------------------------
class LevelTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(LevelTest);

    CPPUNIT_TEST(testXMLContentOfAllLevels);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
        Configuration::getInstance()->searchDataDir();

        Tiles::init();
        SurfacesBase::init();

        PlayerConfiguration::create();

        GameInterface::setInstance(GameBridge::getInstance());
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
        PlayerConfiguration::destroy();

        SurfacesBase::destroy();
        Tiles::destroy();
    }


protected:

    //------------------------------------------------------------------------
    void testXMLContentOfAllLevels()
    {
        MissionReader mr;
        LevelReader lr;

        MissionReader::const_iterator mIter;
        LevelReader::const_iterator lIter;

        try
        {
            for (mIter = mr.begin(); mIter != mr.end(); ++mIter)
            {
                lr.init(mIter->first.c_str());

                for (lIter = lr.begin(); lIter != lr.end(); ++lIter)
                {
                    GameControlBase::init(
                        lr.getMission().c_str(), lIter->c_str());
                    checkGameControlNode();
                }
            }
        }
        catch (Exception &e)
        {
            std::cerr << std::endl
                      << "Error in '" << mIter->first << "/" << *lIter
                      << "': " << e << "." << std::endl;
            throw;
        }

        GameControlBase::destroy();
    }

private:
    //------------------------------------------------------------------------
    /**
     * Visitor for plausibility checks of the <gamecontrol> content.
     */
    class CheckGameControlNodeVisitor : public XMLConstVisitor
    {
        //--------------------------------------------------------------------
        /**
         * Inner visitor to check the event handlers of each objective node.
         */
        class CheckEventHandlersVisitor : public XMLConstVisitor
        {
        public:
            //----------------------------------------------------------------
            CheckEventHandlersVisitor(CheckGameControlNodeVisitor *parent)
                    : m_parent(parent) {}
            ~CheckEventHandlersVisitor() { m_parent = NULL; }

        private:
            //----------------------------------------------------------------
            void do_visit(const XMLProperty *p)
            {
                const std::string &name = p->getName();
                std::vector<unsigned> ids;

                if (name == "show")
                {
                    m_parent->checkCrateIds(p->getValue());
                }
                else if (name == "status")
                {
                    // Nothing to check here.
                }
                else if (name == "deactivate" ||
                         name == "activate" ||
                         name == "toggle")
                {
                    m_parent->checkBarrierIds(p->getValue());
                }
                else
                {
                    CPPUNIT_FAIL("Unknown event");
                }
            }

            //----------------------------------------------------------------
            void do_visit(const XMLNode *n)
            {
                const std::string &name = n->getName();

                if (name == "onpickup" ||
                    name == "onreached")
                {
                    n->acceptAllProperties(*this);
                }
                else
                {
                    CPPUNIT_FAIL("Unknown event handler");
                }
            }

            //----------------------------------------------------------------
            CheckGameControlNodeVisitor *m_parent;
        };
        
    public:
        //--------------------------------------------------------------------
        CheckGameControlNodeVisitor() {}
        ~CheckGameControlNodeVisitor() {}

    private:
        //--------------------------------------------------------------------
        void do_visit(const XMLNode *n)
        {
            const std::string &name = n->getName();

            if (name == "pickup")
            {
                checkCrateIds(n->getStringProperty("id"));
                checkPlatformId(n->getStringProperty("platform"));
            }
            else if (name == "land")
            {
                checkPlatformIds(n->getStringProperty("platform"));
            }
            else if (name == "destroy")
            {
                checkObjectIds(n->getStringProperty("id"));
            }
            else
            {
                CPPUNIT_FAIL("Unknown objective");
            }

            CheckEventHandlersVisitor v(this);
            n->acceptAllNodes(v);
        }

        //--------------------------------------------------------------------
        void checkObjectId(const std::string &id)
        {
            ObjectBase *object =
                ObjectRepository::getInstance()->getObject(
                    STRING_TOOLS::toUnsigned(id));

            CPPUNIT_ASSERT(object != NULL);
        }

        //--------------------------------------------------------------------
        void checkCrateId(const std::string &id)
        {
            Crate *crate = dynamic_cast<Crate*>(
                ObjectRepository::getInstance()->getObject(
                    STRING_TOOLS::toUnsigned(id)));

            CPPUNIT_ASSERT(crate != NULL);
            CPPUNIT_ASSERT(crate->getType() == CrateSurfaces::T_SMALL ||
                           crate->getType() == CrateSurfaces::T_MEDIUM ||
                           crate->getType() == CrateSurfaces::T_BIG);
        }

        //--------------------------------------------------------------------
        void checkPlatformId(const std::string &id)
        {
            Platform *platform = dynamic_cast<Platform*>(
                ObjectRepository::getInstance()->getObject(
                    STRING_TOOLS::toUnsigned(id)));

            CPPUNIT_ASSERT(platform != NULL);
        }

        //--------------------------------------------------------------------
        void checkBarrierId(const std::string &id)
        {
            Barrier *barrier = dynamic_cast<Barrier*>(
                ObjectRepository::getInstance()->getObject(
                    STRING_TOOLS::toUnsigned(id)));

            CPPUNIT_ASSERT(barrier != NULL);
        }

        //--------------------------------------------------------------------
        void checkObjectIds(const std::string &ids)
        {
            StringTokenizer st(ids, ",");
            std::string token;

            while (st.hasMoreTokens())
            {
                checkObjectId(st.nextToken(token));
            }
        }

        //--------------------------------------------------------------------
        void checkCrateIds(const std::string &ids)
        {
            StringTokenizer st(ids, ",");
            std::string token;

            while (st.hasMoreTokens())
            {
                checkCrateId(st.nextToken(token));
            }
        }
        
        //--------------------------------------------------------------------
        void checkPlatformIds(const std::string &ids)
        {
            StringTokenizer st(ids, ",");
            std::string token;

            while (st.hasMoreTokens())
            {
                checkPlatformId(st.nextToken(token));
            }
        }

        //--------------------------------------------------------------------
        void checkBarrierIds(const std::string &ids)
        {
            StringTokenizer st(ids, ",");
            std::string token;

            while (st.hasMoreTokens())
            {
                checkBarrierId(st.nextToken(token));
            }
        }
    };

    //------------------------------------------------------------------------
    void checkGameControlNode()
    {
        GameControlBase *gc = GameControlBase::getInstance();

        const XMLNode *node = gc->getGameControlNode();

        CheckGameControlNodeVisitor v;
        node->acceptAllNodes(v);
    }
};

CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(LevelTest, "Level");
