/*
messagewall.c - MessageWall main definitions
Copyright (C) 2002 Ian Gulliver

This program is free software; you can redistribute it and/or modify
it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define _MESSAGEWALL_C

#include <stdlib.h>
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <openssl/rand.h>
#include <firestring.h>
#include <firedns.h>
#include "messagewall.h"
#include "security.h"
#include "smtp.h"
#include "mime.h"
#include "dbase.h"
#include "client.h"
#include "rfc822.h"
#include "dnsbl.h"
#include "dnsdcc.h"
#include "rdns.h"
#include "virus.h"
#include "rmx.h"
#include "auth.h"
#include "tls.h"

static const char tagstring[] = "$Id: messagewall.c,v 1.98.2.3 2002/08/15 01:36:47 ian Exp $";

const char *domain; /* fqdn used to say hello and in subsequent smtp conversation */
const char *path_charset; /* valid chars in MAIL FROM: and RCPT TO: */
const char *local_domains; /* sending to these domains is not considered relaying */
const char *relay_ips; /* these clients (senders) are allowed to relay */
const char *special_users; /* these recipients have special profiles that override the client's */
const char *relay_auth; /* these clients are allowed to relay if they authenticate */
const char *cert;
const char *backend_cert;
const char *pid_dir;
int max_rcpt; /* maximum number of recipients per message */
int sevenbit; /* flag (boolean): 7bit mode */
long max_message_size;
int reload;
int reload_viruses;
int dnsbls;
int dnsbls_domain;
int dnsdccs;

struct firestring_estr_t greeting; /* if defined in messagewall.conf, overrides the default smtp banner */
/* hash tables for local domains, relays, special recipients, authenticated relays: */
struct messagewall_local_domain_t **local_domain_hash;
struct messagewall_relay_ip_t **relay_ip_hash;
struct messagewall_special_user_t **special_user_hash;
struct messagewall_relay_auth_t **relay_auth_hash;

int max_clients; /* max number of simultaneous clients */
struct messagewall_client_t *clients; /* array of client control blocks */

int max_backends;
struct messagewall_backend_t *backends;

int max_parts;
int max_depth;

struct messagewall_profile_t *profile_head = NULL; /* linked list of profile control blocks */
struct messagewall_profile_t *profile_relay;
struct messagewall_profile_t *profile_default;
struct messagewall_dnsbl_global_t *dnsbl_head = NULL;
struct messagewall_dnsbl_global_t *dnsbl_domain_head = NULL;
struct messagewall_dnsdcc_global_t *dnsdcc_head = NULL;
struct messagewall_virus_database_t *virus_head = NULL;
/* settings from messagewall.conf: */
int dnsbl_timeout;
int dnsbl_resend;
int dnsdcc_timeout;
int dnsdcc_resend;
int rmx_timeout;
int rmx_resend;
int dnsbl_domain_timeout;
int dnsbl_domain_resend;
int rdns_timeout;
int rdns_resend;

int processes;
int process;
int *pidmap;
int *pipemap;
int *auth_readmap;
int *auth_writemap;
int *auth_pidmap;
int mypipe;

SSL_CTX *client_ctx;
SSL_CTX *backend_ctx;

int max_errors; /* max number of invalid smtp commands allowed before connection dropped */

struct sockaddr_in backendaddr;

int main() {
	struct firestring_conf_t *config; /* temporary storage for settings from config file. Populated by firestring_conf_parse(), read by firestring_conf_find() */
	const char *tempstr;
	const char *listen_ip;
	const char *profile_dir; /* profiles directory, from messagewall.conf */
	const char *relay_profile;
	const char *default_profile;
	struct sockaddr_in listenaddr;
	struct timeval tv;
	struct in_addr *ip;
	int i,j,k;
	int n;
	int listensock;
	int fd;
	size_t s;
	struct firestring_estr_t buffer;
	fd_set readset, writeset;
	int max_idle;
	int max_per_ip;
	time_t t;


	/*
	 * load the configuration file
	 */
	config = firestring_conf_parse(CONFDIR "/messagewall.conf");
	if (config == NULL) {
		fprintf(stderr,"STARTUP/FATAL: Unable to read " CONFDIR "/messagewall.conf\n");
		exit(100);
	}

	tempstr = firestring_conf_find(config,"max_message_size");
	if (tempstr == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'max_message_size' not set in config\n");
		exit(100);
	}

	max_message_size = atol(tempstr);
	if (max_message_size <= 512) {
		fprintf(stderr,"STARTUP/FATAL: 'max_message_size' set to invalid value: %ld\n",max_message_size);
		exit(100);
	}

	listen_ip = firestring_conf_find(config,"listen_ip");
	if (listen_ip == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'listen_ip' not set in config\n");
		exit(100);
	}

	memset(&listenaddr,0,sizeof(listenaddr));

	tempstr = firestring_conf_find(config,"listen_port");
	if (tempstr == NULL)
		listenaddr.sin_port = htons(25);
	else
		listenaddr.sin_port = htons(atoi(tempstr));

	listenaddr.sin_family = AF_INET;

	domain = firestring_conf_find(config,"domain");
	if (domain == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'domain' not set in config\n");
		exit(100);
	}

	if (strlen(domain) > SMTP_DOMAIN_MAXLEN) {
		fprintf(stderr,"STARTUP/FATAL: invalid 'domain': too long\n");
		exit(100);
	}
	domain = firestring_strdup(domain);

	path_charset = firestring_conf_find(config,"path_charset");
	if (path_charset == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'path_charset' not set in config\n");
		exit(100);
	}
	path_charset = firestring_strdup(path_charset);

	tempstr = firestring_conf_find(config,"backend_ip");
	if (tempstr == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'backend_ip' not set in config\n");
		exit(100);
	}

	memset(&backendaddr,0,sizeof(backendaddr));

	ip = firedns_aton4_s(tempstr,&backendaddr.sin_addr);
	if (ip == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'backend_ip' is set to an invalid value\n");
		exit(100);
	}

	tempstr = firestring_conf_find(config,"backend_port");
	if (tempstr == NULL)
		backendaddr.sin_port = htons(25);
	else
		backendaddr.sin_port = htons(atoi(tempstr));

	backendaddr.sin_family = AF_INET;

	profile_dir = firestring_conf_find(config,"profile_dir");
	if (profile_dir == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'profile_dir' not set in config\n");
		exit(100);
	}

	pid_dir = firestring_conf_find(config,"pid_dir");
	if (pid_dir == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'pid_dir' not set in config\n");
		exit(100);
	}
	pid_dir = firestring_strdup(pid_dir);

	relay_profile = firestring_conf_find(config,"relay_profile");
	if (relay_profile == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'relay_profile' not set in config\n");
		exit(100);
	}

	default_profile = firestring_conf_find(config,"default_profile");
	if (default_profile == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'default_profile' not set in config\n");
		exit(100);
	}

	greeting.l = 0;
	tempstr = firestring_conf_find(config,"greeting");
	if (tempstr != NULL) {
		firestring_estr_alloc(&greeting,strlen(tempstr));
		firestring_estr_strcpy(&greeting,tempstr);
	}

	tempstr = firestring_conf_find(config,"7bit");
	sevenbit = 0;
	if (tempstr != NULL && atoi(tempstr) == 1)
		sevenbit = 1;

	load_int(config,"max_clients",&max_clients);
	load_int(config,"max_backends",&max_backends);
	load_int(config,"max_rcpt",&max_rcpt);
	load_int(config,"max_idle",&max_idle);
	load_int(config,"max_per_ip",&max_per_ip);
	load_int(config,"max_parts",&max_parts);
	load_int(config,"max_depth",&max_depth);
	load_int(config,"max_errors",&max_errors);
	load_int(config,"processes",&processes);

	load_timeout(config,"dnsbl_timeout",&dnsbl_timeout,&dnsbl_resend);
	load_timeout(config,"dnsdcc_timeout",&dnsdcc_timeout,&dnsdcc_resend);
	load_timeout(config,"rmx_timeout",&rmx_timeout,&rmx_resend);
	load_timeout(config,"dnsbl_domain_timeout",&dnsbl_domain_timeout,&dnsbl_domain_resend);
	load_timeout(config,"rdns_timeout",&rdns_timeout,&rdns_resend);

	local_domains = firestring_conf_find(config,"local_domains");
	if (local_domains != NULL)
		local_domains = firestring_strdup(local_domains);
	relay_ips = firestring_conf_find(config,"relay_ips");
	if (relay_ips != NULL)
		relay_ips = firestring_strdup(relay_ips);
	special_users = firestring_conf_find(config,"special_users");
	if (special_users != NULL)
		special_users = firestring_strdup(special_users);
	relay_auth = firestring_conf_find(config,"relay_auth");
	if (relay_auth != NULL)
		relay_auth = firestring_strdup(relay_auth);
	cert = firestring_conf_find(config,"certificate");
	if (cert != NULL)
		cert = firestring_strdup(cert);
	backend_cert = firestring_conf_find(config,"backend_certificate");
	if (backend_cert != NULL)
		backend_cert = firestring_strdup(backend_cert);


	/*
	 * memory for maps
	 */
	pidmap = (int *) firestring_malloc(sizeof(int) * (processes));
	pipemap = (int *) firestring_malloc(sizeof(int) * (processes));
	auth_readmap = (int *) firestring_malloc(sizeof(int) * (processes));
	auth_writemap = (int *) firestring_malloc(sizeof(int) * (processes));
	auth_pidmap = (int *) firestring_malloc(sizeof(int) * (processes));

	/*
	 * split off auth processes
	 */
	if (relay_auth != NULL) {
		for (i = 0; i < processes; i++) {
			int readpipe[2];
			int writepipe[2];
	
			if (pipe(readpipe) != 0) {
				perror("pipe");
				exit(100);
			}
	
			if (pipe(writepipe) != 0) {
				perror("pipe");
				exit(100);
			}
	
			auth_pidmap[i] = fork();
			if (auth_pidmap[i] == -1) {
				perror("fork");
				exit(100);
			} else if (auth_pidmap[i] == 0) {
				/*
			 	* auth process
			 	*/
				for (j = 0; j < i; j++) {
					close(auth_writemap[j]);
					close(auth_readmap[j]);
				}
				close(readpipe[0]);
				close(writepipe[1]);
				relay_auth_hash = (struct messagewall_relay_auth_t **) firestring_malloc(HASH_SIZE * sizeof(struct messagewall_relay_auth_t *));
				auth_main(config,writepipe[0],readpipe[1],i);
			} else {
				/*
			 	* parent
			 	*/
				close(readpipe[1]);
				close(writepipe[0]);
				auth_writemap[i] = writepipe[1];
				auth_readmap[i] = readpipe[0];
			}
		}
	}

	/*
	 * hash memory
	 */
	local_domain_hash = (struct messagewall_local_domain_t **) firestring_malloc(HASH_SIZE * sizeof(struct messagewall_local_domain_t *));
	relay_ip_hash = (struct messagewall_relay_ip_t **) firestring_malloc(HASH_SIZE * sizeof(struct messagewall_relay_ip_t *));
	special_user_hash = (struct messagewall_special_user_t **) firestring_malloc(HASH_SIZE * sizeof(struct messagewall_special_user_t *));

	/*
	 * allocate required memory for all clients & their member variables
	 */
	clients = (struct messagewall_client_t *) firestring_malloc(max_clients * sizeof(struct messagewall_client_t));
	for (i = 0; i < max_clients; i++) {
		clients[i].fd = -1;
		firestring_estr_alloc(&clients[i].username,SMTP_LINE_MAXLEN);
		firestring_estr_alloc(&clients[i].password,SMTP_LINE_MAXLEN);
		firestring_estr_alloc(&clients[i].message,max_message_size);
		firestring_estr_alloc(&clients[i].buffer,SMTP_LINE_MAXLEN);
		firestring_estr_alloc(&clients[i].from,SMTP_PATH_MAXLEN);
		firestring_estr_alloc(&clients[i].fromdomain,SMTP_PATH_MAXLEN);
		clients[i].to = (struct firestring_estr_t *) firestring_malloc(max_rcpt * sizeof(struct firestring_estr_t));
		for (n = 0; n < max_rcpt; n++)
			firestring_estr_alloc(&clients[i].to[n],SMTP_PATH_MAXLEN);
		clients[i].points = (int *) firestring_malloc(SMTP_MAX_WARNINGS * sizeof(int));
		clients[i].warnings = (struct firestring_estr_t *) firestring_malloc(SMTP_MAX_WARNINGS * sizeof(struct firestring_estr_t));
		for (n = 0; n < SMTP_MAX_WARNINGS; n++)
			firestring_estr_alloc(&clients[i].warnings[n],SMTP_WARNING_MAXLEN);
		clients[i].have_from = 0;
		clients[i].num_to = 0;
		clients[i].ssl = NULL;
	}

	/*
	 * allocate required memory for all backends & their buffers
	 */
	backends = (struct messagewall_backend_t *) firestring_malloc(max_clients * sizeof(struct messagewall_backend_t));
	for (i = 0; i < max_backends; i++) {
		firestring_estr_alloc(&backends[i].buffer,SMTP_LINE_MAXLEN);
		firestring_estr_alloc(&backends[i].message,max_message_size + (SMTP_WARNING_MAXLEN * SMTP_MAX_WARNINGS) + SMTP_LINE_MAXLEN);
		backends[i].ssl = NULL;
	}

	/*
	 * allocate mime parsing memory
	 */
	for (j = 0; j < max_clients; j++) {
		clients[j].parts = (struct mime_part_t *) firestring_malloc(max_parts * sizeof(struct mime_part_t));
		for (i = 0; i < max_parts; i++) {
			firestring_estr_alloc(&clients[j].parts[i].content_type,RFC822_VALUE_MAXLEN);
			firestring_estr_alloc(&clients[j].parts[i].content_transfer_encoding,RFC822_VALUE_MAXLEN);
			firestring_estr_alloc(&clients[j].parts[i].boundary,RFC822_VALUE_MAXLEN);
		}
	}

	/*
	 * allocate memory for outgoing buffer
	 */
	firestring_estr_alloc(&buffer,SMTP_LINE_MAXLEN);

	/*
	 * let MIME do its startup
	 */
	mime_startup();

	/*
	 * load the profiles
	 */
	dbase_load_profiles(profile_dir);
	profile_relay = dbase_get_profile(relay_profile);
	profile_default = dbase_get_profile(default_profile);

	/*
	 * create client space for dnsbl fds
	 * (now that we know how many there are)
	 */
	if (dnsbl_head == NULL)
		dnsbls = 0;
	else
		dnsbls = dnsbl_head->i + 1;
	for (i = 0; i < max_clients; i++) {
		clients[i].dnsbl_fd = (int *) firestring_malloc(sizeof(int) * dnsbls);
		clients[i].dnsbl_ip = (struct in_addr *) firestring_malloc(sizeof(struct in_addr) * dnsbls);
		for (j = 0; j < dnsbls; j++)
			clients[i].dnsbl_fd[j] = -1;
	}

	if (dnsbl_domain_head == NULL)
		dnsbls_domain = 0;
	else
		dnsbls_domain = dnsbl_domain_head->i + 1;
	for (i = 0; i < max_clients; i++) {
		clients[i].dnsbl_domain_fd = (int *) firestring_malloc(sizeof(int) * dnsbls_domain);
		clients[i].dnsbl_domain_ip = (struct in_addr *) firestring_malloc(sizeof(struct in_addr) * dnsbls_domain);
		for (j = 0; j < dnsbls_domain; j++)
			clients[i].dnsbl_domain_fd[j] = -1;
	}

	if (dnsdcc_head == NULL)
		dnsdccs = 0;
	else
		dnsdccs = dnsdcc_head->i + 1;
	for (i = 0; i < max_clients; i++) {
		clients[i].dnsdcc_fd = (int **) firestring_malloc(sizeof(int *) * max_parts);
		for (j = 0; j < max_parts; j++) {
			clients[i].dnsdcc_fd[j] = (int *) firestring_malloc(sizeof(int) * dnsdccs);
			for (k = 0; k < dnsdccs; k++)
				clients[i].dnsdcc_fd[j][k] = -1;
		}
	}

	/*
	 * load the dns configuration so we have it inside the chroot
	 */
	firedns_init();

	/*
	 * SSL library startup
	 * SECURITY: if you don't have /dev/urandom, you're SOL
	 */
	SSL_load_error_strings();
	SSL_library_init();       
	RAND_load_file("/dev/urandom",4096);
	
	if (cert != NULL) {
		client_ctx = SSL_CTX_new(SSLv23_server_method());
		if (client_ctx == NULL) {
			fprintf(stderr,"STARTUP/FATAL: failed to allocate SSL certificate memory\n");
			exit(100);
		}

		if (SSL_CTX_use_RSAPrivateKey_file(client_ctx,cert,SSL_FILETYPE_PEM) != 1) {
			fprintf(stderr,"STARTUP/FATAL: unable to load key\n");
			exit(100);
		}

		if (SSL_CTX_use_certificate_chain_file(client_ctx,cert) != 1) {
			fprintf(stderr,"STARTUP/FATAL: unable to load certificate\n");
			exit(100);
		}

		SSL_CTX_set_cipher_list(client_ctx,"DEFAULT");
	}

	if (backend_cert != NULL) {
		backend_ctx = SSL_CTX_new(SSLv23_client_method());
		if (backend_ctx == NULL) {
			fprintf(stderr,"STARTUP/FATAL: failed to allocate SSL backend certificate memory\n");
			exit(100);
		}

		if (SSL_CTX_use_RSAPrivateKey_file(backend_ctx,backend_cert,SSL_FILETYPE_PEM) != 1) {
			fprintf(stderr,"STARTUP/FATAL: unable to load backend key\n");
			exit(100);
		}

		if (SSL_CTX_use_certificate_chain_file(backend_ctx,backend_cert) != 1) {
			fprintf(stderr,"STARTUP/FATAL: unable to load backend certificate\n");
			exit(100);
		}

		SSL_CTX_set_cipher_list(backend_ctx,"DEFAULT");
	}
	/*
	 * set up the listening socket
	 */
	/*
	 * this socket must be created as root so we can bind low
	 * messagewall does not execute outside programs, so we don't have to worry about ioctl DoS
         * SECURITY: anyone who can compromise the server can execute ioctls on listensock (possible DoS attack after compromise)
	 * SECURITY: do not alter messagewall to execute external programs without closing listensock first
         */
	listensock = socket(PF_INET, SOCK_STREAM, 0); /* ITS4: ignore socket */

	i = 1;
	setsockopt(listensock, SOL_SOCKET, SO_REUSEADDR, (char *)&i, sizeof(i));

	ip = firedns_aton4_s(listen_ip,&listenaddr.sin_addr);
	if (ip == NULL) {
		fprintf(stderr,"STARTUP/FATAL: listen_ip is not a valid IP address\n");
		exit(100);
	}

	/*
	 * very old linux kernel bug for packet stealing
	 * this is on a privileged port, so danger is minimized
	 * SECURITY: do not use this with linux kernels < 1.3.60 or 1.2.13
	 */
	if (bind(listensock,(struct sockaddr *) &listenaddr,sizeof(listenaddr)) != 0) { /* ITS4: ignore bind */
		perror("bind");
		exit(100);
	}

	socket_nonblock(listensock);

	/*
	 * security context
	 */
	security_context(config,SECURITY_MWALL);

	firestring_conf_free(config);

	/*
	 * clear database storage
	 */
	for (i = 0; i < HASH_SIZE; i++) {
		local_domain_hash[i] = NULL;
		relay_ip_hash[i] = NULL;
		special_user_hash[i] = NULL;
	}

	/*
	 * simulate a hup to register the handler and schedule database load
	 */
	reload = 1;
	reload_viruses = 1;
	signal(SIGPIPE,SIG_IGN);
	signal(SIGHUP,handle_hup);
	signal(SIGUSR1,handle_usr1);
	signal(SIGTERM,handle_term);

	/* 
	 * actually listen for connections
	 */
	if (listen(listensock,max_clients) != 0) {
		perror("listen");
		exit(100);
	}

	/*
	 * fork off children
	 */
	for (process = 1; process < processes; process++) {
		int pipes[2];
		if (pipe(pipes) != 0) {
			perror("pipe");
			exit(100);
		}
		pipemap[process] = pipes[0];
		mypipe = pipes[1];
		i = fork();
		if (i == -1) {
			perror("fork");
			exit(100);
		} else if (i == 0) {
			/*
			 * child
			 */
			fprintf(stderr,"{%d} PROCESS/STATUS: start\n",process);
			close(pipemap[process]);
			break;
		} else {
			/*
			 * parent
			 */
			close(mypipe);
			pidmap[process] = i;
		}
	}
	if (i != 0) {
		process = 0;
		fprintf(stderr,"{%d} PROCESS/STATUS: start\n",process);
	}

	/*
	 * close any auth fd's that aren't mine
	 */
	for (i = 0; i < processes; i++) {
		if (i == process)
			continue;
		close(auth_writemap[i]);
		close(auth_readmap[i]);
	}

	/*
	 * write out my pid file
	 */
	{
		FILE *f;
		char filename[512];

		firestring_snprintf(filename,512,"%s/mwall.%d.pid",pid_dir,process);
		f = fopen(filename,"w");
		if (f == NULL) {
			perror("fopen(pid_dir/mwall.pid)");
			exit(100);
		}
		fprintf(f,"%d\n",getpid());
		fclose(f);
	}



	/*
	 * start up the backend connections
	 */
	for (i = 0; i < max_backends; i++) {
		if (client_connect(i) != 0) {
			/* 
			 * if we can't get through this, we've got no chance of succeeding later
			 */
			perror("client_connect");
			exit(100);
		}
	}

	/*
	 * main loop
	 */
	while (1) {
		if (reload == 1) {
			reload = 0;
			dbase_load();
		}
		if (reload_viruses == 1) {
			reload_viruses = 0;
			virus_reload();
		}
		FD_ZERO(&readset);
		FD_ZERO(&writeset);
		FD_SET(listensock,&readset);
		n = listensock;
		tv.tv_usec = 0;
		tv.tv_sec = 1;
		
		if (process == 0)
			for (i = 1; i < processes; i++) {
				FD_SET(pipemap[i],&readset);
				n = max(n,pipemap[i]);
			}
		else {
			FD_SET(mypipe,&readset);
			n = max(n,mypipe);
		}

		if (relay_auth != NULL) {
			FD_SET(auth_readmap[process],&readset);
			n = max(n,auth_readmap[process]);
		}

		for (i = 0; i < max_clients; i++) {
			if (clients[i].fd == -1)
				continue;
			if (clients[i].have_from == 1) {
				if (clients[i].rmx_fd >= 0) {
					FD_SET(clients[i].rmx_fd,&readset);
					n = max(n,clients[i].rmx_fd);
				}
				if (clients[i].rmx_a_fd >= 0) {
					FD_SET(clients[i].rmx_a_fd,&readset);
					n = max(n,clients[i].rmx_a_fd);
				}
				for (j = 0; j < dnsbls_domain; j++)
					if (clients[i].dnsbl_domain_fd[j] >= 0) {
						FD_SET(clients[i].dnsbl_domain_fd[j],&readset);
						n = max(n,clients[i].dnsbl_domain_fd[j]);
					}
			}
			if (clients[i].rdns_fd >= 0) {
				FD_SET(clients[i].rdns_fd,&readset);
				n = max(n,clients[i].rdns_fd);
			}
			for (j = 0; j < dnsbls; j++)
				if (clients[i].dnsbl_fd[j] >= 0) {
					FD_SET(clients[i].dnsbl_fd[j],&readset);
					n = max(n,clients[i].dnsbl_fd[j]);
				}
			if (clients[i].send_progress == 0) {
				FD_SET(clients[i].fd,&readset);
				n = max(n,clients[i].fd);
			} else {
				/*
				 * we've received the message and are waiting for something
				 * this means that we've sent dnsdcc queries
				 */
				for (j = 0; j <= clients[i].num_parts; j++)
					for (k = 0; k < dnsdccs; k++)
						if (clients[i].dnsdcc_fd[j][k] >= 0) {
							FD_SET(clients[i].dnsdcc_fd[j][k],&readset);
							n = max(n,clients[i].dnsdcc_fd[j][k]);
						}
			}
		}

		for (i = 0; i < max_backends; i++) {
			switch (backends[i].state) {
				case BACKEND_STATE_CONNECTING:
				case BACKEND_STATE_SENDING:
					FD_SET(backends[i].fd,&writeset);
					n = max(n,backends[i].fd);
					break;
				case BACKEND_STATE_GREETING:
				case BACKEND_STATE_EHLO_RESPONSE:
				case BACKEND_STATE_IDLE:
				case BACKEND_STATE_CLEARPIPE:
				case BACKEND_STATE_WAITDELIVER:
				case BACKEND_STATE_HELO_RESPONSE:
				case BACKEND_STATE_STARTTLS_RESPONSE:
				case BACKEND_STATE_SSL_HANDSHAKE:
				case BACKEND_STATE_RSET:
					FD_SET(backends[i].fd,&readset);
					n = max(n,backends[i].fd);
					break;
			}
		}

		/*
		 * we do not use MAX_FDS
		 */
		n = select(n + 1, &readset, &writeset, NULL, &tv); /* ITS4: ignore select */

		if (n == -1)
			continue;

		if (process == 0)
			for (i = 1; i < processes; i ++) {
				if (FD_ISSET(pipemap[i],&readset)) {
					fprintf(stderr,"{%d} PROCESS/FATAL: lost connection to child process %d, aborting\n",process,i);
					exit(1);
				}
			}
		else if (FD_ISSET(mypipe,&readset)) {
			fprintf(stderr,"{%d} PROCESS/FATAL: lost connection to parent process, aborting\n",process);
			exit(1);
		}

		if (relay_auth != NULL && FD_ISSET(auth_readmap[process],&readset)) {
			fprintf(stderr,"{%d} PROCESS/FATAL: lost connection to auth process, aborting\n",process);
			exit(1);
		}

		if (FD_ISSET(listensock,&readset)) {
			/*
			 * new connection attempt
			 * accept it first
			 */
			s = sizeof(listenaddr);
			fd = accept(listensock,(struct sockaddr *) &listenaddr,&s);
			if (fd == -1)
				/*
				 * client disappeared
				 * we'll spin around again for simplicity sake
				 */
				continue;

			/*
			 * non blocking
			 */
			if (socket_nonblock(fd) != 0) {
				close(fd);
				continue;
			}

			/* 
			 * check that we don't have too many connections from this IP already
			 */
			for (i = 0, n = 0; i < max_clients; i++)
				if (clients[i].fd != -1 && memcmp(&clients[i].ip,&listenaddr.sin_addr,sizeof(listenaddr.sin_addr)) == 0)
					n++;
			if (n >= max_per_ip) {
				/* 
				 * too many from this IP
				 */
				fprintf(stderr,"{%d} SERVER/TEMPORARY: per-ip overflow from %s\n",process,firedns_ntoa4(&listenaddr.sin_addr));
				firestring_estr_sprintf(&buffer,"421 %s MessageWall: Too many connections from your IP\r\n",domain);
				write(fd,buffer.s,buffer.l);
				close(fd);
				/*
				 * spin back around for simplicity
				 */
				continue;
			}

			/*
			 * see if we have space to handle it
			 */
			for (i = 0; i < max_clients; i++) {
				if (clients[i].fd == -1)
					break;
			}
			if (i == max_clients) {
				/*
				 * no free client spaces
				 * inform the client that we can't talk to them now
				 * NOTE: temporary error on greeting is a violation of RFC 2821
				 */
				fprintf(stderr,"{%d} SERVER/TEMPORARY: client overflow from %s\n",process,firedns_ntoa4(&listenaddr.sin_addr));
				firestring_estr_sprintf(&buffer,"421 %s MessageWall: Too busy, please try again later\r\n",domain);
				write(fd,buffer.s,buffer.l);
				close(fd);
				/*
				 * spin back around for simplicity
				 */
				continue;
			}

			/*
			 * register and welcome the new client
			 */
			fprintf(stderr,"{%d} (%d) SERVER/STATUS: connection from %s:%d\n",process,i,firedns_ntoa4(&listenaddr.sin_addr),ntohs(listenaddr.sin_port));
			smtp_clear(i);
			clients[i].fd = fd;
			clients[i].buffer.l = 0;
			clients[i].have_from = 0;
			clients[i].num_to = 0;
			clients[i].in_data = 0;
			clients[i].send_progress = 0;
			clients[i].errors = 0;
			clients[i].rmx_fd = -1;
			clients[i].rmx_a_fd = -1;
			clients[i].rdns_fd = -1;
			clients[i].auth_status = AUTH_STATUS_NONE;
			clients[i].username.l = 0;
			clients[i].ssl = NULL;
			clients[i].ssl_handshake = 0;
			time(&clients[i].lasttalk);
			time(&clients[i].signon);
			/*
			 * memcpy uses sizeof destination, this is safe
			 */
			memcpy(&clients[i].ip,&listenaddr.sin_addr,sizeof(clients[i].ip)); /* ITS4: ignore memcpy */
			if (dbase_ip_can_relay((unsigned char *) &clients[i].ip) == 0)
				clients[i].can_relay = 1;
			else
				clients[i].can_relay = 0;
			if (greeting.l == 0)
				firestring_estr_sprintf(&buffer,"220 %s MessageWall " MESSAGEWALL_VERSION " (%s)\r\n",domain,clients[i].can_relay == 1 ? "You may relay" : "You may not relay");
			else
				firestring_estr_sprintf(&buffer,"220 %s %e\r\n",domain,&greeting);
			tls_client_write(i,buffer.s,buffer.l);
			dnsbl_send_queries(i);
			rdns_send_query(i);
		}
		time(&t);
		for (i = 0; i < max_clients; i++) {
			if (clients[i].fd == -1)
				continue;
			if (clients[i].have_from != 0) {
				/*
				 * completion checks
				 */

				/*
				 * we've seen the from line and sent these queries
				 */
				if (clients[i].rmx_fd >= 0 && FD_ISSET(clients[i].rmx_fd,&readset)) {
					rmx_response(i,clients[i].rmx_fd);
					if (clients[i].send_progress == PROGRESS_WAITRMX && rmx_check_profile(i) != -1)
						smtp_checks_gotRMX(i,0);
				}
				if (clients[i].rmx_a_fd >= 0 && FD_ISSET(clients[i].rmx_a_fd,&readset)) {
					rmx_response(i,clients[i].rmx_a_fd);
					if (clients[i].send_progress == PROGRESS_WAITRMX && rmx_check_profile(i) != -1)
						smtp_checks_gotRMX(i,0);
				}
				for (j = 0; j < dnsbls_domain; j++)
					if (clients[i].dnsbl_domain_fd[j] >= 0 && FD_ISSET(clients[i].dnsbl_domain_fd[j],&readset)) {
						dnsbl_domain_response(i,j);
						if (clients[i].send_progress == PROGRESS_WAITDNSBL_DOMAIN && dnsbl_domain_check_profile(i,0) != -1)
							smtp_checks_gotDNSBL_DOMAIN(i);
					}
			}
			if (clients[i].rdns_fd >= 0 && FD_ISSET(clients[i].rdns_fd,&readset)) {
				rdns_response(i);
				if (clients[i].send_progress == PROGRESS_WAITRDNS && rdns_check_profile(i) != -1)
					smtp_checks_gotRDNS(i,0);
			}
			for (j = 0; j < dnsbls; j++)
				if (clients[i].dnsbl_fd[j] >= 0 && FD_ISSET(clients[i].dnsbl_fd[j],&readset)) {
					dnsbl_response(i,j);
					if (clients[i].send_progress == PROGRESS_WAITDNSBL && dnsbl_check_profile(i,0) != -1)
						smtp_checks_gotDNSBL(i);
				}
			if (clients[i].send_progress != 0) {
				/*
				 * have received message
				 */
				for (j = 0; j <= clients[i].num_parts; j++)
					for (k = 0; k < dnsdccs; k++)
						if (clients[i].dnsdcc_fd[j][k] >= 0 && FD_ISSET(clients[i].dnsdcc_fd[j][k],&readset)) {
							dnsdcc_response(i,j,k);
							if (clients[i].send_progress == PROGRESS_WAITDNSDCC && dnsdcc_check_profile(i,0) != -1)
								smtp_checks_gotDNSDCC(i);
						}
			}


			/*
			 * resend checks
			 */
			if (clients[i].have_from != 0) {
				/*
				 * have received from line 
				 */
				if (clients[i].resent_rmx == 0 && t - clients[i].rmx_request >= rmx_resend) {
					clients[i].resent_rmx = 1;
					rmx_resend_queries(i);
				}
				if (clients[i].resent_dnsbl_domain == 0 && t - clients[i].rmx_request >= dnsbl_domain_resend) {
					clients[i].resent_dnsbl_domain = 1;
					dnsbl_domain_resend_queries(i);;
				}
			}
			if (clients[i].resent_rdns == 0 && t - clients[i].signon >= rdns_resend) {
				clients[i].resent_rdns = 1;
				rdns_resend_query(i);
			}
			if (clients[i].resent_dnsbl == 0 && t - clients[i].signon >= dnsbl_resend) {
				clients[i].resent_dnsbl = 1;
				dnsbl_resend_queries(i);
			}
			if (clients[i].send_progress != 0) {
				/*
				 * have received message
				 */
				if (clients[i].resent_dnsdcc == 0 && t - clients[i].dnsdcc_send >= dnsdcc_resend) {
					clients[i].resent_dnsdcc = 1;
					dnsdcc_resend_queries(i);
				}
			}


			/*
			 * timeout checks
			 */
			if (clients[i].send_progress == PROGRESS_WAITRMX && t - clients[i].rmx_request >= rmx_timeout) {
				fprintf(stderr,"{%d} (%d) RMX/STATUS: wait timed out, going on in fail-closed\n",process,i);
				smtp_checks_gotRMX(i,1);
			}
			if (clients[i].send_progress == PROGRESS_WAITDNSBL_DOMAIN && t - clients[i].rmx_request >= dnsbl_domain_timeout) {
				fprintf(stderr,"{%d} (%d) DNSBL-DOMAIN/STATUS: wait timed out, going on in fail-open\n",process,i);
				smtp_checks_gotDNSBL_DOMAIN(i);
			}
			if (clients[i].send_progress == PROGRESS_WAITRDNS && t - clients[i].signon >= rdns_timeout) {
				fprintf(stderr,"{%d} (%d) RDNS/STATUS: wait timed out, going on in fail-closed\n",process,i);
				smtp_checks_gotRDNS(i,1);
			}
			if (clients[i].send_progress == PROGRESS_WAITDNSBL && t - clients[i].signon >= dnsbl_timeout) {
				fprintf(stderr,"{%d} (%d) DNSBL/STATUS: wait timed out, going on in fail-open\n",process,i);
				smtp_checks_gotDNSBL(i);
			}
			if (clients[i].send_progress == PROGRESS_WAITDNSDCC && t - clients[i].dnsdcc_send >= dnsdcc_timeout) {
				fprintf(stderr,"{%d} (%d) DNSDCC/STATUS: wait timed out, going on in fail-open\n",process,i);
				smtp_checks_gotDNSDCC(i);
			}

			if (FD_ISSET(clients[i].fd,&readset)) {
				/*
				 * client is speaking
				 */

				/*
				 * check if we're in TLS handshake
				 */
				if (clients[i].ssl_handshake == 1) {
					if (tls_handshake(i) == 1) {
						fprintf(stderr,"{%d} (%d) TLS/FATAL: handshake failed\n",process,i);
						smtp_clear(i);
					}
				}

				/*
				 * check if we're in a DATA block
				 */
				else if (clients[i].in_data == 1) {
					/*
					 * do fast reading
					 */
					j = tls_client_eread(i,&clients[i].message);
					if (j == 0) {
						/*
						 * quick parse
						 */
						smtp_parsedata(i);
					} else if (j == 2) {
						/*
						 * ran out of buffer space
						 */
						fprintf(stderr,"{%d} (%d) SERVER/FATAL: message too long\n",process,i);
						tls_client_write(i,SMTP_MESSAGE_TOOLONG,sizeof(SMTP_MESSAGE_TOOLONG) + 1);
						smtp_clear(i);
					} else {
						/*
					 	* client error, probably disconnected on us
					 	*/
						fprintf(stderr,"{%d} (%d) SERVER/FATAL: disconnect inside DATA\n",process,i);
						smtp_clear(i);
					}
				} else {
					/*
					 * do line-by-line parsing
					 */
					j = tls_client_eread(i,&clients[i].buffer);
					if (j == 0) {
						/*
					 	* call SMTP handlers to take care of the line
					 	*/
						while (smtp_parseline(i) == 0);
					} else if (j == 2) {
						/*
						 * ran out of buffer space
						 */
						fprintf(stderr,"{%d} (%d) SERVER/FATAL: line too long\n",process,i);
						tls_client_write(i,SMTP_TOOLONG,sizeof(SMTP_TOOLONG) - 1);
						smtp_clear(i);
					} else {
						/*
					 	* client error, probably disconnected on us
					 	*/
						fprintf(stderr,"{%d} (%d) SERVER/FATAL: disconnect\n",process,i);
						smtp_clear(i);
					}
				}
			} else {
				/*
				 * not active this time 'round
				 */
				if (t - clients[i].lasttalk > max_idle) {
					fprintf(stderr,"{%d} (%d) SERVER/FATAL: idle timeout\n",process,i);
					tls_client_write(i,SMTP_TIMEOUT,sizeof(SMTP_TIMEOUT) - 1);
					if (clients[i].send_progress == PROGRESS_WORKING) {
						/*
						 * backend timed out, kill it
						 */
						fprintf(stderr,"{%d} [%d] BACKEND/FATAL: idle timeout, restarting connection\n",process,clients[i].backend);
						if (client_connect(clients[i].backend) != 0) {
							fprintf(stderr,"{%d} [%d] BACKEND/FATAL: failed to start backend connection\n",process,clients[i].backend);
							exit(1);
						}
					}
					smtp_clear(i);
				}
			}
		}
		for (i = 0; i < max_backends; i++) {
			if (backends[i].fd != -1 && (FD_ISSET(backends[i].fd,&readset) || FD_ISSET(backends[i].fd,&writeset))) {
				if (backends[i].state == BACKEND_STATE_SENDING)
					client_send(i);
				else if (client_read_event(i) != 0) {
					fprintf(stderr,"{%d} [%d] BACKEND/FATAL: disconnected\n",process,i);
					/*
					 * error while processing
					 */
					while (client_connect(i) != 0);
				}
			}
		}
	}

	return 0;
}

void handle_hup(int s) {
	reload = 1;

	fprintf(stderr,"{%d} PROCESS/STATUS: received HUP, reloading databases\n",process);

	signal(s,handle_hup);
}

void handle_usr1(int s) {
	reload_viruses = 1;

	fprintf(stderr,"{%d} PROCESS/STATUS: received USR1, reloading virus definitions\n",process);

	signal(s,handle_usr1);
}

void handle_term(int s) {
	fprintf(stderr,"{%d} PROCESS/STATUS: received TERM, shutting down\n",process);

	exit(0);
}

int socket_nonblock(int s) {
	if (fcntl(s, F_SETFL, O_NONBLOCK) != 0) {
		perror("fcntl");
		return 1;
	}

	return 0;
}

void load_timeout(struct firestring_conf_t *config, const char *name, int *timeout, int *resend) {
	const char *tempstr;

	tempstr = firestring_conf_find(config,name);
	if (tempstr == NULL) {
		fprintf(stderr,"STARTUP/FATAL: '%s' not set in config\n",name);
		exit(100);
	}

	*timeout = atoi(tempstr);
	if (*timeout <= 0) {
		fprintf(stderr,"STARTUP/FATAL: '%s' is set to an invalid value: %d\n",name,*timeout);
		exit(100);
	}
	*resend = *timeout / 2;
}

void load_int(struct firestring_conf_t *config, const char *name, int *value) {
	const char *tempstr;

	tempstr = firestring_conf_find(config,name);
	if (tempstr == NULL) {
		fprintf(stderr,"STARTUP/FATAL: '%s' is not set in config\n",name);
		exit(100);
	}

	*value = atoi(tempstr);
	if (*value <= 0) {
		fprintf(stderr,"STARTUP/FATAL: '%s' is set to an invalid value: %d\n",name,*value);
		exit(100);
	}
}
