/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: showspace.c,v 1.9 2003/02/05 04:40:14 kos Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "keyspace.h"
#include "generator.h"
#include "llog.h"

#define DEF_LINELEN 1024
#define DEF_VERBOSE 1
#define DEF_MIN_PASSLEN 1
#define DEF_MAX_PASSLEN 6
#define DEF_KEYSPACE "eairtonslcumdphgbyfwvkzxqSMCBAjZYXWVUTRQPONLKJIHGFED9876543210"

void
usage(char *argv0, int code) {
  fprintf(stderr, "Usage: %s [options]\n\
\t-V: Print version string\n\
\t-v: Verbosity (default: %d)\n\
\t-s: Starting index (default: %s)\n\
\t-g: Generator type (default: %s)\n\
\t-o: Generator options (default: %s)\n\
\t-l: Available generators\n\
\t-t: Target key (default: %s)\n",
	  argv0, DEF_VERBOSE, "0", "bruteforce", "", "test01");
  exit(code);
}

int
show_space(char *generator, char *opts, char *what, char *start) {
  generator_t *g;
  char buffer[DEF_LINELEN];
  char current[DEF_LINELEN];
  char min[DEF_LINELEN];
  char max[DEF_LINELEN];
  key_index_t st;
  int keylen;
    
  if(!(g = generator_init(generator, opts))) {
    llog(1, "generator_init(%s, %s) failed\n", generator, opts);
    return 1;
  }

  if(start) {
    keyspace_init(st);
    keyspace_fromstr(st, start);
    generator_set(g, st);
    keyspace_destroy(st);
  }

  while(!generator_done(g)) {
    memset(buffer, '\0', DEF_LINELEN);
    generator_fetch(g, buffer, DEF_LINELEN, &keylen); 
    keyspace_tostr(current, DEF_LINELEN, *generator_curindex(g));
    keyspace_tostr(min, DEF_LINELEN, *generator_minindex(g));
    keyspace_tostr(max, DEF_LINELEN, *generator_maxindex(g));

    if(!strcmp(buffer, what)) {
      llog(1, "fetch: success at %s %s %s\n", current, min, max, buffer);
      llog_hexdump(1, buffer, keylen);
      return 0;
    }
    llog(2, "fetch: %s %s %s\n", current, min, max);
    llog_hexdump(2, buffer, keylen);
  }
  generator_destroy(g);
  return 0;
}

int
main(int argc, char **argv) {
  char c;
  int verbose;
  char what[DEF_LINELEN];
  char *start;
  char generator[DEF_LINELEN];
  char *opts;
  int i;
  
  start = NULL;
  verbose = DEF_VERBOSE;
  strcpy(what, "test01");
  strcpy(generator, "bruteforce");
  opts = NULL;
  
  while((c = getopt(argc, argv, "v:o:t:g:s:lV")) != EOF) {
    switch(c) {
    case 'v':
      verbose = atoi(optarg);
      break;
    case 'o':
      opts = optarg;
      break;
    case 't':
      strncpy(what, optarg, DEF_LINELEN);
      break;
    case 'g':
      strncpy(generator, optarg, DEF_LINELEN);
      break;
    case 's':
      start = strdup(optarg);
      break;
    case 'l':
      printf("Available generators: ");
      for(i=0; !generator_query(i, what, DEF_LINELEN); i++)
	printf("%s ", what);
      printf("\n");
      exit(0);
      break;
    case 'V':
      printf("showspace version %s\n", VERSION);
      exit(0);
    case '?':
      usage(argv[0], 1);
      break;
    }
  }

  llog_init(LLOG_STDERR);
  llog_level(verbose);
  
  llog(1, "generator: %s, options %s\n", generator, opts?opts:"None");
  show_space(generator, opts, what, start);
  llog_close();

  exit(0);
}
