// -*- c++ -*-
#ifndef INCLUDED_MATH3D_MQUAT_H
#define INCLUDED_MATH3D_MQUAT_H
/*
 * Math3d - The 3D Computer Graphics Math Library
 * Copyright (C) 1996-2000 by J.E. Hoffmann <je-h@gmx.net>
 * All rights reserved.
 *
 * This program is  free  software;  you can redistribute it and/or modify it
 * under the terms of the  GNU Lesser General Public License  as published by 
 * the  Free Software Foundation;  either version 2.1 of the License,  or (at 
 * your option) any later version.
 *
 * This  program  is  distributed in  the  hope that it will  be useful,  but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or  FITNESS FOR A  PARTICULAR PURPOSE.  See the  GNU Lesser General Public  
 * License for more details.
 *
 * You should  have received  a copy of the GNU Lesser General Public License
 * along with  this program;  if not, write to the  Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: mquat.h,v 1.5 2000/10/09 12:17:54 jeh Exp $
 */

#ifndef INCLUDED_MATH3D_MATH3DDEF_H
#include <math3d/math3ddef.h>
#endif

namespace Math3d {
  
  class M2d;
  class M3d;
  class M4x4;

/**
 * Quaternion (hypercomplex number) class.
 */
  class _CCMATH3D MQuat {
    protected:
      double d_v[4];
    public:
      MQuat() {d_v[0]=0.0; d_v[1]=0.0, d_v[2]=0.0; d_v[3]=1.0;};
      MQuat(double x, double y, double z, double w);
      MQuat(const float q[4]);
      MQuat(const double q[4]);
      MQuat(const M3d& axis, double angle);
      MQuat(const MQuat& q);
      MQuat(const M4x4& m);
      const MQuat& operator=(const MQuat& q);

      void zero();
      void identity();
      void set(double x, double y, double z, double w);
      void set(const M3d& axis, double angle);
      void set(const M4x4& M);
      void copy(const MQuat& q);
      
      inline double& operator[](int i);
      operator double*() {return(d_v);}
      double& x() {return(d_v[0]);}
      double& y() {return(d_v[1]);}
      double& z() {return(d_v[2]);}
      double& w() {return(d_v[3]);}
      double& get(int i);

      MQuat operator*(const MQuat& q)  const;
      const MQuat& operator*=(const MQuat& q);
      MQuat operator*(double k)  const;
      const MQuat& operator*=(double k);
      void neg();
      void abs();
      void cnj();
      void mul(const MQuat& p, const MQuat& q);
      void scalar(double k);
      void normalize();
      void inv();
      void ln();
      void lnDif(const MQuat& p, const MQuat& q);
      void exp();
      void slerp(const MQuat& p, const MQuat& q, double t);
      void squad(const MQuat& p, const MQuat& Tp, const MQuat& Tq, 
        const MQuat& q, double t);
      void tangent(const MQuat& qp, const MQuat& q, const MQuat& qn);
      void trackball(const M2d& p1, const M2d p2);

      inline double operator[](int i) const;
      operator const double*() const {return(d_v);}
      double x() const {return(d_v[0]);}
      double y() const {return(d_v[1]);}
      double z() const {return(d_v[2]);}
      double w() const {return(d_v[3]);}
      double get(int i) const;
      
      bool operator==(const MQuat& q) const;
      bool operator!=(const MQuat& q) const;
      double dot(const MQuat& q) const;
      bool cmp(const MQuat& q, double epsilon=EPSILON) const;
      double squared() const;
      double length() const;

      friend M4x4;
};

  inline double&
  MQuat::operator[](int i)
  {
    ASSERT(i>=0 && i<4);
    return(d_v[i]);
  }
  inline double
  MQuat::operator[](int i) const
  {
    ASSERT(i>=0 && i<4);
    return(d_v[i]);
  }

  extern _CCMATH3D ostream& operator << (ostream& co, const MQuat& q);

}
#endif // INCLUDED_MATH3D_MQUAT_H







