// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
#ifndef PACKAGE_NAMESPACE_POINTERLIST_H_INCLUDED
#define PACKAGE_NAMESPACE_POINTERLIST_H_INCLUDED
#include <string>
#include <list>
#include <map>
#define POINTERLIST_USES_VECTOR 0	// define to true if you want to use a vector instead of a list (should be a bit faster?)
#if POINTERLIST_USES_VECTOR
#  include <vector>
#else
#  include <list>
#endif

namespace PACKAGE_NAMESPACE
{
	using namespace std;

	/**
           pointer_list is a simple template class for a container of pointers
           to T, plus some memory management features.

           It's usage is STL-like.
           
           Parameterized on:

           - ChildType: the type of object to hold pointers to.
           
           pointer_lists with auto_delete(true) often make useful ad-hoc
           poor-man's garbage collectors.

           Known caveats:

           Inserting the same pointer into the same list multiple
           times is fatal if the list's auto_delete() is enabled. It
           will crash in the list's dtor when it deletes the pointer
           twice. Multiple pointers to the same object are allowed
           because the STL containers allow it, and i want to conform
           with that basic behaviour.
        */
        template < class ChildType > class pointer_list
	{
	      public:
		/**
                   A typedef to make the rest of the implementation easier to write.
                 */
		typedef pointer_list < ChildType > ThisType;

		/**
                 value_type defines the child type of this list. Note
                 that it is a pointer type (because this is a /pointer
                 list/ ;), so don't be confused by constructs like:

                 value_type objptr = 0;

                 (note the distinct lack of a *).

                 This odd definition is required for compatibility
                 with std::vector and std::list. i would prefer to
                 define it without the *. :/
                 */
		typedef ChildType *value_type;

                /**
                   list_type is the type of underlying pointer
                   container.
                */
#if POINTERLIST_USES_VECTOR
		typedef std::vector < value_type > list_type;
#else
		typedef std::list < value_type > list_type;
#endif

                /**
                   iterator which can be dereferenced to a (value_type *).
                 */
		typedef typename list_type::iterator iterator;

                /**
                   const iterator which can be dereferenced to a (value_type *).
                 */
		typedef typename list_type::const_iterator const_iterator;


                /**
                   Creates a pointer_list with the given auto_delete policy.
                */
                pointer_list( bool autodel = false ):m_autodel( autodel )
		{
		}

                /**
                   Deletes all child pointers if this->auto_delete().
                */
		virtual ~pointer_list()
		{
			if ( this->auto_delete() ) this->delete_all();
		}

		/**
                   If auto_delete() is true then all objects in this
                   list will be deleted when this object dies or when
                   erase() is called. Autodelete is false by default.
                */
                bool auto_delete() const
		{
			return this->m_autodel;
		}

                /**
                   Sets the autodelete policy. See auto_delete().
                */
		void auto_delete( bool autodel )
		{
			this->m_autodel = autodel;
		}


                /**
                   Return a const iterator for the first element in
                   this container.
                */
		typename ThisType::const_iterator begin() const
		{
			return this->list.begin();
		}

                /**
                   Return an iterator pointing to the first element in
                   this container.
                */
		typename ThisType::iterator begin()
		{
			return this->list.begin();
		}

                /**
                   Return the after-the-end iterator.
                */
		typename ThisType::const_iterator end() const
		{
			return this->list.end();
		}

                /**
                   Return the after-the-end iterator.
                */
		typename ThisType::iterator end()
		{
			return this->list.end();
		}


                /**
                   Returns an iterator pointing to the given pointer,
                   or end() if this object does not contain that
                   pointer.
                 */
		typename ThisType::iterator find( typename ThisType::value_type a )
		{
			typename ThisType::iterator iter = list.begin();
			typename ThisType::iterator enditer = list.end();
			for ( ; iter != enditer; ++iter )
			{
				if ( ( *iter ) == a )
				{
					//LIBE_CERR << "find( " << a << " ) found " << (*iter) << endl;
					return iter;
				}
			}
			return list.end();
		}


                /**
                   Returns the number of child pointers in this list.
                 */
		unsigned long count() const
		{
			return ( unsigned long ) this->list.size();	// i hope that's implemented constant-time. C++ Standard doesn't define performance on ::std::list::size().
		}
                /**
                   Same as count().
                 */
		unsigned long size() const
		{
			return ( unsigned long ) this->list.size();
		}

		/**
                   Adds the given pointer to the list. If front == true then the pointer
                   is added to the front of the list, else the end of the list.

                   If the pointer is NULL then this function does nothing.

                   Returns the pointer passed to it.
                */
		typename ThisType::value_type add( typename ThisType::value_type a, bool front = false )
		{
			if ( !a )
				return NULL;
			if ( !front )
				list.push_back( a );
			else
			{
#if POINTERLIST_USES_VECTOR
				list.insert( list.begin(), a );
#else
				list.push_front( a );
#endif
			}
			return a;
		}

                /**
                   For STL compatibility.

                   Adds an entry to the back of this list.
                */
		typename ThisType::value_type push_back( typename ThisType::value_type a )
		{
			return this->add( a, false );
		}

                /**
                   For STL compatibility.

                   Adds an entry to the front of this list.
                */
		typename ThisType::value_type push_front( typename ThisType::value_type a )
		{
			return this->add( a, true );
		}


                /**
                   Defines the deletion policies recognized by remove().
                */
		enum DeletionPolicy
		{
	                /**
                           Use the policy set in <code>auto_delete()</code>.
        	         */
			CheckAutoDelete = -1,
                        /**
                           Do not delete object.
                         */
			DoNotDelete = 0,
                        /**
                           Delete object.
                         */
			Delete = 1
		};

		/**
                   Removes the given pointer from this list, assuming
                   it contains the pointer. If the pointer is NULL or
                   the list does not contain the entry, it returns
                   NULL, otherwise it returns the pointer passed to
                   it.

                   The passed-in pointer may or may not be deleted,
                   depending on the value of deletionPolicy:
                   
                   - CheckAutoDelete (-1, default) == check auto_delete() and do whatever
                   that says.

                   - DoNotDelete (0) == do not delete

                   - Delete (1) == delete, regardless of auto_delete()

                   Obviously, if this function deletes the pointer
                   then it will return NULL.
                 */
		typename ThisType::value_type remove( typename ThisType::value_type a, DeletionPolicy deletionPolicy = CheckAutoDelete )
		{
                        // todo: try to replace this stuff with:
                        // std::remove_if( begin(), end(), std::equal_to<value_type>() );
			if ( !a )
				return NULL;
			typename ThisType::iterator iter = list.begin();
			while ( iter != list.end() )
			{
				if ( ( *iter ) == a )
				{
					list.erase( iter );
					if ( deletionPolicy > 0 || ( ( deletionPolicy == -1 ) && this->auto_delete() ) )
					{
						delete( a );
						return NULL;
					}
					return a;
				}
				++iter;
			}
			return NULL;
		}

                /**
                   For STL compatibility.

                   if ( this->auto_delete() ) then (*it) is deleted.

                 */
                void erase( ThisType::iterator it )
                {
                        if ( this->m_autodel ) delete( *it );
                        this->list.erase( it );
                }

                /**
                   For STL compatibility.

                   if ( this->auto_delete() ) then the pointers in (*it) are deleted.

                   TODO: validate that the iterators are guaranteed to
                   be valid after list_type::erase() is called!
                 */
                void erase( ThisType::iterator begin, ThisType::iterator end )
                {
			for( ; begin != end; ++begin )
			{
                                if( this->m_autodel ) delete( *begin );
                                this->list.erase( begin );
                        }
                }

                /**
                  For STL compatibility. Calls erase( begin(), end() ).
                 */
		void clear()
		{
                        this->erase( this->begin(), this->end() );
		}

		/**
                   Deletes all children, irrespective of auto_delete().
                */
		void delete_all()
		{
                        typedef std::map < typename ThisType::value_type, int > Protector;
			typename ThisType::value_type ptr;
#define POINTER_LIST_SAFE_DELETE 0
#if POINTER_LIST_SAFE_DELETE // causes a hang sometimes. See below.
			Protector delmap;	// to avoid multi-deletes.
#endif
			typename ThisType::iterator iter = list.begin();
			typename ThisType::iterator eter = list.end();
			for( ; iter != eter; ++iter )
			{
				ptr = ( *iter );
#if POINTER_LIST_SAFE_DELETE
                                if ( delmap.end() != delmap.find(ptr) ) continue; // hangs on the comparison sometimes!
                                delmap[ptr] = 0;
#endif
                                delete( ptr );
			}
                        list.clear();
#undef POINTER_LIST_SAFE_DELETE
		}

		/**
                   Returns true if this list contains a.
                */
		virtual bool contains( typename ThisType::value_type a )
		{
			return this->list.end() != this->find( a );
		}


	      private:
                pointer_list( const ThisType & ); // intentionally unimplemented.
                ThisType & operator=( const ThisType & ); // intentionally unimplemented.
		bool m_autodel;
		list_type list;

	}; // class pointer_list
} // namespace

/***
    These ostream operators are only useful for debuggering.
*/
using PACKAGE_NAMESPACE::pointer_list;
template < class Type > std::ostream & operator<<( std::ostream & os, const pointer_list < Type > &obj )
{
	typename pointer_list < Type >::const_iterator citer = obj.begin();
	os << "pointer_list@" << std::hex << &obj << ":";
	for ( ; citer != obj.end(); ++citer )
	{
		os << " ptr[";
		os << std::hex << ( *citer );
		os << "]";
	}
	return os;
}

/**
   Adds the given T pointer to the given pointer_list<T>.
*/
template < class Type >
PACKAGE_NAMESPACE::pointer_list < Type > & operator +=( PACKAGE_NAMESPACE::pointer_list < Type > &el, Type * ptr )
{
	el.add( ptr );
	return el;
}


/**
   Removes the  given T pointer using pointer_list<T>::remove().
*/
template < class Type >
PACKAGE_NAMESPACE::pointer_list < Type > & operator -=( PACKAGE_NAMESPACE::pointer_list < Type > &el, Type * ptr )
{
	el.remove( ptr );
	return el;
}

#endif // PACKAGE_NAMESPACE_POINTERLIST_H_INCLUDED
