#ifndef PACKAGE_NAMESPACE_ALGO_H_INCLUDED
#define PACKAGE_NAMESPACE_ALGO_H_INCLUDED 1

/////////////////////////////////////////////////////////////////////////
// algo.h: generic algorithms
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
/////////////////////////////////////////////////////////////////////////

#include <functional> // for_each()

#include "functor.h" // object_deleter()

namespace PACKAGE_NAMESPACE {


        /**
           For each item in (begin,end] object_deleter()(*item) is called.

           After this call the container from which the iterators come
           still contains the items but they are invalid - deleted
           pointers. The client should call erase(begin,end) to delete
           the entries, or use convenience function
           free_list_entries(), which accepts a list-style
           container.
        */
        template <typename IterT>
        void delete_objects( IterT begin, IterT end )
        {
                std::for_each( begin, end, object_deleter() );
        }



        /**
           Calls free_list_entries( c.begin(), c.end() )
           and then erases the entries from c using it's
           erase() member fuction.

           After this call, c will be empty.
        */
        template <typename ListType>
        void free_list_entries( ListType & c )
        {
                object_reference_wrapper<ListType> wr(c); // in case ListType is a pointer type
                if( ! wr.good() ) return;
                delete_objects( wr().begin(), wr().end() );
                wr().clear();
        }


       /**
           Deletes any pointers in map, and does nothing for reference
           types. This is used to treat arbitrary maps containing
           pointers or value types identically, by applying the same
           set of deallocation rules for both cases, simplifying some
           template implementations.
        */
        template <typename MapType>
        void free_map_entries( MapType & map )
        {
                object_reference_wrapper<MapType> wr(map); // in case MapType is a pointer type
                if( ! wr.good() ) return;
                std::for_each( wr().begin(), wr().end(), pair_entry_deallocator() );
                map.clear();
        }


}

#endif // PACKAGE_NAMESPACE_ALGO_H_INCLUDED
