#ifndef S11N_NS_DATA_NODE_FORMAT_H_INCLUDED
#define S11N_NS_DATA_NODE_FORMAT_H_INCLUDED
////////////////////////////////////////////////////////////////////////////////
// data_node_format.h
// Contains some helpers related to parsing/formating data_node-style objects.
//
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////////////


#include <string>
#include <list>
#include <map>
#include <stdexcept>
// #include <typeinfo>


#include <S11N_NS/to_string.h> // to/from_string()
#include <S11N_NS/debuggering_macros.h> // COUT/CERR
#include <S11N_NS/file_util.h> // get_i/ostream()
#include <S11N_NS/string_util.h> // translate_entities()


#include <S11N_NS/s11n_core.h> // classload()

#include "data_node_functor.h" // some utility functors
#include "data_node_serialize.h" // data_node_serializer<> and friends
#include "data_node_io.h" // default serializer interfaces
////////////////////////////////////////////////////////////////////////////////
// NO DEPS ON data_node.h ALLOWED!
////////////////////////////////////////////////////////////////////////////////

#ifndef yyFlexLexer // aaarrrgggg!
#  include "FlexLexer.h"
#endif
S11N_NS_CLASSLOADER_ABSTRACT_BASE(FlexLexer);


namespace S11N_NS {
        namespace io {

                namespace Private {
                        /**
                           A helper to hide FlexLexer subclasses from needing to be
                           included in header files. (FlexLexer's subclassing technique
                           (via macros) makes them impracticle to mix together
                           in the same headers.)

                           May throw std::runtime_error.
                         */
                        int lex_api_hider_yylex( FlexLexer *, std::istream & );

                }

                /**
                   A typedef representing a map of tokens used for
                   "entity translations" by s11n parsers/serializers.
                */
                typedef std::map<std::string,std::string> entity_translation_map;


                /**
                   tree_builder exists mainly so some lex-based code
                   can get access to a non-templated type (so we don't
                   have to hard-code the parsers to a node_type).

                   It provides only the interface needed by the current
                   lex-based parsers, not some ultimately reusable
                   interface.

                   It is not functionally useful by itself - it must
                   be subclassed and all of it's virtual methods must be
                   implemented.
                */
                class tree_builder
                {
                public:
                        tree_builder() : m_autodel(true) {}

                        virtual ~tree_builder() {}

                        /**
                           Starts a new node with the the given class
                           name and node name.

                           Return value indicates success or failure.
                        */
                        virtual bool
                        open_node( const std::string & classname, const std::string & nodename ) = 0;

                        /**
                           Closes the current node.

                           Return value indicates success or failure.
                        */
                        virtual bool
                        close_node() = 0;

                        /**
                           Sets property key to val for the current node.

                           Return value indicates success or failure.
                        */
                        virtual bool
                        add_property( const std::string & key, const std::string & val ) = 0;

                        /**
                           Returns the depth level of the parser,
                           where the root node is 1.
                        */
                        virtual size_t node_depth() const = 0;


                        /**
                           Changes the implementation class name of
                           the current node.
                        */
                        virtual bool change_node_class( const std::string & newclassname ) = 0;

                        /**
                           If auto_delete() is on (the default) then
                           this object should delete it's children
                           when it is destroyed, otherwise it will
                           not. It is up to subclasses to honor this,
                           as this base type does no handling of
                           children.
                        */
                        void auto_delete( bool b )
                        {
                                this->m_autodel = b;
                        }

                        /**
                           This is the getter for auto_delete( bool ).
                        */
                        bool auto_delete() const
                        {
                                return this->m_autodel;
                        }


                private:
                        bool m_autodel;

                };



                /**
                   tree_builder_context is a helper for accessing some
                   template-dependent code from non-template-aware
                   lexer code. It's usage is admitedly a bit strange
                   (indeed, it's whole existance is).

                   This object sets up a "context channel" where
                   a given FlexLexer can, in a thread-safe manner,
                   communicate data back to a data_tree_builder<NodeType>
                   without knowing the exact NodeType.

                   For samples see the s11n lexers, under src/node/lex.

                   All of it's methods, except for builder(), mirror
                   those of a tree_builder object, so see that class
                   for the API docs. For the "mirrored" functions, the
                   object being proxied is that set via builder(). It
                   is intended that only the lexers associated with
                   this context actually use it's API.

                   Calling the proxied functions when no builder is
                   set has no effect. Calling them with no bind()ed 
                   FlexLexer may be fatal.

                */
                template <typename ContextT>
                class tree_builder_context
                {
                public:
                        /** The context type for this class. */
                        typedef ContextT context_type;
                        /**
                           Sets the current builder object for this context.

                           Pointer ownership does not change by
                           calling this function.

                           This must be carefully marshalled: it must
                           always be set immediately before the
                           matching lexer is used, and unbind(lexer)
                           should be called immediately afterwards to
                           free up the internal marshaling data. Failing
                           to call unbind will mean a resource leak
                           (albeit a small one).

                           Preconditions:

                           - lexer and builder must be valid pointers
                           and must out-live the expected lifetime of
                           this context object, which internally
                           associates these two objects.
                        */
                        static void bind( const FlexLexer * lexer, tree_builder * builder )
                        {
                                lmap()[lexer].builder = builder;
                        }

                        /**
                           Frees up the internal resources used by the
                           marshaling process for the given lexer.
                        */
                        static void unbind( const FlexLexer * lexer )
                        {
                                lmap().erase( lexer );
                        }

                        /**
                           Gets the current builder object for this
                           context, which must have been previously
                           set up via a call to bind(lexer,builder).

                           Ownership of the returned pointer does not
                           change by calling this function.
                        */
                        static tree_builder * builder( const FlexLexer * lexer )
                        {
                                return lmap()[lexer].builder;
                        }


#define IFNOLEXER(RET) if( lmap().end() == lmap().find(lexer) ) return RET;
                        /**
                           See tree_builder::open_node().
                        */
                        static bool open_node( const FlexLexer * lexer,
                                               const std::string & classname,
                                               const std::string & nodename )
                        {
                                IFNOLEXER(false);
                                return lmap()[lexer].builder->open_node( classname, nodename );
                        }

                        /**
                           See tree_builder::clode_node().
                        */
                        static bool close_node( const FlexLexer * lexer )
                        {
                                IFNOLEXER(false);
                                return lmap()[lexer].builder->close_node();
                        }

                        /**
                           See tree_builder::add_property().
                        */
                        static bool
                        add_property(  const FlexLexer * lexer,
                                       const std::string & key,
                                       const std::string & val )
                        {
                                IFNOLEXER(false);
                                return lmap()[lexer].builder->add_property( key, val );
                        }

                        /**
                           See tree_builder::node_depth().
                        */
                        static size_t node_depth(  const FlexLexer * lexer )
                        {
                                IFNOLEXER(0);
                                return lmap()[lexer].builder->node_depth();
                        }

                        /**
                           See tree_builder::change_node_class().
                        */
                        static bool change_node_class(  const FlexLexer * lexer,
                                                        const std::string & newclassname )
                        {
                                IFNOLEXER(false);
                                return lmap()[lexer].builder->change_node_class( newclassname );
                        }
#undef IFNOLEXER

                        /**
                           This is intended for direct access by a lexer associated
                           with this context, and ONLY by such lexers.

                           Except for the builder member, these are
                           temporary holding points for vars common to
                           most lexers, placed here to avoid using
                           global data in the lexer code.
                        */
                        struct lexer_metadata
                        {
                                tree_builder * builder;

                                size_t internaldepth; // current internal depth (not always the same as node_depth())
                                std::string nodename; // name of current node
                                std::string nodeclass; // class name of current node
                                std::string property; // property value buffer
                                std::string bufferyy; // lexer-dependent
                                lexer_metadata()
                                {
                                        builder = 0;
                                        internaldepth = 0;
                                        nodename = nodeclass = property = bufferyy = "";
                                }
                        };

                        /**
                           Returns the lexer_metadata for the given lexer, creating one
                           if needed. It is assumed that the lexer has been bound via a
                           call to bind().
                        */
                        static lexer_metadata & metadata( const FlexLexer * lexer )
                        {
                                return lmap()[lexer];
                        }

                private:
                        /** Convenience typedef. */
                        typedef tree_builder_context<context_type> this_type;
                        /** lexer-to-metadata map */
                        typedef std::map<const FlexLexer *,lexer_metadata> lexer_map;
                        static lexer_map & lmap()
                        {
                                return S11N_NS::phoenix<
                                        lexer_map,
                                        this_type
                                        >::instance();
                        }

                };

                /**
                   data_node_tree_builder is a helper class for
                   building trees from deserialized data, designed
                   particularly for use with lex/callback-based tree
                   builders.

                   It owns all objects which build up it's tree. If
                   you want them you must manually remove them from the
                   container. You normally do not want them, however - they're
                   mostly throwaway nodes on their way to becoming fully
                   deserialized objects.

                   This class only provides methods for building a tree, not
                   for traversing it. Once you have built a tree, traverse it
                   starting at the root_node().

                   Based on usage conventions this type supports only
                   a single root node.
                */
                template <typename NodeType>
                class data_node_tree_builder : public tree_builder
                {
                public:
                        typedef NodeType node_type;

                        typedef std::list< node_type * > child_list_type;

                        /** Creates a default builder. */
                        data_node_tree_builder() : m_node_count(0), m_node(0),m_root(0)
                        {
                        }

                        /**
                           Deletes this object's children if
                           auto_delete() returns true.
                        */
                        virtual ~data_node_tree_builder()
                        {
                                if( this->auto_delete() && this->m_root )
                                {
                                        //CERR << "data_node_tree_builder<> cleaning up root node.\n";
                                        delete( this->m_root );
                                }
                                else
                                {
                                        //CERR << "data_node_tree_builder<> was relieved of child duty (or had no child).\n";
                                }
                        }


                        /**
                           Opens a new node, making that the current node.
                           classname will be used for the node's impl_class()
                           (see docs for node_type::impl_class()). name will
                           be the object's name, which is important for
                           de/serializing the node (see node_type::name()).

                           It returns false on error, else true. The default
                           implementation has no error conditions, and
                           therefor always returns true.

                           Node that classnames and node names need not be
                           unique (nor make up unique combinations). Any
                           number of nodes may have the same name or
                           classname.
                        */
                        bool open_node( const std::string & classname, const std::string & nodename )
                        {
                                ++m_node_count;

                                this->m_node = ( this->m_nodestack.empty() 
                                                 ? 0
                                                 : this->m_nodestack.back() );
                                node_type * newnode = new node_type();
                                if ( m_node )
                                { // if we're in a node, add new node as a child to that one:
                                        m_node->children().push_back( newnode );
                                }
                                this->m_node = newnode;
                                m_node->name( nodename );
                                m_node->impl_class( classname );
                                this->m_nodestack.push_back( m_node );
                                bool ret = true;
                                if ( 1 == this->m_nodestack.size() )
                                {
                                        if( m_root )
                                        {
                                                CERR << "open_node("<<classname<<","<<nodename<<") WARNING: deleting extra root node!\n";
                                                delete( m_node );
                                                ret = false;
                                        }
                                        else
                                        {
                                                m_root = m_node;
                                        }
                                }
                                return ret;
                        }

                        /**
                           Closes the most-recently-opened node, effectively
                           popping the previous node off of the node stack (it
                           is not destroyed).  It is an error to call this more
                           often than calling open_node().

                           It returns false on error (e.g., called
                           with no opened node).
                        */
                        virtual bool close_node()
                        {
                                if ( !m_node || m_nodestack.empty() )
                                {
                                        CERR << "close_node() error: called with an empty node stack!" << std::endl;
                                        return false;
                                }
                                m_nodestack.pop_back();
                                if ( m_nodestack.empty() )
                                {
                                        m_node = NULL;
                                }
                                else
                                {
                                        m_node = m_nodestack.back();
                                }
                                return true;
                        }


                        /**
                           Adds the given key/value pair to the
                           current node and returns true. If no node
                           is currently opened it returns false.
                        */
                        virtual bool add_property( const std::string & key, const std::string & val )
                        {
                                if( ! this->m_node ) return false;
                                this->m_node->set( key, val );
                                return true;
                        }

                        /**
                           Returns the total number of nodes opened via open_node().
                        */
                        size_t node_count() const
                        {
                                return m_node_count;
                        }


                        /**
                           Returns the current depth of opened nodes. A return
                           value of 1 means the current node is the root node,
                           for example, and 0 means that no node has yet been
                           opened.
                        */
                        size_t node_depth() const
                        {
                                return m_nodestack.size();
                        }


                        /**
                           Returns the most recent root node parsed out of the
                           input object.

                           Use auto_delete() to determine ownership of
                           the returned pointer.
                        */
                        node_type * root_node() const
                        {
                                return m_root;
                        }


                        /**
                           Returns the current node.

                           Use auto_delete() to determine ownership of
                           the returned pointer.
                        */
                        node_type * current_node() const
                        {
                                return m_node;
                        }

                        /**
                           Changes class name of current node, if one
                           is set. Returns false only if no node is
                           currently opened, else it returns true.
                        */
                        virtual bool change_node_class( const std::string & newclassname )
                        {
                                if( ! this->m_node ) return false;
                                this->m_node->impl_class( newclassname );
                                return true;
                        }

                private:
                        size_t m_node_count;
                        node_type * m_node;
                        node_type * m_root;
                        typedef std::deque < node_type * > node_stack;
                        node_stack m_nodestack;                        
                };



                /**
                   This function exists for a really long, strange
                   reason involving accessing templatized types from
                   template-free code (FlexLexers).

                   - lexerClassName is the name of a FlexLexer
                   subclass. It must be registered with the FlexLexer
                   classloader.

                   - src is the stream to pass on to the lexer.

                   - BuilderContext should be the same one expected by
                   the specific lexer. See the existing lexers for
                   examples. You want to pass the actual
                   BuilderContext's context here, not a
                   tree_builder_context<> type.

                   The caller owns the returned poiner, which may be 0.
                */
                template <typename NodeType, typename BuilderContext>
                NodeType * deserialize_lex_forwarder( const std::string & lexerClassName,
                                                      std::istream & src
                                                      )
                {
                        // CERR << "deserialize_lex_forwarder("<<lexerClassName<<")\n";
                        FlexLexer * lexer = S11N_NS::classload<FlexLexer>( lexerClassName );
                        if( ! lexer )
                        {
                                CERR << "Lexer '"<<lexerClassName
                                     <<"' was not found by classload<FlexLexer>()."
                                     << " It is probably not registered with class_loader<FlexLexer>.\n";
                                return 0;
                        }

                        typedef S11N_NS::io::data_node_tree_builder<NodeType> BuilderType;
                        typedef tree_builder_context<BuilderContext> BC;

                        NodeType * ret = 0;
                        BuilderType * treebuilder = new BuilderType();
                        treebuilder->auto_delete( false ); // we want to steal it's nodes.
                        bool err = false;
                        try
                        {
                                BC::bind( lexer, treebuilder );
                                // ^^^ sets up the comm channel between the builder and lexer
                                /**
                                   Bug-in-waiting: we don't yet have a way of knowing
                                   if a lexer partially populates the builder.
                                */
                                Private::lex_api_hider_yylex(lexer,src);
                        }
                        catch ( std::runtime_error & ex )
                        {
                                err = true;
                                CERR << "deserialize_lex_forwarder(): Doh! Exception during lexing: " << ex.what() << "\n";
                        }
                        catch (...)
                        {
                                err = true;
                                CERR << "deserialize_lex_forwarder(): Doh! Unknown exception during lexing:\n";
                        }
                        BC::unbind( lexer ); // free up lexer-to-builder binding
                        delete( lexer );
                        if( err )
                        {
                                treebuilder->auto_delete( true ); // let it delete it's children
                        }
                        else
                        {
                                ret = treebuilder->root_node();
                        }
                        // CERR << "Loaded node: " << std::hex << ret << '\n';
                        delete( treebuilder );
                        return ret;
                }

                /**
                   tree_builder_lexer is a type intended to ease the
                   implementation of lex-based node tree parsers.

                   It is useless standalone: it must be subclassed.

                   It holds the class name of a FlexLexer type so it
                   can be dynamically loaded as needed. Also, for
                   subclasses it takes the responsibility of
                   instantiating this type and passing off input to
                   it.
                */
                template <typename NodeType, typename LexerSharingContext>
                class tree_builder_lexer : public data_node_serializer<NodeType>
                {
                public:

                        typedef NodeType node_type;
                        typedef LexerSharingContext sharing_context;

                        /**
                           lexerClassName = the class name of the FlexLexer subtype
                           associated with this serializer.
                        */
                        explicit tree_builder_lexer( const std::string & lexerClassName )
                                : m_impl(lexerClassName)
                        {}

                        virtual ~tree_builder_lexer(){}

                        /**
                           Overridden to parse src using this object's lexer.
                           It uses <code>deserialize_lex_forwarder<sharing_context>()</code>,
                           passing it this object's lexer_class().
                        */
                        virtual node_type * deserialize( std::istream & src )
                        {
                                return deserialize_lex_forwarder<
                                        node_type,
                                        sharing_context
                                        >( this->lexer_class(), src );
                        }


                        /**
                           Returns this object's lexer class name.
                        */
                        std::string lexer_class() const { return this->m_impl; }


                protected:
                        /**
                           Sets this object's lexer class name.
                        */
                        void lexer_class( const std::string & classname )
                        {
                                this->m_impl = classname;
                        }

                private:
                        std::string m_impl; // implementation class name for a FlexLexer subclass
                };

        } // namespace io
} // namespace S11N_NS

S11N_NS_CLASSLOADER_ABSTRACT_BASE(S11N_NS::io::tree_builder);

#endif // S11N_NS_DATA_NODE_FORMAT_H_INCLUDED
