// A module for reading outline files. -*- c++ -*-

#include <module.h>
inherit "module";
inherit "roxenlib";

#if !efun(implode)
import Simulate;
#endif

void create()
{
    defvar("extension", "outline", "Outline file extension", TYPE_STRING,
	   "All files ending with this extension will be parsed as outline files.");
    defvar("foldedimage", "/internal-roxen-fold", 
	   "Folded image", TYPE_STRING,
	   "The url of the image used to show a folded structure.");
    defvar("unfoldedimage", "/internal-roxen-unfold", 
	   "Unfolded image", TYPE_STRING,
	   "The url of the image used to show a folded structure.");
}

mixed * register_module()
{
    return ({ MODULE_FILE_EXTENSION,
		"Outline file handler module",
		"Presentation of outline files.",
		({ ({ "Empty all outline files.", "reset_buffer", }), })
		});
}

// Reads an outline file and returns an array structure.
// This also has an array of remembered files so it won't have to explode
// them again and can return a part of a html page with the fold and unfold
// depending on what we gave as input.
// 
// For an outline file like:
// Bupp
// * Question 1
// * Question 2
// Hepp
// ** Subquestion 2.1
// Hipp
// ** Subquestion 2.2
// hopp
// * Question 3
//
// Important format: NEWLINE START STAR* SPACE
// The structure looks like:
// ({ "Bupp",
//    ({ "Question 1", 1, ({ "", })}),
//    ({ "Question 2", 2, ({ "Hepp",
//			     ({ "Subquestion 2.1", 3, ({ "Hipp", }), }),
//			     ({ "Subquestion 2.2", 4, ({ "hopp", }), }), }),
//    ({ "Question 3", 5, ({ "", }), })
// })


int number;

// Split while there is something to split.
mixed *
split(string what_to_split, int levels)
{
    string scanstr = "\n";
    int i;
    string * exploded;
    mixed * result;

    for (i = 0; i < levels; i++)
        scanstr += "*";
    scanstr += " ";

    exploded = explode("\n" + what_to_split, scanstr);

    result = ({ exploded[0], });

    for (i = 1; i < sizeof(exploded); i++)
    {
        string title, rest;

	if (sscanf(exploded[i], "%s\n%s", title, rest) == 2)
	    ;
	else
	{
	    title = exploded[i];
	    rest = "";
	}
        result += ({ ({ title, number ++,
			rest == "" 
			? ({ "", }) 
			: split(rest, levels + 1), }),
		       });
    }
    
    return result;
}

mapping
initiate(string contents, string path)
{
    mapping files = ([ ]);

    number = 1;

    files["info"] = split(contents, 1);
    files["number"] = number + 1; // Anvnds fr all!
    files["path"] = path;

    return files;
}

// The url argument contains information about whether or not we are to
// show every part of the message as folded or unfolded.
// It is the presence of the numbers in the list of unfolded that tells us
// about it.
#ifdef ROXEN
multiset thislist;
#else
list thislist;
#endif
int
unfolded(int element)
{
    return thislist[element];
}

string
#ifdef ROXEN
unfoldedlist(multiset newlist)
#else
unfoldedlist(list newlist)
#endif
{
    return implode(map_array(indices(newlist),
			     lambda(int arg) { return sprintf("%d", arg); }),
		   ",");
}


// Return a generate part of the page.
string
generate_part(mixed * arr, string base, int level)
{
    if (sizeof(arr) == 0)
        return "";

    return "<dl>\n"
      + implode(map_array(arr, lambda(mixed * arr, string base, int level)
    {
        if (unfolded(arr[1]))
	{
	    return "<dt><h" + level + "><a href=\"" + base + "?" 
	      + unfoldedlist(thislist - (<arr[1]>))
	      + "#p" + arr[1] + "\" name=\"p" + arr[1] + "\">"
	      + "<img border=0 src=\"" + query("foldedimage")
	      + "\" alt=\"\\/\"></a>"
	      + arr[0]
	      + "</h" + level + ">\n"
	      + "</dt>\n"
	      + "<dd>\n"
	      + arr[2][0]
	      + generate_part(arr[2][1..10000], base, level + 1)
	      + "</dd>\n";
	}
	else
	{
	    return "<dt><h" + level + "><a href=\"" + base + "?" 
	      + unfoldedlist(thislist + (<arr[1]>))
	      + "#p" + arr[1] + "\" name=\"p" + arr[1] + "\">"
	      + "<img border=0 src=\"" + query("unfoldedimage")
	      + "\" alt=\"--\"></a>"
	      + arr[0]
	      + "</h" + level + ">\n"
	      + "</dt>\n"
	      + "<dd>\n"
	      + "</dd>\n";
	}
    },
	base,
	level),
		   "\n")
      + "</dl>\n";
}


string
generate_dump_legal(mixed * arr, int level, string prefix)
{
    int i;
    string output = "";

    if (sizeof(arr) == 0)
	return output;

    for	(i = 0; i < sizeof(arr); i++)
    {
	output += "<h" + level + ">" + prefix + (i + 1) + ". " + arr[i][0] + "</h" + level + ">\n";
	output += arr[i][2][0] + "\n";
	output += generate_dump_legal(arr[i][2][1..10000], level + 1,
				      prefix + (i + 1) + ".");
    }
    return output;
}


// Returns a generated page.
string
generate(mapping thefile, string what)
{
    if (what == "all")
    {
        // Set all!
	int i;

#ifdef ROXEN
	thislist = aggregate_multiset();
#else
	thislist = aggregate_list();
#endif
	for (i = 1; i < thefile["number"]; i++)
	    thislist[i] = 1;
    }
    else if (what == "none")
    {
#ifdef ROXEN
        thislist = aggregate_multiset();
#else
        thislist = aggregate_list();
#endif
    }
    else if (what == "dump")
    {
	return thefile["info"][0]
	    + generate_dump_legal(thefile["info"][1..10000], 1, "");
    }
    else
#ifdef ROXEN
        thislist = aggregate_multiset(@map_array(explode(what || "0", 
							 ","),
						 lambda(string arg) { return (int)arg; }));
#else
        thislist = aggregate_list(@map_array(explode(what || "0", 
						     ","),
					     lambda(string arg) { return (int)arg; }));
#endif

    // Now, lets start generating the page.
    return thefile["info"][0]
      + generate_part(thefile["info"][1..10000], thefile["path"], 1)
      + "\n<hr>\n" 
      + "<a href=\"" + thefile["path"] + "?all\">Unfold all</a>\n|\n"
      + "<a href=\"" + thefile["path"] + "?none\">Fold all</a>\n|\n"
      + "<a href=\"" + thefile["path"] + "?dump\">Pretty dump</a>\n";
}

// The parse function is probably not needed in the module version.
string
parse(object request)
{
    mixed info = initiate(read_bytes("/home/epact/linus/.public/spinnerfaq/outlinedemo.out"), 
			  "outline.ulpc");

    return generate(info, request->variables->unfold);
}


string basename(string path)
{
    string * allparts = explode(path, "/");
    return allparts[sizeof(allparts) - 1];
}

// Here we do the handling as a module.
// This is a list of cached files. They are indexed on their filenames.
mapping outline_cache = ([ ]);

// Status information
int count_requests = 0;
int count_file_parseings = 0;

// #define DEBUG

mapping|string handle_file_extension(object file, string ext, object id)
{
#ifdef DEBUG
    perror(sprintf("File: %O\nExt: %O\nId: %O\nId->query: %O\nId->not_query: %O\nId->pragma: %O\n", file, ext, id, id->query, id->not_query, id->pragma));
#endif

    count_requests ++;
    if (!outline_cache[id->not_query]
	|| id->pragma["no-cache"])
    {
        // Read the file.
        mapping info = initiate(file->read(0x7ffffff), 
				basename(id->not_query));
	count_file_parseings ++;

	// Enter in the cache
	outline_cache[id->not_query] = info;

	call_out(lambda(string what) {
	    m_delete(outline_cache, what);
	}, 300 /* Five minutes */, id->not_query);	

#ifdef DEBUG
	perror(sprintf("Info: %O\n", info));
#endif
    }

    // Process and return the page.
#ifdef DEBUG
    perror(sprintf("Generate: %O\n", generate(outline_cache[id->not_query],
					      id->query)));
#endif

    return ([ "data": parse_rxml(generate(outline_cache[id->not_query],
					  id->query), id),
    "type":"text/html" ]);
}


string status()
{
    return "<dl>\n"
      + "<dt>Requests:<dd>" + count_requests
      + "<dt>File parsings:<dd>" + count_file_parseings
      + "<dt>Cached files:<dd>" + sizeof(outline_cache)
      + "</dl>";
}


array (string) query_file_extensions()
{
    return ({ query("extension"), });
}

