/***************************************************************************
 $RCSfile: limit.h,v $
 -------------------
 cvs         : $Id: limit.h,v 1.2 2003/06/04 10:50:33 cstim Exp $
 begin       : Fri Jan 17 2003
 copyright   : (C) 2003 by Fabian Kaiser
 email       : fabian@openhbci.de


 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef HBCI_LIMIT_H
#define HBCI_LIMIT_H

/** @file limit.h
 *
 * @short HBCI::Limit and its C wrappers @ref HBCI_Limit */

#ifdef __cplusplus
#include <string>
#endif /* __cplusplus */

#include <openhbci/dllimport.h>
#include <openhbci/value.h>

typedef enum LimitType {
  /** Limit per day */
  LIMIT_DAY = 1,
  /** Limit per job */
  LIMIT_JOB,
  /** Limit per week */
  LIMIT_WEEK,
  /** Limit per month */
  LIMIT_MONTH,
  /** so called "Rollierendes Zeitlimit" */
  LIMIT_TIME,
  /** internal indicator for an invalid limit */
  NO_LIMIT
} LimitType;

#ifdef __cplusplus
using namespace std;

namespace HBCI {

  class DLLIMPORT Limit {	
  public:
	/**
	 * Create a blank limit that will return <false> on calling isValid()
	 */
	Limit() {};

	/**
	 * Creates a new limit whith the specified limit data
	 * @param limitValue The value of the limit in currency of the account
	 * @param type Specifies the kind of limit
	 * @param days Only used if type() = LIMIT_TIME. The days between the 
	 * limits.
	 */
	Limit(const Value &limitValue, LimitType type, int days=0);	

	/**
	 * Returns <true> is this is a valid limit. Some banks tell us limits, 
	 * some don't, for some accounts there are limits, for others not. 
	 * If it is not valid, then we don't have any information about
	 * the existance of a limit.
	 */
	bool isValid() const;

	/**  
	 * Returns the value of the limit */
	const Value &value() const { return _limitValue; };

	/**
	 *  Returns the type of limit.
	 *
	 * @note: This is *not* the one-character type representation
	 * that is used in the HBCI communication. Instead, this is
	 * only a value from the enum LimitType. If you need the
	 * HBCI-compliant type character, then put the result from
	 * here through Limit::typeToChar.  */
	LimitType type() const { return _limitType; };

	/**  
	 * Returns the number of days for rotating time limits
	 * (Zeitlimit), if it is of type LIMIT_TIME
	 * (Zeitlimit). Otherwise this is zero.  */
	int daysForTimeLimit() const { return _limitDays; };

	/** Returns the enum value from a HBCI compliant one-character
	    type. */
	static LimitType typeFromChar(char type_char);
	/** Returns the HBCI compliant one-character type that belongs
	    to one particular enum value. */
	static char typeToChar(LimitType type_enum);
  private:
    LimitType _limitType;
    Value _limitValue;
    int _limitDays;
  };

} /* namespace HBCI */
typedef struct HBCI::Limit HBCI_Limit;

#else /* __cplusplus */
typedef struct Limit HBCI_Limit;
#endif /* __cplusplus */

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */
    /** @name HBCI_Limit */
    /*@{*/
    extern int HBCI_Limit_isValid(const HBCI_Limit *l);
    extern const HBCI_Value *HBCI_Limit_value(const HBCI_Limit *l);
    extern LimitType HBCI_Limit_type(const HBCI_Limit *l);
    extern int HBCI_Limit_daysForTimeLimit(const HBCI_Limit *l);
    extern void HBCI_Limit_delete(HBCI_Limit *d);
    /*@}*/
#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* HBCI_LIMIT_H */
