/***************************************************************************
 $RCSfile: ctfilesystem2.cpp,v $
                             -------------------
    cvs         : $Id: ctfilesystem2.cpp,v 1.5 2003/04/24 01:43:29 aquamaniac Exp $
    begin       : Fri Aug 09 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif


#if DEBUGMODE > 0
# include <stdio.h>
#endif

#include <string>
using namespace std;

#include "ctfilesystem2.h"
#include "libchipcard.h"
#include "ctmisc.h"


#define CTFILESYSTEM_FAT_FREE     0xff
#define CTFILESYSTEM_FAT_LAST     0xfe
#define CTFILESYSTEM_FAT_RESERVED 0xfd



/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                             CTBlockMedium
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


CTBlockMedium::CTBlockMedium(const CTCard &c)
:CTMemoryCard(c)
{
}


CTBlockMedium::~CTBlockMedium(){
}


CTError CTBlockMedium::readBlocks(int bn, int n, string &bl){
    CTError err;

#if DEBUGMODE>15
    fprintf(stderr,
            "CTBlockMedium::readBlocks()\n"
            "  First Block=%d, Blocks=%d\n",
            bn,n);
#endif

    bl.erase();
    try {
        err=readBinary(bl,
                       bn*CTFILESYSTEM_BASEBLOCKSIZE,
                       CTFILESYSTEM_BASEBLOCKSIZE*n);
    }
    catch (CTError lerr) {
        err=lerr;
    }
    if (!err.isOk())
        return err;
    if ((int)(bl.length())!=CTFILESYSTEM_BASEBLOCKSIZE*n)
        return CTError("CTBlockMedium::readBlocks()",
                       k_CTERROR_INVALID,0,0,
                       "bad data size");

    return CTError();
}


CTError CTBlockMedium::writeBlocks(int bn, int n, const string &bl){
    CTError err;

#if DEBUGMODE>15
    fprintf(stderr,
            "CTBlockMedium::writeBlocks()\n"
            "  First Block=%d, Blocks=%d\n",
            bn,n);
#endif
    if ((int)(bl.length())!=CTFILESYSTEM_BASEBLOCKSIZE*n)
        return CTError("CTBlockMedium::writeBlocks()",
                       k_CTERROR_INVALID,0,0,
                       "bad data size");

    try {
        err=updateBinary(bl,bn*CTFILESYSTEM_BASEBLOCKSIZE);
    }
    catch (CTError lerr) {
        err=lerr;
    }
    return err;
}


CTError CTBlockMedium::mountMedium(){
    return openCard();
}


CTError CTBlockMedium::unmountMedium(){
    // forced closure
    return closeCard(true);
}





/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                           CTCachedBlockMedium
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


CTCachedBlockMedium::CTCachedBlockMedium(const CTCard &c)
:CTBlockMedium(c)
{
}


CTCachedBlockMedium::~CTCachedBlockMedium(){
}


CTError CTCachedBlockMedium::readBlocks(int bn, int n, string &bl){
    int i;
    int j;
    int k;
    string tmp;
    CTError err;
    bool lastWasError;

#if DEBUGMODE>15
    fprintf(stderr,
            "CTCachedBlockMedium::readBlocks()\n"
            "  First Block=%d, Blocks=%d\n",
            bn,n);
#endif

    bl.erase();
    i=0;
    lastWasError=false;
    while (i<n) {
        tmp.erase();
        if (_cache.isValid(bn+i)) {
            //cache is valid
#if DEBUGMODE>15
            fprintf(stderr,"Cache hit on block %d\n",bn+i);
#endif
            tmp=_cache.data((bn+i)*CTFILESYSTEM_BASEBLOCKSIZE,
                            CTFILESYSTEM_BASEBLOCKSIZE);
            bl+=tmp;
            i++;
            lastWasError=false;
        }
        else {
            // cache is invalid
#if DEBUGMODE>15
            fprintf(stderr,"Cache miss on block %d\n",bn+i);
#endif
            if (lastWasError)
                return CTError("CTCachedBlockMedium::readBlocks()",
                               k_CTERROR_INVALID,0,0,
                               "error loop encountered.",
                               "This is an internal error, so please report"
                               " to <martin@libchipcard.de>\n");
            // count how many blocks are invalid for faster reading
            j=1;
            while((i+j)<n) {
                if (_cache.isValid(bn+i+j))
                    break;
#if DEBUGMODE>15
                fprintf(stderr,"Cache miss on block %d\n",bn+i+j);
#endif
                j++;
            } // while
            // j now has the number of consecutive invalid blocks, read them
            err=CTBlockMedium::readBlocks(bn+i,
                                          j,
                                          tmp);
            if (!err.isOk())
                return err;
            // write them to the cache
            _cache.setData(tmp,(bn+i)*CTFILESYSTEM_BASEBLOCKSIZE);
            // validate cache
            for (k=0; k<j; k++) {
                _cache.setValid(bn+i+k);
                _cache.setClean(bn+i+k);
            } // for

            /* we will retry in the next round, but remember that the cache
             * was invalid here to avoid error loops. Please note:
             * "i" is not incremented, so the next loop will retry and in
             * THAT attempt succeed.*/
            lastWasError=true;
        }
    } // while

    return CTError();
}


CTError CTCachedBlockMedium::writeBlocks(int bn, int n, const string &bl){
    int i;
    string tmp;

#if DEBUGMODE>15
    fprintf(stderr,
            "CTCachedBlockMedium::writeBlocks()\n"
            "  First Block=%d, Blocks=%d\n",
            bn,n);
#endif

    if ((int)(bl.length())!=CTFILESYSTEM_BASEBLOCKSIZE*n)
        return CTError("CTCachedBlockMedium::writeBlocks()",
                       k_CTERROR_INVALID,0,0,
                       "bad data size");

    // write data to the cache
    for (i=0; i<n; i++) {
        tmp=bl.substr(i*CTFILESYSTEM_BASEBLOCKSIZE,
                      CTFILESYSTEM_BASEBLOCKSIZE);
        if (_cache.isValid(bn+i)) {
            // cach block is valid, check if it is to be changed
            if (_cache.data((bn+i)*CTFILESYSTEM_BASEBLOCKSIZE,
                            CTFILESYSTEM_BASEBLOCKSIZE)!=tmp) {
                // it is, so change it
#if DEBUGMODE>15
                fprintf(stderr,"Cache data block %d changed.\n",
                        bn+i);
#endif
                _cache.setData(tmp,(bn+i)*CTFILESYSTEM_BASEBLOCKSIZE);
                _cache.setDirty(bn+i);
            } // if cache content is to be changed
            else {
                // otherwise the cache is not touched thus saving writing time
#if DEBUGMODE>15
                fprintf(stderr,"Cache data block %d not changed.\n",
                        bn+i);
#endif
            }
        }
        else {
            // cache block is not valid, so set it
            _cache.setData(tmp,(bn+i)*CTFILESYSTEM_BASEBLOCKSIZE);
            _cache.setDirty(bn+i);
            _cache.setValid(bn+i);
        }
    } // for

    return CTError();
}


int CTCachedBlockMedium::flush(int maxb) throw (class CTError){
    int i;
    int j;
    int k;
    CTError err;
    int blocksFlushed;

    i=0;
    blocksFlushed=0;
    while (i<CTFILESYSTEM_MAXSIZE/CTFILESYSTEM_BASEBLOCKSIZE &&
           maxb>0) {
        if (_cache.isValid(i) && _cache.isDirty(i)) {
            // block is valid but dirty
            // count number of consecucutive dirty blocks
            j=1;
            maxb--;
            while(
                (i+j)<(CTFILESYSTEM_MAXSIZE/CTFILESYSTEM_BASEBLOCKSIZE)
                && maxb>0) {
                if (!(_cache.isValid(i+j) && _cache.isDirty(i+j)))
                    break;
                j++;
                maxb--;
            } // while
            // j now has the number of consecutive invalid blocks, write them
            err=CTBlockMedium::writeBlocks(i,
                                           j,
                                           _cache.data(
                                               i*CTFILESYSTEM_BASEBLOCKSIZE,
                                               CTFILESYSTEM_BASEBLOCKSIZE*j));
            if (!err.isOk())
                throw err;
            blocksFlushed+=j;

            // clean cache
            for (k=0; k<j; k++)
                _cache.setClean(i+k);
            // advance
            i+=j;
        }
        else {
            // block is clean or invalid
            i++;
        }
    } // while

    return blocksFlushed;
}


void CTCachedBlockMedium::purge(){
  _cache.setNoneDirty();
  _cache.setNoneValid();
}


CTError CTCachedBlockMedium::mountMedium(){
    _cache.setNoneDirty();
    _cache.setNoneValid();
    return CTBlockMedium::mountMedium();
}


CTError CTCachedBlockMedium::unmountMedium(){
    CTError err1, err2;

    try {
        flush();
    }
    catch(CTError xerr) {
        err1=xerr;
    }

    err2=CTBlockMedium::unmountMedium();
    if (!err1.isOk())
        return err1;
    return err2;
}





/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                           CTCryptedBlockMedium
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


CTCryptedBlockMedium::CTCryptedBlockMedium(const CTCard &c)
:CTCachedBlockMedium(c)
,_desKeyIsValid(false)
{
}


CTCryptedBlockMedium::~CTCryptedBlockMedium(){
}


CTError CTCryptedBlockMedium::readBlocks(int bn, int n,
                                         string &bl,
                                         bool cr){
#if DEBUGMODE>15
    fprintf(stderr,
            "CTCryptedBlockMedium::readBlocks() [cr=%d]\n",
            cr);
#endif

    bl.erase();
    if (!cr)
        return CTCachedBlockMedium::readBlocks(bn,n,bl);
#ifndef CT_USE_ENCRYPTION
    return CTError("CTCryptedBlockMedium::readBlocks()",
                   k_CTERROR_INVALID,0,0,
                   "Encryption not supported.");
#else
    string tmp;
    CTError err;

    // first read the blocks
    err=CTCachedBlockMedium::readBlocks(bn,n,tmp);
    if (!err.isOk())
        return err;
    // then decrypt it
    return crypt(false,tmp,bl);
#endif
}


CTError CTCryptedBlockMedium::writeBlocks(int bn, int n,
                                          const string &bl,
                                          bool cr){
#if DEBUGMODE>15
    fprintf(stderr,
            "CTCryptedBlockMedium::writeBlocks() [cr=%d]\n",
            cr);
#endif

    if ((int)(bl.length())!=n*CTFILESYSTEM_BASEBLOCKSIZE)
        return CTError("CTCryptedBlockMedium::writeBlocks()",
                       k_CTERROR_INVALID,0,0,
                       "Data size does not match block size*number.");

    if (!cr)
        return CTCachedBlockMedium::writeBlocks(bn,n,bl);
#ifndef CT_USE_ENCRYPTION
    return CTError("CTCryptedBlockMedium::writeBlocks()",
                   k_CTERROR_INVALID,0,0,
                   "Encryption not supported.");
#else
    string tmp;
    CTError err;

    // first encrypt the blocks
    err=crypt(true,bl,tmp);
    if (!err.isOk())
        return err;
    // then write them
    return CTCachedBlockMedium::writeBlocks(bn,n,tmp);
#endif
}


CTError CTCryptedBlockMedium::setPassword(const string &pw){
#ifndef CT_USE_ENCRYPTION
    return CTError("CTCryptedBlockMedium::setPassword()",
                   k_CTERROR_INVALID,0,0,
                   "Encryption not supported.");
#else
    // make two 8 byte keys from the password
    des_string_to_2keys(pw.c_str(), &_desKey1, &_desKey2);
    _desKeyIsValid=true;
    return CTError();
#endif
}


void CTCryptedBlockMedium::clearPassword(){
#ifndef CT_USE_ENCRYPTION
    return CTError("CTCryptedBlockMedium::clearPassword()",
                   k_CTERROR_INVALID,0,0,
                   "Encryption not supported.");
#else
    int i;

    // invalidate keys
    _desKeyIsValid=false;
    // clear first key
    for (i=0; i<8; i++)
        _desKey1[i]=0;
    // clear second key
    for (i=0; i<8; i++)
        _desKey2[i]=0;
#endif
}


CTError CTCryptedBlockMedium::crypt(bool encrypt,
                                    const string &src,
                                    string &dest){
#ifndef CT_USE_ENCRYPTION
    return CTError("CTCryptedBlockMedium::crypt()",
                   k_CTERROR_INVALID,0,0,
                   "Encryption not supported.");
#else
    des_cblock ivec;
    des_key_schedule key1, key2;
    char *dstbuf;
    int i;

    // check is key is given
    if (!_desKeyIsValid) {
        dest=src;
        return CTError("CTCryptedBlockMedium::crypt()",
                       k_CTERROR_INVALID,0,0,
                       "Password not set.");
    }

    des_set_key_unchecked(&_desKey1, key1);
    des_set_key_unchecked(&_desKey2, key2);

    dstbuf=new char[src.length()];
    // preset init vector
    for (i=0; i<8; i++)
        ivec[i] = 0;
    // encrypt/decrypt
    des_ede2_cbc_encrypt((unsigned char*)src.c_str(),
                         (unsigned char*)dstbuf,
                         src.length(),
                         key1,
                         key2,
                         &ivec,
                         encrypt);
    // assign data to destination
    dest.assign(dstbuf,src.length());
    delete [] dstbuf;
#endif // CT_USE_ENCRYPTION
    return CTError();
}





/* __________________________________________________________________________
 * AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                               CTSuperBlock
 * YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


#define CTFS_SUPERBLOCK_LENGTH             48
#define CTFS_SUPERBLOCK_VERSION_MAJOR      0
#define CTFS_SUPERBLOCK_VERSION_MINOR      1
#define CTFS_SUPERBLOCK_BLOCKSIZE_HIGH     2
#define CTFS_SUPERBLOCK_BLOCKSIZE_LOW      3
#define CTFS_SUPERBLOCK_BLOCKS_HIGH        4
#define CTFS_SUPERBLOCK_BLOCKS_LOW         5
#define CTFS_SUPERBLOCK_MEDIUMSIZE_3       6
#define CTFS_SUPERBLOCK_MEDIUMSIZE_2       7
#define CTFS_SUPERBLOCK_MEDIUMSIZE_1       8
#define CTFS_SUPERBLOCK_MEDIUMSIZE_0       9
#define CTFS_SUPERBLOCK_FLAGS_HIGH         10
#define CTFS_SUPERBLOCK_FLAGS_LOW          11
#define CTFS_SUPERBLOCK_RESERVED_BLOCKS    12
#define CTFS_SUPERBLOCK_FIRST_DIRBLOCK     13

#define CTFS_SUPERBLOCK_CHECKSUM           15
#define CTFS_SUPERBLOCK_MEDIUMNAME         16
#define CTFS_SUPERBLOCK_USERNAME           32


CTSuperBlock::CTSuperBlock()
:_changed(false)
,_blockSize(0)
,_blocks(0)
,_mediumSize(0)
,_versionMajor(CTFILESYSTEM_VERSION_MAJOR)
,_versionMinor(CTFILESYSTEM_VERSION_MINOR)
,_flags(0)
{
}


CTSuperBlock::CTSuperBlock(const string &block)
:_changed(false)
,_blockSize(0)
,_blocks(0)
,_mediumSize(0)
,_versionMajor(CTFILESYSTEM_VERSION_MAJOR)
,_versionMinor(CTFILESYSTEM_VERSION_MINOR)
,_flags(0)
,_reservedBlocks(0)
,_firstDirBlock(0)
{
    CTError err;

    err=_fromString(block);
    if (!err.isOk())
        throw err;
}


CTSuperBlock::CTSuperBlock(unsigned int mediumsize)
:_changed(false)
,_blockSize(0)
,_blocks(0)
,_mediumSize(mediumsize)
,_versionMajor(CTFILESYSTEM_VERSION_MAJOR)
,_versionMinor(CTFILESYSTEM_VERSION_MINOR)
,_flags(0)
,_reservedBlocks(0)
,_firstDirBlock(0)
{
    int i;
    int p;
    int l;
    int datasize;

    // calculate data size
    // number of base blocks for superblock
    p=CTFS_SUPERBLOCK_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
    if (CTFS_SUPERBLOCK_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
        p++;

    // calculate number of base blocks for FAT
    l=CTFILESYSTEM_FAT_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
    if (CTFILESYSTEM_FAT_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
        l++;

    datasize=mediumsize-((p+l)*CTFILESYSTEM_BASEBLOCKSIZE);

    // calculate appropriate block size
    _blockSize=datasize/253;
    // size smaller than base block size ?
    if (_blockSize<CTFILESYSTEM_BASEBLOCKSIZE)
        _blockSize=CTFILESYSTEM_BASEBLOCKSIZE;
    // is there a modulo ?
    i=_blockSize % CTFILESYSTEM_BASEBLOCKSIZE;
    if (i) {
        // make block big enough to be divideable by base block size
        _blockSize+=(CTFILESYSTEM_BASEBLOCKSIZE-i);
    }

    // calculate block count (ignore modulo)
    _blocks=datasize/_blockSize;

    _changed=true;
}


CTSuperBlock::~CTSuperBlock(){
}


CTError CTSuperBlock::_fromString(const string &header){
    unsigned int tv;
    int i;
    unsigned char j;

    // read version
    _versionMajor=(unsigned char)header.at(CTFS_SUPERBLOCK_VERSION_MAJOR);
    if (_versionMajor!=CTFILESYSTEM_VERSION_MAJOR)
        return CTError("CTSuperBlock::_fromSting()",
                       k_CTERROR_INVALID,0,0,
                       "invalid file system version.");
    _versionMinor=(unsigned char)header.at(CTFS_SUPERBLOCK_VERSION_MINOR);

    // check checksum
    j=0;
    for (i=0; i<CTFS_SUPERBLOCK_LENGTH; i++)
        j=j^header[i];
    if (j) {
#if DEBUGMODE>0
        fprintf(stderr,"ERROR in checksum:\n");
#endif
        return CTError("CTSuperBlock::_fromSting()",
                       k_CTERROR_INVALID,
                       0,
                       0,
                       "invalid XOR-checksum.");
    }

    // read block size
    tv=((unsigned char)header.at(CTFS_SUPERBLOCK_BLOCKSIZE_HIGH))<<8;
    tv+=(unsigned char)header.at(CTFS_SUPERBLOCK_BLOCKSIZE_LOW);
    _blockSize=tv;

    // read blocks
    tv=((unsigned char)header.at(CTFS_SUPERBLOCK_BLOCKS_HIGH))<<8;
    tv+=(unsigned char)header.at(CTFS_SUPERBLOCK_BLOCKS_LOW);
    _blocks=tv;

    // read reserved blocks
    _reservedBlocks=(unsigned char)header.at(CTFS_SUPERBLOCK_RESERVED_BLOCKS);

    // read medium size
    tv=((unsigned char)header.at(CTFS_SUPERBLOCK_MEDIUMSIZE_3))<<24;
    tv+=((unsigned char)header.at(CTFS_SUPERBLOCK_MEDIUMSIZE_2))<<16;
    tv+=((unsigned char)header.at(CTFS_SUPERBLOCK_MEDIUMSIZE_1))<<8;
    tv+=(unsigned char)header.at(CTFS_SUPERBLOCK_MEDIUMSIZE_0);
    _mediumSize=tv;

    // read flags
    tv=((unsigned char)header.at(CTFS_SUPERBLOCK_FLAGS_HIGH))<<8;
    tv+=(unsigned char)header.at(CTFS_SUPERBLOCK_FLAGS_LOW);
    _flags=tv;

    // read medium name
    for (i=CTFS_SUPERBLOCK_MEDIUMNAME;
         i<CTFS_SUPERBLOCK_MEDIUMNAME+16;
         i++)
        if (header[i]==0)
            break;
    _mediumName=header.substr(CTFS_SUPERBLOCK_MEDIUMNAME,
                              i-CTFS_SUPERBLOCK_MEDIUMNAME);

    // read username name (it's possibly encrypted)
    _userName=header.substr(CTFS_SUPERBLOCK_USERNAME,16);

    // read first dir block
    _firstDirBlock=(unsigned char)header.at(CTFS_SUPERBLOCK_FIRST_DIRBLOCK);

    return CTError();
}


string CTSuperBlock::toString(){
    unsigned int tv;
    unsigned char buffer[CTFS_SUPERBLOCK_LENGTH];
    int i;
    unsigned char j;
    string result;

    // compose header
    memset(buffer,0,sizeof(buffer));

    // version
    buffer[CTFS_SUPERBLOCK_VERSION_MAJOR]=_versionMajor & 255;
    buffer[CTFS_SUPERBLOCK_VERSION_MINOR]=_versionMinor & 255;

    // block size
    tv=_blockSize;
    buffer[CTFS_SUPERBLOCK_BLOCKSIZE_HIGH]=(tv>>8) & 255;
    buffer[CTFS_SUPERBLOCK_BLOCKSIZE_LOW]=tv & 255;

    // blocks
    tv=_blocks;
    buffer[CTFS_SUPERBLOCK_BLOCKS_HIGH]=(tv>>8) & 255;
    buffer[CTFS_SUPERBLOCK_BLOCKS_LOW]=tv & 255;

    // flags
    tv=_flags;
    buffer[CTFS_SUPERBLOCK_FLAGS_HIGH]=(tv>>8) & 255;
    buffer[CTFS_SUPERBLOCK_FLAGS_LOW]=tv & 255;

    // mediumsize
    tv=_mediumSize;
    buffer[CTFS_SUPERBLOCK_MEDIUMSIZE_3]=(tv>>24) & 255;
    buffer[CTFS_SUPERBLOCK_MEDIUMSIZE_2]=(tv>>16) & 255;
    buffer[CTFS_SUPERBLOCK_MEDIUMSIZE_1]=(tv>>8) & 255;
    buffer[CTFS_SUPERBLOCK_MEDIUMSIZE_0]=tv & 255;

    // first dir block
    buffer[CTFS_SUPERBLOCK_FIRST_DIRBLOCK]=_firstDirBlock & 255;

    // medium name
    if (_mediumName.length()>16)
        throw CTError("CTSuperBlock::toString()",
                      k_CTERROR_INVALID,
                      0,
                      0,
                      "MediumName too long");
    for (i=0; i<(int)(_mediumName.length()); i++)
        buffer[CTFS_SUPERBLOCK_MEDIUMNAME+i]=_mediumName[i];

    // user name
    if (_userName.length()>16)
        throw CTError("CTSuperBlock::toString()",
                      k_CTERROR_INVALID,
                      0,
                      0,
                      "UserName too long");
    for (i=0; i<(int)(_userName.length()); i++)
        buffer[CTFS_SUPERBLOCK_USERNAME+i]=_userName[i];

    // XOR over all ("CHECKSUM" field in buffer is 0, so we may count it
    j=0;
    for (i=0; i<CTFS_SUPERBLOCK_LENGTH; i++) {
        j=j^buffer[i];
    }
    buffer[CTFS_SUPERBLOCK_CHECKSUM]=j;

    // create result
    result.assign((char*)buffer,CTFS_SUPERBLOCK_LENGTH);

    return result;
}


string CTSuperBlock::dump(){
    string result;

    result+="SuperBlock\n";
    result+="--------------------------------\n";
    result+="Medium Name     : ";
    result+=_mediumName+"\n";
    result+="Version         : ";
    result+=CTMisc::num2string(_versionMajor);
    result+=".";
    result+=CTMisc::num2string(_versionMinor)+"\n";
    result+="Medium Size     : ";
    result+=CTMisc::num2string(_mediumSize);
    result+=" bytes (";
    result+=CTMisc::num2string(_mediumSize,"%08x");
    result+=")\n";
    result+="Data Space      : ";
    result+=CTMisc::num2string(_blocks*_blockSize);
    result+=" bytes (";
    result+=CTMisc::num2string(_blocks*_blockSize,"%08x");
    result+=")\n";
    result+="Block Size      : ";
    result+=CTMisc::num2string(_blockSize);
    result+=" bytes (";
    result+=CTMisc::num2string(_blockSize,"%04x");
    result+=")\n";
    result+="Blocks          : ";
    result+=CTMisc::num2string(_blocks);
    result+=" (";
    result+=CTMisc::num2string(_blocks,"%04x");
    result+=")\n";
    result+="Reserved Blocks : ";
    result+=CTMisc::num2string(_reservedBlocks);
    result+=" (";
    result+=CTMisc::num2string(_reservedBlocks,"%04x");
    result+=")\n";
    result+="First Dir Block : ";
    result+=CTMisc::num2string(_firstDirBlock);
    result+=" (";
    result+=CTMisc::num2string(_firstDirBlock,"%04x");
    result+=")\n";

    return result;
}





/* __________________________________________________________________________
 * AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                               CTBlockManager
 * YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


CTBlockManager::CTBlockManager()
:_blocks(0)
,_changed(false)
{
    int i;

    // clear FAT
    for (i=0; i<(int)sizeof(_fat); i++)
        _fat[i]=CTFILESYSTEM_FAT_FREE;
}


CTBlockManager::CTBlockManager(int blocks, const string &fat)
:_blocks(blocks)
,_changed(false)
{
    int i;

    // clear FAT
    for (i=0; i<(int)sizeof(_fat); i++)
        _fat[i]=CTFILESYSTEM_FAT_FREE;

    // copy new fat in
    for (i=0; i<blocks; i++) {
        if (i>=(int)(fat.length()))
            break;
        else
            _fat[i]=fat[i];
    } // for
}


CTBlockManager::~CTBlockManager(){
}


int CTBlockManager::allocateBlock(int bl){
    int lastblock;
    int i;

#if DEBUGMODE>15
    fprintf(stderr,"CTBlockManager::allocateBlock()\n");
#endif

    if (bl>=_blocks || bl>=(int)sizeof(_fat)) {
#if DEBUGMODE>0
        fprintf(stderr,
                "Block number too high.\n");
#endif
        return -1;
    }

    // get last block
    if (bl!=-1) {
        lastblock=lastBlock(bl);
        if (lastblock==-1) {
#if DEBUGMODE>0
            fprintf(stderr,
                    "No last block.\n");
#endif
            return -1;
        }
    }
    else
        lastblock=-1;

    // allocate new block
    for (i=0; i<_blocks; i++) {
        if (i>(int)sizeof(_fat)) {
#if DEBUGMODE>0
            fprintf(stderr,
                    "Block number too high (2).\n");
#endif
            return -1;
        }
#if DEBUGMODE>0
        fprintf(stderr,"  FAT Entry[%d]: %d (%02x)\n",i,_fat[i], _fat[i]);
#endif
        if (_fat[i]==(unsigned char)CTFILESYSTEM_FAT_FREE) {
            // found a free block
            _fat[i]=CTFILESYSTEM_FAT_LAST;
            if (lastblock!=-1)
                _fat[lastblock]=i;
            _changed=true;
            return i;
        }
    } // for

    // no free block found
#if DEBUGMODE>0
    fprintf(stderr,
            "No free block found.\n");
#endif
    return -1;
}


void CTBlockManager::freeBlock(int bl){
    int previousblock;

    if (bl>=_blocks || bl>=(int)sizeof(_fat))
        return;

    previousblock=previousBlock(bl);

    _fat[bl]=CTFILESYSTEM_FAT_FREE;
    _changed=true;
    if (previousblock!=-1)
        _fat[previousblock]=CTFILESYSTEM_FAT_LAST;

}


void CTBlockManager::freeChain(int bn){
    int i;

    while(bn!=-1) {
        i=nextBlock(bn);
        freeBlock(bn);
        bn=i;
    } // while
}


int CTBlockManager::freeBlocks(){
    int b=0;
    int i;

    for (i=0; i<_blocks; i++) {
        if (i>(int)sizeof(_fat))
            return -1;
        if (_fat[i]==CTFILESYSTEM_FAT_FREE)
            b++;
    } // for

    return b;
}


int CTBlockManager::blocks(int bl){
    int i;
    int j;

    if (bl==-1)
        return _blocks;

    i=bl;
    j=1;
    while(i!=-1) {
        if (nextBlock(i)==-1)
            return j;
        i=nextBlock(i);
        j++;
    } // while
    return -1;
}


int CTBlockManager::nextBlock(int bl){
    if (bl>=_blocks || bl>=(int)sizeof(_fat))
        return -1;
    if (_fat[bl]==CTFILESYSTEM_FAT_LAST ||
        _fat[bl]==CTFILESYSTEM_FAT_FREE ||
        _fat[bl]==CTFILESYSTEM_FAT_RESERVED)
        return -1;
    return _fat[bl];
}


int CTBlockManager::previousBlock(int bl){
    int i;

    for (i=0; i<_blocks; i++) {
        if (i>(int)sizeof(_fat))
            return -1;
        if (_fat[i]==bl)
            return i;
    } // for
    return -1;
}


int CTBlockManager::lastBlock(int bl){
    int i;

    i=bl;

    while(i!=-1) {
        if (nextBlock(i)==-1)
            return i;
        i=nextBlock(i);
    } // while
    return -1;
}


int CTBlockManager::blockAt(int first, int idx) {
    int i;

    if (first==-1)
        return -1;

    i=first;
    while(1) {
        if (idx<1)
            break;
        i=nextBlock(i);
        if (i==-1)
            return -1;
        idx--;
    } // while

    return i;
}


string CTBlockManager::toString(){
    return string((char*)_fat,sizeof(_fat));
}






/* __________________________________________________________________________
 * AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                               CTDataBlockMedium
 * YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


CTDataBlockMedium::CTDataBlockMedium(const CTCard &c)
:CTCryptedBlockMedium(c)
,_isMounted(false)
{
#if DEBUGMODE>15
    fprintf(stderr,"CTDataBlockMedium::CTDataBlockMedium(c)\n");
#endif
}


CTDataBlockMedium::~CTDataBlockMedium(){
#if DEBUGMODE>15
    fprintf(stderr,"CTDataBlockMedium::~CTDataBlockMedium()\n");
#endif
}


CTError CTDataBlockMedium::_readFAT(){
    int l;
    int p;
    string fat;
    CTError err;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataBlockMedium::_readFAT()\n");
#endif

    // get number of first block behind superblock
    p=CTFS_SUPERBLOCK_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
    if (CTFS_SUPERBLOCK_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
        p++;

    // calculate number of base blocks to read
    l=CTFILESYSTEM_FAT_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
    if (CTFILESYSTEM_FAT_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
        l++;

    // read FAT
    err=CTCryptedBlockMedium::readBlocks(p,
                                         l,
                                         fat,
                                         _superBlock.isCrypted());
    if (!err.isOk())
        return err;

    // install blockmanager
    _blockManager=CTBlockManager(_superBlock.blocks(),
                                 fat);
    // done
    return CTError();
}


CTError CTDataBlockMedium::_writeFAT(){
    int l;
    int p;
    CTError err;
    string fat;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataBlockMedium::_writeFAT()\n");
#endif

    if (!_blockManager.changed())
        return CTError();

    // get number of first block behind superblock
    p=CTFS_SUPERBLOCK_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
    if (CTFS_SUPERBLOCK_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
        p++;

    // calculate number of base blocks to write
    fat=_blockManager.toString();
    l=fat.length()/CTFILESYSTEM_BASEBLOCKSIZE;
    if (fat.length() % CTFILESYSTEM_BASEBLOCKSIZE)
        l++;

    // write FAT
    err=CTCryptedBlockMedium::writeBlocks(p,
                                          l,
                                          fat,
                                          _superBlock.isCrypted());
    if (!err.isOk())
        return err;
    _blockManager.setChanged(false);

    return CTError();
}


CTError CTDataBlockMedium::_readSuperBlock(){
    string tmp;
    CTError err;
    int l;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataBlockMedium::_readSuperBlock()\n");
#endif

    // calculate number of base blocks to read
    l=CTFS_SUPERBLOCK_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
    if (CTFS_SUPERBLOCK_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
        l++;

    // read first l blocks (NO ENCRYPTION!)
    err=CTCryptedBlockMedium::readBlocks(0,
                                         l,
                                         tmp,
                                         false);
    if (!err.isOk())
        return err;

    try {
        _superBlock=CTSuperBlock(tmp);
    }
    catch(CTError lerr) {
        return lerr;
    }
#if DEBUGMODE>15
    fprintf(stderr,"%s\n",_superBlock.dump().c_str());
#endif

    return CTError();
}


CTError CTDataBlockMedium::_writeSuperBlock(){
    string tmp;
    int l;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataBlockMedium::_writeSuperBlock()\n");
#endif
    // nothing to do if no change
    if (!_superBlock.changed())
        return CTError();

#if DEBUGMODE>15
    fprintf(stderr,"%s\n",_superBlock.dump().c_str());
#endif

    // calculate number of base blocks to read
    l=CTFS_SUPERBLOCK_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
    if (CTFS_SUPERBLOCK_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
        l++;

    // create superblock string
    try {
        tmp=_superBlock.toString();
    }
    catch(CTError lerr) {
        return lerr;
    }

    // align if necessary
    if ((int)tmp.length()<l*CTFILESYSTEM_BASEBLOCKSIZE)
        tmp+=string(l*CTFILESYSTEM_BASEBLOCKSIZE-tmp.length(),
                    (char)0);

    // write first l blocks (NO ENCRYPTION!)
    return CTCryptedBlockMedium::writeBlocks(0,
                                             l,
                                             tmp,
                                             false);
}


CTError CTDataBlockMedium::_createMedium(unsigned int mediumSize,
                                         const string &mediumName,
                                         const string &userName,
                                         const string &passwd){
    CTError err;
    int i;
    string tmp;
    string tmp2;
    int p;
    int l;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataBlockMedium::_createMedium()\n");
#endif
    if (mediumSize>CTFILESYSTEM_MAXSIZE)
      return CTError("CTDataBlockMedium::_createMedium",
		     k_CTERROR_INVALID,
		     0,
		     0,
		     "Mediumsize too high");

    // initialize super block and FAT
    _superBlock=CTSuperBlock(mediumSize);
    _blockManager=CTBlockManager(_superBlock.blocks());
    _blockManager.setChanged(true);
    _superBlock.setMediumName(mediumName);
    if (!passwd.empty()) {
        _superBlock.setIsCrypted(true);
        err=setPassword(passwd);
        if (!err.isOk())
            return err;
        tmp=userName;
        // fill name with zeroes
        if (tmp.length()<16)
            tmp+=string(16-tmp.length(),(char)0);
        // encrypt name
        err=crypt(true,tmp,tmp2);
        if (!err.isOk())
            return err;
        // set encrypted username
        _superBlock.setUserName(tmp2);
    }

    // calculate number first data block
    // number of base blocks for superblock
    p=CTFS_SUPERBLOCK_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
    if (CTFS_SUPERBLOCK_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
        p++;

    // calculate number of base blocks for FAT
    l=CTFILESYSTEM_FAT_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
    if (CTFILESYSTEM_FAT_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
        l++;

    _firstDataBlock=p+l;
    _isMounted=true;

    // allocate block for dir
    i=-1;
    err=allocateBlock(i);
    if (!err.isOk())
      return CTError("CTDataBlockMedium::_createMedium",err);
    _superBlock.setFirstDirBlock(i);

    _isMounted=false;

    // save superblock
    err=_writeSuperBlock();
    if (!err.isOk())
      return CTError("CTDataBlockMedium::_createMedium",err);

    return _writeFAT();
}


CTError CTDataBlockMedium::createMedium(unsigned int mediumSize,
                                        const string &mediumName,
                                        const string &userName,
                                        const string &passwd){
  CTError err1, err2;

  // physically mount medium
  err1=CTCachedBlockMedium::mountMedium();
  if (!err1.isOk())
    return err1;

  err1=_createMedium(mediumSize,
		     mediumName,
		     userName,
		     passwd);
  err2=CTCachedBlockMedium::unmountMedium();
  _isMounted=false;

  if (!err1.isOk())
    return err1;
  return err2;
}


CTError CTDataBlockMedium::mountMedium(const string &username,
                                       const string &passwd){
    int p;
    int l;
    CTError err;
    string tmp;
    unsigned int i;

    if (_isMounted)
        return CTError("CTDataBlockMedium::mountMedium()",
                       k_CTERROR_INVALID,0,0,
                       "already mounted");

    try {
        // mount physically
        err=CTCryptedBlockMedium::mountMedium();
        if (!err.isOk())
            return err;

        // read superblock
        err=_readSuperBlock();
        if (!err.isOk()) {
            CTCryptedBlockMedium::unmountMedium();
            return err;
        }

        // set password, if crypted
        if (_superBlock.isCrypted()) {
            if (passwd.empty())
                return CTError("CTDataBlockMedium::mountMedium()",
                               k_CTERROR_AUTH,0,0,
                               "password needed");
            if (username.empty())
                return CTError("CTDataBlockMedium::mountMedium()",
                               k_CTERROR_AUTH,0,0,
                               "user name needed");

            // set password
            err=setPassword(passwd);
            if (!err.isOk()) {
                CTCryptedBlockMedium::unmountMedium();
                return err;
            }

            // decrypt user name from superblock
            err=crypt(false,_superBlock.userName(),tmp);
            if (!err.isOk())
                return err;
            // remove trailing zeroes
            for (i=0; i<tmp.length(); i++)
                if (tmp[i]==0)
                    break;
            tmp=tmp.substr(0,i);

            // compare username to see if the password was correct
            if (tmp!=username)
                return CTError("CTDataBlockMedium::mountMedium()",
                               k_CTERROR_INVALID,0,0,
                               "bad password");
        }

        // read FAT
        err=_readFAT();
        if (!err.isOk()) {
            CTCryptedBlockMedium::unmountMedium();
            return err;
        }

        // calculate number first data block
        // number of base blocks for superblock
        p=CTFS_SUPERBLOCK_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
        if (CTFS_SUPERBLOCK_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
            p++;

        // calculate number of base blocks for FAT
        l=CTFILESYSTEM_FAT_LENGTH/CTFILESYSTEM_BASEBLOCKSIZE;
        if (CTFILESYSTEM_FAT_LENGTH % CTFILESYSTEM_BASEBLOCKSIZE)
            l++;

        _firstDataBlock=p+l;

        // done.
        _isMounted=true;
        return CTError();
    } // try
    catch (CTError xerr) {
        return xerr;
    }
}


CTError CTDataBlockMedium::unmountMedium(){
    CTError err;

    if (!_isMounted)
        return CTError("CTDataBlockMedium::unmountMedium()",
                       k_CTERROR_INVALID,0,0,
                       "not mounted");

    // write FAT
    err=_writeFAT();
    if (!err.isOk())
        return err;

    // write SuperBlock
    err=_writeSuperBlock();
    if (!err.isOk())
        return err;

    // physically unmount
    _isMounted=false;
    err=CTCryptedBlockMedium::unmountMedium();
    purge();
    return err;
}


int CTDataBlockMedium::flush(int maxb) throw (class CTError){
    CTError err;

    // write FAT
    err=_writeFAT();
    if (!err.isOk())
        throw err;

    // write SuperBlock
    err=_writeSuperBlock();
    if (!err.isOk())
        throw err;

    return CTCryptedBlockMedium::flush(maxb);
}


void CTDataBlockMedium::purge(){
  CTCachedBlockMedium::purge();
  _blockManager.setChanged(false);
  _superBlock.setChanged(false);
}


CTError CTDataBlockMedium::readBlock(int bn, string &bl){
    if (!_isMounted)
        return CTError("CTDataBlockMedium::readBlock()",
                       k_CTERROR_INVALID,0,0,
                       "not mounted");

    return CTCryptedBlockMedium::readBlocks(
        _firstDataBlock+bn*_superBlock.blockSize()/CTFILESYSTEM_BASEBLOCKSIZE,
        _superBlock.blockSize()/CTFILESYSTEM_BASEBLOCKSIZE,
        bl,
        _superBlock.isCrypted());
}


CTError CTDataBlockMedium::writeBlock(int bn, const string &bl){
    if (!_isMounted)
        return CTError("CTDataBlockMedium::writeBlock()",
                       k_CTERROR_INVALID,0,0,
                       "not mounted");
    return CTCryptedBlockMedium::writeBlocks(
        _firstDataBlock+bn*_superBlock.blockSize()/CTFILESYSTEM_BASEBLOCKSIZE,
        _superBlock.blockSize()/CTFILESYSTEM_BASEBLOCKSIZE,
        bl,
        _superBlock.isCrypted());
}


CTError CTDataBlockMedium::allocateBlock(int &bn){
    int i;
    string tmp;
    CTError err;

    try {
      if (!_isMounted)
	return CTError("CTDataBlockMedium::allocateBlock()",
		       k_CTERROR_INVALID,0,0,
		       "not mounted");
      i=_blockManager.allocateBlock(bn);
      if (i==-1)
	return CTError("CTDataBlockMedium::allocateBlock()",
		       k_CTERROR_INVALID,0,0,
		       "Medium full.");

      // overwrite block with zeroes
      tmp=string(blockSize(),(char)0);
      err=writeBlock(i,tmp);
      if (!err.isOk()) {
	_blockManager.freeBlock(i);
	return err;
      }
      // done.
      bn=i;
      return CTError();
    } // try
    catch (CTError xerr) {
      return CTError("CTDataBlockMedium::allocateBlock",xerr);
    }
}

CTError CTDataBlockMedium::freeBlock(int bn){
    if (!_isMounted)
        return CTError("CTDataBlockMedium::freeBlock()",
                       k_CTERROR_INVALID,0,0,
                       "not mounted");
    _blockManager.freeBlock(bn);

    return CTError();
}


CTError CTDataBlockMedium::freeChain(int bn){
    if (!_isMounted)
        return CTError("CTDataBlockMedium::freeChain()",
                       k_CTERROR_INVALID,0,0,
                       "not mounted");
    _blockManager.freeChain(bn);

    return CTError();
}


CTError CTDataBlockMedium::nextBlock(int &bn){
    int i;

    if (!_isMounted)
        return CTError("CTDataBlockMedium::nextBlock()",
                       k_CTERROR_INVALID,0,0,
                       "not mounted");
    i=_blockManager.nextBlock(bn);
    if (i==-1)
        return CTError("CTDataBlockMedium::nextBlock()",
                       k_CTERROR_INVALID,1,0,
                       "end of block chain");
    // done
    bn=i;
    return CTError();
}


int CTDataBlockMedium::blockSize() const{
    return _superBlock.blockSize();
}


int CTDataBlockMedium::blockAt(int first, int idx){
    int i;

    i=_blockManager.blockAt(first,idx);
#if DEBUGMODE>15
    fprintf(stderr,"Block at %d is %d\n",
            idx,i);
#endif
    return i;
}


int CTDataBlockMedium::blocks(int bn) {
    return _blockManager.blocks(bn);
}


int CTDataBlockMedium::freeBlocks() {
    return _blockManager.freeBlocks();
}


int CTDataBlockMedium::firstDirBlock(){
    return _superBlock.firstDirBlock();
}





/* __________________________________________________________________________
 * AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                               CTDataFile
 * YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


CTDataFile::CTDataFile()
:_block(0)
,_pos(0)
,_dirty(false)
,_valid(false)
,_allocatedSize(-1)
,_firstBlock(-1)
,_medium(0)
{
#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::CTDataFile()\n");
#endif
}


CTDataFile::CTDataFile(CTPointer<CTDataBlockMedium> medium,
                       int firstBlock)
:_block(0)
,_pos(0)
,_dirty(false)
,_valid(false)
,_allocatedSize(-1)
,_firstBlock(firstBlock)
,_medium(medium)
{
#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::CTDataFile(medium,firstblock)\n");
#endif
}


CTDataFile::~CTDataFile(){
#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::~CTDataFile()\n");
#endif
}


CTError CTDataFile::_readBlock(){
    int i;
    CTError err;

#if DEBUGMODE > 1
    fprintf(stderr,"CTDataFile::_readBlock()\n");
#endif

    if (!_medium.isValid())
        return CTError("CTDataFile::_readBlock()",
                       k_CTERROR_INVALID,
                       0,
                       0,
                       "no medium");

    // get block in chain
    i=_medium.ref().blockAt(_firstBlock,_block);
    if (i==-1)
        return CTError("CTDataFile::_readBlock()",
                       k_CTERROR_INVALID,
                       0,
                       0,
                       "block not allocated");

    // read block
    err=_medium.ref().readBlock(i,_buffer);
    if (!err.isOk())
        return err;
    // setup block
    _valid=true;
    _dirty=false;

    return CTError();
}


CTError CTDataFile::_writeBlock(){
    int i;
    CTError err;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::_writeBlock()\n");
#endif

    if (!_medium.isValid())
        return CTError("CTDataFile::_writeBlock()",
                       k_CTERROR_INVALID,
                       0,
                       0,
                       "no medium");
    if (!_valid)
        return CTError("CTDataFile::_writeBlock()",
                       k_CTERROR_INVALID,
                       0,
                       0,
                       "block does not contain valid data");
    if (!_dirty)
        return CTError();

    // get block in chain
    i=_medium.ref().blockAt(_firstBlock,_block);
    if (i==-1)
        return CTError("CTDataFile::_writeBlock()",
                       k_CTERROR_INVALID,
                       0,
                       0,
                       "block not allocated");

    // check block size
#if DEBUGMODE>0
    fprintf(stderr,"Buffer length: %d, blockSize:%d\n",
            _buffer.length(),_medium.ref().blockSize());
#endif
    if ((int)(_buffer.length())!=_medium.ref().blockSize())
        return CTError("CTDataFile::_writeBlock()",
                       k_CTERROR_INVALID,
                       0,
                       0,
                       "bad buffer size (INTERNAL ERROR)");

    // write block
    err=_medium.ref().writeBlock(i,_buffer);
    if (!err.isOk())
        return err;

    // setup block
    _dirty=false;

    return CTError();

}


CTError CTDataFile::seek(int where){
    CTError err;
    int blk;
    int pos;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::seek()\n");
#endif

    /*
     if (where>=allocatedSize())
     return CTError("CTDataFile::seek()",
     k_CTERROR_INVALID,
     0,
     0,
     "seek behind end of file");
     */
    blk=where/_medium.ref().blockSize();
    pos=where%_medium.ref().blockSize();
    if (blk==_block) {
        _pos=pos;
        return CTError();
    }

    if (_valid && _dirty) {
        err=_writeBlock();
        if (!err.isOk())
            return err;
    }

    _block=blk;
    _pos=pos;
    _valid=false;
    _dirty=false;

    return CTError();
}


int CTDataFile::position() {
#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::position()\n");
#endif

    if (_block==-1)
        return -1;
    return _block*_medium.ref().blockSize()+_pos;
}


unsigned char CTDataFile::readChar(){
    CTError err;

#if DEBUGMODE>2
    fprintf(stderr,"CTDataFile::readChar()\n");
#endif

    if (_block==-1)
        throw CTError("CTDataFile::readChar()",
                      k_CTERROR_INVALID,
                      0,
                      0,
                      "no block selected");

    if (!_valid) {
        // load block
        err=_readBlock();
        if (!err.isOk())
            throw err;
    }

    if (_pos>=(int)(_buffer.length())) {
        // write block if dirty
        err=_writeBlock();
        if (!err.isOk())
            throw err;
        // advance to next block
        _block++;
        _pos=0;
        // read block
        err=_readBlock();
        if (!err.isOk())
            throw err;
    }

    // return char and advance
    return (unsigned char)_buffer[_pos++];
}


CTError CTDataFile::writeChar(unsigned char c){
    CTError err;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::writeChar()\n");
#endif

    if (!_valid) {
        // if we write behind EOF we'll have to append blocks
        while(_medium.ref().blockAt(_firstBlock,_block)==-1) {
            // append a block
            err=appendBlock();
            if (!err.isOk())
	      return CTError("CTDataFile::writeChar",err);
        } // while

        // load block
        err=_readBlock();
        if (!err.isOk())
	  return CTError("CTDataFile::writeChar",err);
    }

    if (_pos>=(int)_buffer.length())
        return CTError("CTDataFile::writeChar()",
                       k_CTERROR_INVALID,
                       0,
                       0,
                       "pos outside buffer !");

    _buffer[_pos++]=c;
    _dirty=true;
    if (_pos>=(int)(_buffer.length())) {
        // write current block
        err=_writeBlock();
        if (!err.isOk())
	  return CTError("CTDataFile::writeChar",err);
        // advance to next block
        _block++;
        _pos=0;
        _valid=false;
    }

    return CTError();
}


int CTDataFile::blocks(){
#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::blocks() for %d\n",_firstBlock);
#endif

    if (_firstBlock==-1)
        return 0;
    return _medium.ref().blocks(_firstBlock);
}


int CTDataFile::allocatedSize(){
#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::allocatedSize() for %d\n",
            _firstBlock);
#endif

    if (_allocatedSize==-1)
        _allocatedSize=blocks()*_medium.ref().blockSize();

#if DEBUGMODE>0
    fprintf(stderr,
            "Allocated size=%d (%d blocks, %d bytes per block)\n",
            _allocatedSize,
            blocks(),
            _medium.ref().blockSize());
#endif
    return _allocatedSize;
}


CTError CTDataFile::appendBlock(){
    int i;
    CTError err;
    string tmp;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::appendBlock()\n");
#endif

    i=_firstBlock;
    err=_medium.ref().allocateBlock(i);
    if (!err.isOk())
      return CTError("CTDataFile::appendBlock",err);
    if (_firstBlock==-1)
      _firstBlock=i;
    _allocatedSize=-1;

    return CTError();
}


CTError CTDataFile::truncate(){
    CTError err;
    string tmp;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::truncate()\n");
#endif

    if (_firstBlock==-1)
        // no blocks to remove
        return CTError();

    err=_medium.ref().freeChain(_firstBlock);
    if (!err.isOk())
        return err;
    _firstBlock=-1;
    _allocatedSize=-1;

    return CTError();
}


CTError CTDataFile::flush(){
#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::flush()\n");
#endif

    if (_valid && _dirty)
        return _writeBlock();

    return CTError();
}


CTError CTDataFile::writeString(const string &s){
    CTError err;
    unsigned int i;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::writeString()\n");
#endif

    for (i=0; i<s.length();i++) {
        err=writeChar(s[i]);
        if (!err.isOk())
	  return CTError("CTDataFile::writeString",err);
    } // for

    return CTError();
}


string CTDataFile::readString(int len){
    string result;

#if DEBUGMODE>15
    fprintf(stderr,"CTDataFile::readString()\n");
#endif

    while(len--)
        result+=readChar();

    return result;
}



/* __________________________________________________________________________
 * AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                               CTDirEntry
 * YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


#define CTFS_DIRENTRY_LENGTH      32
#define CTFS_DIRENTRY_PARENT_HIGH 0
#define CTFS_DIRENTRY_PARENT_LOW  1
#define CTFS_DIRENTRY_FIRST_HIGH  2
#define CTFS_DIRENTRY_FIRST_LOW   3
#define CTFS_DIRENTRY_SIZE_3      4
#define CTFS_DIRENTRY_SIZE_2      5
#define CTFS_DIRENTRY_SIZE_1      6
#define CTFS_DIRENTRY_SIZE_0      7
#define CTFS_DIRENTRY_ATTR_HIGH   8
#define CTFS_DIRENTRY_ATTR_LOW    9
#define CTFS_DIRENTRY_NAME        16


CTDirEntry::CTDirEntry()
:_changed(false)
,_attributes(0)
,_size(0)
,_firstBlock(-1)
,_parent(-1)
{
}


CTDirEntry::CTDirEntry(int parent,
                       const string &name,
                       unsigned int attribs,
                       unsigned int size,
                       int first)
:_changed(false)
,_name(name)
,_attributes(attribs)
,_size(size)
,_firstBlock(first)
,_parent(parent)
,_indexInParent(-1)
{
}


CTDirEntry::CTDirEntry(const string &s)
:_changed(false)
,_attributes(0)
,_size(0)
,_firstBlock(-1)
,_parent(0)
,_indexInParent(-1)
{
    CTError err;

    err=_fromString(s);
    if (!err.isOk())
        throw err;
}


CTDirEntry::CTDirEntry(int parent, int indexInParent)
:_changed(false)
,_attributes(0)
,_size(0)
,_firstBlock(0)
,_parent(parent)
,_indexInParent(indexInParent)
{
}


CTDirEntry::~CTDirEntry(){
}


CTError CTDirEntry::_fromString(const string &header){
    unsigned int tv;
    int i;

    // read parent
    tv=((unsigned char)header.at(CTFS_DIRENTRY_PARENT_HIGH))<<8;
    tv+=(unsigned char)header.at(CTFS_DIRENTRY_PARENT_LOW);
    if (tv==0xffff)
        _parent=-1;
    else
        _parent=tv;

    // read first block
    tv=((unsigned char)header.at(CTFS_DIRENTRY_FIRST_HIGH))<<8;
    tv+=(unsigned char)header.at(CTFS_DIRENTRY_FIRST_LOW);
    if (tv==0xffff)
        _firstBlock=-1;
    else
        _firstBlock=tv;

    // read attributes
    tv=((unsigned char)header.at(CTFS_DIRENTRY_ATTR_HIGH))<<8;
    tv+=(unsigned char)header.at(CTFS_DIRENTRY_ATTR_LOW);
    _attributes=tv;

    // read file size
    tv=((unsigned char)header.at(CTFS_DIRENTRY_SIZE_3))<<24;
    tv+=((unsigned char)header.at(CTFS_DIRENTRY_SIZE_2))<<16;
    tv+=((unsigned char)header.at(CTFS_DIRENTRY_SIZE_1))<<8;
    tv+=(unsigned char)header.at(CTFS_DIRENTRY_SIZE_0);
    _size=tv;

    // read name
    for (i=CTFS_DIRENTRY_NAME;
         i<CTFS_DIRENTRY_LENGTH;
         i++)
        if (header[i]==0)
            break;
    _name=header.substr(CTFS_DIRENTRY_NAME,
                        i-CTFS_DIRENTRY_NAME);

    return CTError();
}


string CTDirEntry::toString(){
    unsigned int tv;
    unsigned char buffer[CTFS_DIRENTRY_LENGTH];
    int i;
    string result;

    // compose header
    memset(buffer,0,sizeof(buffer));

    // parent
    if (_parent==-1)
        tv=0xffff;
    else
        tv=_parent;
    buffer[CTFS_DIRENTRY_PARENT_HIGH]=(tv>>8) & 255;
    buffer[CTFS_DIRENTRY_PARENT_LOW]=tv & 255;

    // first block
    if (_firstBlock==-1)
        tv=0xffff;
    else
        tv=_firstBlock;
    buffer[CTFS_DIRENTRY_FIRST_HIGH]=(tv>>8) & 255;
    buffer[CTFS_DIRENTRY_FIRST_LOW]=tv & 255;

    // attributes
    tv=_attributes;
    buffer[CTFS_DIRENTRY_ATTR_HIGH]=(tv>>8) & 255;
    buffer[CTFS_DIRENTRY_ATTR_LOW]=tv & 255;

    // size
    tv=_size;
    buffer[CTFS_DIRENTRY_SIZE_3]=(tv>>24) & 255;
    buffer[CTFS_DIRENTRY_SIZE_2]=(tv>>16) & 255;
    buffer[CTFS_DIRENTRY_SIZE_1]=(tv>>8) & 255;
    buffer[CTFS_DIRENTRY_SIZE_0]=tv & 255;

    // medium name
    if ((_name.length()+CTFS_DIRENTRY_NAME)>
        CTFS_DIRENTRY_LENGTH)
        throw CTError("CTDireEntry::toString()",
                      k_CTERROR_INVALID,
                      0,
                      0,
                      "Name too long");
    for (i=0; i<(int)(_name.length()); i++)
        buffer[CTFS_DIRENTRY_NAME+i]=_name[i];

    // create result
    result.assign((char*)buffer,CTFS_DIRENTRY_LENGTH);

    return result;
}


string CTDirEntry::dump() {
    string result;

    result+="Directory Entry\n";
    result+="------------------------------\n";
    result+="Name        : ";
    result+=_name+"\n";
    result+="Size        : ";
    result+=CTMisc::num2string(_size)+"\n";
    result+="First Block : ";
    result+=CTMisc::num2string(_firstBlock)+"\n";
    result+="Parent      : ";
    result+=CTMisc::num2string(_parent)+"\n";
    result+="Attributes  : ";
    if (_attributes & CTDirEntry::Attr_USED)
        result+="used ";
    if (_attributes & CTDirEntry::Attr_READ)
        result+="read ";
    if (_attributes & CTDirEntry::Attr_WRITE)
        result+="write ";
    if (_attributes & CTDirEntry::Attr_DIR)
        result+="dir ";
    if (_attributes & CTDirEntry::Attr_HIDDEN)
        result+="hidden ";
    result+="\n";

    return result;
}




CTDirectoryBase::CTDirectoryBase()
:CTDataFile()
,_currEntry(-1)
{
#if DEBUGMODE>15
    fprintf(stderr,"CTDirectoryBase::CTDirectoryBase()\n");
#endif

}


CTDirectoryBase::CTDirectoryBase(CTPointer<CTDataBlockMedium> medium,
                                 int firstBlock)
:CTDataFile(medium,firstBlock)
,_currEntry(-1)
{
#if DEBUGMODE>15
    fprintf(stderr,"CTDirectoryBase::CTDirectoryBase(medium,firstBlock)\n");
#endif

}


CTDirectoryBase::~CTDirectoryBase(){
#if DEBUGMODE>15
    fprintf(stderr,"CTDirectoryBase::~CTDirectoryBase()\n");
#endif

}


CTError CTDirectoryBase::firstEntry(CTDirEntry &de){
    CTError err;

#if DEBUGMODE>15
    fprintf(stderr,"CTDirectoryBase::firstEntry()\n");
#endif

    _currEntry=0;

    err=_readEntry(de,_currEntry);
    if (!err.isOk())
        return err;
    _currEntry++;

    return CTError();
}


CTError CTDirectoryBase::nextEntry(CTDirEntry &de){
    CTError err;

#if DEBUGMODE>15
    fprintf(stderr,"CTDirectoryBase::nextEntry()\n");
#endif

    err=_readEntry(de,_currEntry);
    if (!err.isOk())
        return err;
    _currEntry++;

    return CTError();
}


CTError CTDirectoryBase::_readEntry(CTDirEntry &de, int idx){
    string tmp;
    CTError err;

#if DEBUGMODE>15
    fprintf(stderr,"CTDirectoryBase::_readEntry()\n");
#endif

    err=seek(idx*CTFS_DIRENTRY_LENGTH);
    if (!err.isOk())
        return err;
    try {
        tmp=readString(CTFS_DIRENTRY_LENGTH);
        de=CTDirEntry(tmp);
        de.setIndexInParent(idx);
    }
    catch (CTError lerr) {
        return lerr;
    }

    return CTError();
}


int CTDirectoryBase::_findFreeEntry(){
    CTDirEntry de;
    int i;
    int j;
    CTError err;

#if DEBUGMODE>15
    fprintf(stderr,"CTDirectoryBase::_findFreeEntry()\n");
#endif

    if (allocatedSize()<CTFS_DIRENTRY_LENGTH) {
#if DEBUGMODE>0
        fprintf(stderr,
                "allocated size too small (%d bytes)\n",
                allocatedSize());
#endif
        return -1;
    }
    i=allocatedSize()/CTFS_DIRENTRY_LENGTH;
    for (j=0; j<i; j++) {
        err=_readEntry(de,j);
        if (!err.isOk())
            throw err;
        if (!(de.attributes()&CTDirEntry::Attr_USED)) {
#if DEBUGMODE>0
            fprintf(stderr,"found free entry (%d).\n",j);
#endif
            return j;
        }
    } // for

#if DEBUGMODE>0
    fprintf(stderr,"no free entry found.\n");
#endif
    return -1;
}


int CTDirectoryBase::_findOrAddFreeEntry(){
    int i;
    CTError err;

#if DEBUGMODE>15
    fprintf(stderr,"CTDirectoryBase::_findOrAddFreeEntry()\n");
#endif

    i=_findFreeEntry();
    if (i!=-1)
        return i;

#if DEBUGMODE>0
    fprintf(stderr,"Did not find a free entry, will append a block.\n");
#endif

    // try to append a block
    err=appendBlock();
    if (!err.isOk())
      throw CTError("CTDirectoryBase::_findOrAddFreeEntry",err);

    // try again and fail then, if there is an error again
    return _findFreeEntry();
}


CTError CTDirectoryBase::writeEntry(CTDirEntry &de){
    CTError err;
    int entry;
    string tmp;

#if DEBUGMODE>15
    fprintf(stderr,"CTDirectoryBase::writeEntry()\n");
#endif

    // root ?
    if (de.parent()==-1)
        return CTError("CTDirectoryBase::writeEntry()",
                       k_CTERROR_INVALID,0,0,
                       "cannot write dir entry for root");
    if (de.indexInParent()==-1) {
#if DEBUGMODE>0
        fprintf(stderr,"Will add entry.\n");
#endif
        // not assigned, so we have to look for an entry
        try {
            entry=_findOrAddFreeEntry();
        }
        catch(CTError lerr) {
	  return CTError("CTDirectoryBase::writeEntry()",lerr);
	}
	if (entry==-1)
	  return CTError("CTDirectoryBase::writeEntry()",
			 k_CTERROR_INVALID,0,0,
			 "Directory full");

	// store index
        de.setIndexInParent(entry);
    }

#if DEBUGMODE>0
    fprintf(stderr,"Will write entry %d (for %d).\n",
            de.indexInParent(), firstBlock());
#endif
    // write it directly
    err=seek(de.indexInParent()*CTFS_DIRENTRY_LENGTH);
    if (!err.isOk())
        return err;
    try {
        // create string
        tmp=de.toString();
        // write string to medium
        err=writeString(tmp);
        if (!err.isOk())
            return err;
    }
    catch (CTError lerr) {
        return lerr;
    }
#if DEBUGMODE>0
    fprintf(stderr,"Entry %d written.\n",de.indexInParent());
#endif
    return CTError();
}


CTDirEntry CTDirectoryBase::findEntry(const string &name) {
    CTDirEntry ent;
    CTError err;

    err=firstEntry(ent);
    while(err.isOk()) {
        if (ent.attributes()&CTDirEntry::Attr_USED &&
            ent.name()==name)
            return ent;
        err=nextEntry(ent);
    } // while
    throw err;
}







CTFileBase::CTFileBase()
:_isOpen(false)
{
}


CTFileBase::CTFileBase(CTPointer<CTDataBlockMedium> medium,
             const string &path)
:_medium(medium)
,_path(_normalizeName(path))
,_isOpen(false)
{
}


CTFileBase::~CTFileBase(){
}


string CTFileBase::_normalizeName(string n) {
    unsigned int i;
    string n2;
    bool haveit;

    // remove multiple "/"
    haveit=false;
    for (i=0; i<n.length(); i++) {
        if (n.at(i)=='/') {
            if (!haveit) {
                haveit=true;
                n2+=n.at(i);
            }
        }
        else {
            haveit=false;
            n2+=n.at(i);
        }
    } // for
    n=n2;

    // prepare name
    if (!n.empty())
        if (n.at(0)=='/')
            n.erase(0,1);
    if (!n.empty())
        if (n.at(n.length()-1)=='/')
            n.erase(n.length()-1);

    // thats it
    return n;
}


CTDirEntry CTFileBase::path2Entry(const string &path){
    string part;
    unsigned int pos1, pos2;
    CTDirectoryBase dir;
    CTDirEntry entry;
    int first;
    bool fileEncountered;

    // start with root
    pos1=0;
    fileEncountered=false;
    first=_medium.ref().firstDirBlock();
    // create pseudo entry for root
    entry=CTDirEntry(-1,
                     "",
                     CTDirEntry::Attr_USED |
                     CTDirEntry::Attr_READ |
                     CTDirEntry::Attr_WRITE |
                     CTDirEntry::Attr_DIR,
                     0,
		     0);
    while(pos1<path.length()) {
        // check for first
        if (first==-1)
            throw CTError("CTFileBase::path2Entry()",
                          k_CTERROR_INVALID,0,0,
                          "empty directory in path");

        // cut out current path member
        pos2=path.find("/",pos1);
        if (pos2==string::npos) {
            part=path.substr(pos1);
            pos1=path.length();
        }
        else {
            part=path.substr(pos1,pos2-pos1);
            pos1=pos2+1;
        }

#if DEBUGMODE>0
        fprintf(stderr,"Looking for \"%s\"\n",
                part.c_str());
#endif

        // try to find that path member
        dir=CTDirectoryBase(_medium,first);
        entry=dir.findEntry(part);
#if DEBUGMODE>0
        fprintf(stderr,"Found \"%s\"\n%s\n",
                part.c_str(),entry.dump().c_str());
#endif
        if (!(entry.attributes()&CTDirEntry::Attr_DIR)) {
            if (fileEncountered)
                throw CTError("CTFileBase::path2Entry()",
                              k_CTERROR_INVALID,0,0,
                              "Entry is not a directory");
            else
                fileEncountered=true;
        }

        if (pos1<path.length())
            // dive
            first=entry.firstBlock();
    } // while

    return entry;
}


CTError CTFileBase::openFile(){
    CTError err;

    if (_isOpen)
        return CTError("CTFileBase::openFile()",
                       k_CTERROR_INVALID,0,0,
                       "already open");
    // resolve path
    try {
        _entry=path2Entry(_path);
    }
    catch(CTError lerr) {
        return lerr;
    }

    // create stream for the data of this entry
    _data=CTDataFile(_medium,_entry.firstBlock());

    _isOpen=true;

    return CTError();
}


CTError CTFileBase::closeFile(){
    CTError err;
    CTDirectoryBase dir;

    if (!_isOpen)
        return CTError("CTFileBase::closeFile()",
                       k_CTERROR_INVALID,0,0,
                       "not open");

#if DEBUGMODE>0
    fprintf(stderr,"\n==============================\n");
    fprintf(stderr,"Parent first is: %d\n",_entry.parent());
    fprintf(stderr,"Entry.FirstBlock is: %d\n",_entry.firstBlock());
    fprintf(stderr,"Data.FirstBlock is: %d\n",_data.firstBlock());
    fprintf(stderr,"==============================\n\n");
#endif

    // set new first data block (if changed)
    if (_entry.parent()!=-1)
        if (_entry.firstBlock()==-1 && _data.firstBlock()!=-1)
            _entry.setFirstBlock(_data.firstBlock());

    // flush data
    err=_data.flush();
    if (!err.isOk())
        return err;

    // save entry if changed (and if not root)
    if (!_entry.parent()!=-1 && _entry.changed()) {
#if DEBUGMODE>0
        fprintf(stderr,"\n==============================\n");
        fprintf(stderr,"Will now save this entry:\n%s\n",
                _entry.dump().c_str());
        fprintf(stderr,"==============================\n\n");
#endif
        dir=CTDirectoryBase(_medium,_entry.parent());
        err=dir.writeEntry(_entry);
        if (!err.isOk())
            return err;
        err=dir.flush();
        if (!err.isOk())
            return err;
    }

    // remove datafile
    _data=CTDataFile();

    _isOpen=false;
    // flush medium
    //return _medium.ref().flush();
    return CTError();
}


CTError CTFileBase::_createEntry(const string &n,
                                 unsigned int attribs,
                                 CTDirEntry &fileEntry){
    string path;
    string file;
    unsigned int pos;
    CTDirEntry parentEntry;
    CTDirEntry tmpEntry;;
    CTError err;

    // check whether the file already exists
    err=CTError();
    try {
        tmpEntry=path2Entry(n);
    }
    catch(CTError lerr) {
        err=lerr;
    }
    if (err.isOk())
        return CTError("CTFileBase::_createEntry()",
                       k_CTERROR_INVALID,0,0,
                       "file already exists");

    // split path into path and file
    pos=n.rfind("/");
    if (pos==string::npos) {
        path="";
        file=n;
    }
    else {
      path=n.substr(0,pos);
      if (pos+1<_path.length())
	file=n.substr(pos+1);
      else
	file="";
    }

    if (file.empty())
        return CTError("CTFileBase::_createEntry()",
                       k_CTERROR_INVALID,0,0,
                       "empty name");

#if DEBUGMODE>0
    fprintf(stderr,"PATH is: \"%s\", FILE is \"%s\"\n",
            path.c_str(),file.c_str());
#endif
    // get entry for path
    try {
        parentEntry=path2Entry(path);
    }
    catch(CTError lerr) {
        return lerr;
    }

#if DEBUGMODE>0
    fprintf(stderr,"Will now create file.\n");
#endif
    if (parentEntry.firstBlock()==-1) {
#if DEBUGMODE>0
        fprintf(stderr,"Need to append a block.\n");
#endif
        CTDataFile f(_medium,-1);

        err=f.appendBlock();
        if (!err.isOk())
	  return CTError("CTFileBase::_createEntry",err);
        parentEntry.setFirstBlock(f.firstBlock());
        CTDirectoryBase d(_medium, parentEntry.parent());
        err=d.writeEntry(parentEntry);
        if (!err.isOk())
            return err;
        err=d.flush();
        if (!err.isOk())
            return err;
    }

    // write entry
    fileEntry=CTDirEntry(parentEntry.firstBlock(),
                         file,    // name
                         attribs, // attributes
                         0,       // size
                         -1);     // first block
#if DEBUGMODE>0
    fprintf(stderr,"File Entry is now:\n%s\n",
            fileEntry.dump().c_str());
#endif

    err=writeEntry(fileEntry);
    if (!err.isOk())
        return err;

    return CTError();
}


CTError CTFileBase::createFile(unsigned int attribs){
    string path;
    string file;
    CTDirEntry fileEntry;
    CTDirectoryBase dir;
    CTError err;

    try {
      // check if already open
      if (_isOpen)
	return CTError("CTFileBase::createFile()",
		       k_CTERROR_INVALID,0,0,
		       "already open");
      err=_createEntry(_path,attribs,fileEntry);
      if (!err.isOk())
	return err;

      // use this entry
      _entry=fileEntry;

      // create stream for the data of this entry
      _data=CTDataFile(_medium,_entry.firstBlock());

      _isOpen=true;

      return CTError();
    }
    catch (CTError xerr) {
      return CTError("CTFileBase::createFile",xerr);
    }
}


CTError CTFileBase::writeEntry(CTDirEntry &entry){
    CTDirectoryBase dir;
    CTError err;

    try {
      if (entry.parent()==-1)
	return CTError("CTFileBase::writeEntry()",
		       k_CTERROR_INVALID,0,0,
		       "cannot write root entry");

      dir=CTDirectoryBase(_medium,entry.parent());
      err=dir.writeEntry(entry);
      if (!err.isOk())
	return err;
      return dir.flush();
    }
    catch (CTError xerr) {
      return CTError("CTFileBase::writeEntry",xerr);
    }
}


CTError CTFileBase::renameFile(const string &n) {
    CTDirEntry oldEntry;
    CTDirEntry newEntry;
    CTError err;
    string newName;

    try {
      newName=_normalizeName(n);

      // get old entry
      if (_isOpen)
	oldEntry=_entry;
      else {
	try {
	  oldEntry=path2Entry(_path);
	}
	catch(CTError lerr) {
	  return lerr;
	}
      }

      // create new entry
      err=_createEntry(newName,oldEntry.attributes(),newEntry);
      if (!err.isOk())
	return err;

      /* copy relevant data to new entry
       * attributes and name are already set */
      newEntry.setSize(oldEntry.size());
      newEntry.setFirstBlock(oldEntry.firstBlock());

      // invalidate old entry (just in case ;-)
      oldEntry.setSize(0);
      oldEntry.setFirstBlock(-1);

      // make it unused
      oldEntry.setAttributes(0);

      // write new entry
      err=writeEntry(newEntry);
      if (!err.isOk())
	return err;

      // then write old entry
      err=writeEntry(oldEntry);
      if (!err.isOk())
	return err;

      // use the new entry
      _entry=newEntry;

      return CTError();
    }
    catch (CTError xerr) {
      return CTError("CTFileBase::renameFile",xerr);
    }

}


CTError CTFileBase::removeFile(){
    CTError err;
    CTDirEntry entry;

    if (_isOpen)
        return CTError("CTFileBase::removeFile()",
                       k_CTERROR_INVALID,0,0,
                       "file is open");
    // resolve path
    try {
        entry=path2Entry(_path);
    }
    catch(CTError lerr) {
        return lerr;
    }

    if (entry.firstBlock()!=-1) {
        // free all blocks
#if DEBUGMODE>0
        fprintf(stderr,"\nWill now free chain.\n\n");
#endif
        err=_medium.ref().freeChain(entry.firstBlock());
        if (!err.isOk())
            return err;
        entry.setFirstBlock(0);
    }

    // make entry unused
    entry.setAttributes(0);

    // write entry to medium
#if DEBUGMODE>0
    fprintf(stderr,"\nWill now write entry.\n\n");
#endif
    return writeEntry(entry);
}


CTError CTFileBase::statFile(CTDirEntry &ent){
    CTError err;

    if (!_isOpen) {
        try {
            ent=path2Entry(_path);
        }
        catch (CTError lerr) {
            return lerr;
        }
    }
    else
        ent=_entry;
    return CTError();
}


CTError CTFileBase::seek(int where){
  try {
    return _data.seek(where);
  }
  catch (CTError xerr) {
    return CTError("CTFileBase::seek",xerr);
  }
}


int CTFileBase::position(){
  return _data.position();
}


int CTFileBase::size() {
    if (_entry.attributes() & CTDirEntry::Attr_DIR)
        return _data.allocatedSize();
    return _entry.size();
}


unsigned char CTFileBase::readChar(){
    unsigned char c;

#if DEBUGMODE>15
    fprintf(stderr,"\nCTFileBase::readChar()\n");
#endif
    /*
    if (!(_entry.attributes() & CTDirEntry::Attr_DIR))
        if (position()>_entry.size())
            throw CTError("CTFileBase::readChar()",
                          k_CTERROR_INVALID,0,0,
                          "EOF met");
                          */
    c=_data.readChar();
#if DEBUGMODE>15
    fprintf(stderr,"CTFileBase::readChar() done\n\n");
#endif
    return c;
}


string CTFileBase::readString(int len){
    string result;

#if DEBUGMODE>15
    fprintf(stderr,"CTFileBase::readString()\n");
#endif

    while(len--)
        result+=readChar();

    return result;
}


CTError CTFileBase::writeChar(unsigned char c){
    CTError err;

    err=_data.writeChar(c);
    if (!err.isOk())
      return CTError("CTFileBase::writeChar",err);

    // adjust size
    if (position()>_entry.size())
        _entry.setSize(position());
    // set new first block if there was none
    if (_entry.firstBlock()==-1 && _data.firstBlock()!=-1) {
	_entry.setFirstBlock(_data.firstBlock());
        // first block just assigned, so save it to not loose blocks
        err=writeEntry(_entry);
	if (!err.isOk())
	  return CTError("CTFileBase::writeChar",err);
    }

    return CTError();
}


CTError CTFileBase::writeString(const string &s){
    unsigned int i;
    CTError err;

    for (i=0; i<s.length(); i++) {
        err=writeChar(s[i]);
        if (!err.isOk())
	  return CTError("CTFileBase::writeString",err);
    } // for

    return CTError();
}


CTError CTFileBase::flush(){
    return _data.flush();
}


CTError CTFileBase::truncate(){
    // check if already open
    if (_isOpen)
        return CTError("CTFileBase::createFile()",
                       k_CTERROR_INVALID,0,0,
                       "already open");
    return _data.truncate();
}








CTFile::CTFile()
:CTFileBase()
{
}


CTFile::CTFile(CTPointer<CTCardFS> medium,
               const string &path)
:CTFileBase((CTPointer<CTDataBlockMedium>&)medium, path)
{
}


CTFile::~CTFile(){
}


string CTFile::readString(int len){
    int i;

    i=size()-position();
    if (len>i)
        len=i;
    if (!len)
        return "";
    return CTFileBase::readString(len);
}









CTDirectory::CTDirectory()
:CTFileBase()
{
}


CTDirectory::CTDirectory(CTPointer<CTCardFS> medium,
                         const string &path)
:CTFileBase((CTPointer<CTDataBlockMedium>&)medium,path)
{
}


CTDirectory::~CTDirectory(){
}


CTError CTDirectory::openDirectory(){
    CTError err;

    try {
        err=CTFileBase::openFile();
        if (!err.isOk())
            return err;
        if (!(dirEntry().attributes() & CTDirEntry::Attr_DIR))
            return CTError("CTFileBase::openDirectory()",
                           k_CTERROR_INVALID,0,0,
                           "not a directory");
        return CTError();
    }
    catch (CTError xerr) {
        return xerr;
    }
}


CTError CTDirectory::firstEntry(CTDirEntry &de){
    CTError err;

    try {
        err=seek(0);
        if (!err.isOk())
            return err;

        return nextEntry(de);
    }
    catch (CTError xerr) {
        return xerr;
    }
}


CTError CTDirectory::nextEntry(CTDirEntry &de){
    string tmp;
    CTError err;

    try {
        tmp=readString(CTFS_DIRENTRY_LENGTH);
        de=CTDirEntry(tmp);
    }
    catch(CTError lerr) {
        return lerr;
    }

    return CTError();
}


CTError CTDirectory::entry(CTDirEntry &de, int idx){
    CTError err;

    try {
        err=seek(idx*CTFS_DIRENTRY_LENGTH);
        if (!err.isOk())
            return err;

        return nextEntry(de);
    }
    catch (CTError xerr) {
        return xerr;
    }
}


CTError CTDirectory::removeDirectory(){
    CTError err;
    int i;
    string tmp;
    CTDirEntry de;

    if (isOpen())
        return CTError("CTDirectory::removeDirectory()",
                       k_CTERROR_INVALID,0,0,
                       "directory is open");
    try {
        // check if directory is empty
        err=openDirectory();
        if (!err.isOk())
            return err;

        i=size()/CTFS_DIRENTRY_LENGTH;
        while(i--) {
            try {
                tmp=readString(CTFS_DIRENTRY_LENGTH);
                de=CTDirEntry(tmp);
            }
            catch(CTError lerr) {
                return lerr;
            }
            if (de.attributes() & CTDirEntry::Attr_USED) {
                closeDirectory();
                return CTError("CTDirectory::removeDirectory()",
                               k_CTERROR_INVALID,0,0,
                               "directory not empty");
            }
        } // while
        closeDirectory();

        return CTFileBase::removeFile();
    }
    catch (CTError xerr) {
        return xerr;
    }
}


CTCardFS::CTCardFS(const CTCard &c)
:CTDataBlockMedium(c)
{
}


CTCardFS::~CTCardFS(){
}


CTError CTCardFS::mountMedium(const string &username,
                              const string &passwd){
    return CTDataBlockMedium::mountMedium(username,passwd);
}


CTError CTCardFS::createMedium(unsigned int mediumSize,
                               const string &mediumName,
                               const string &userName,
			       const string &passwd) {
  try {
    return CTDataBlockMedium::createMedium(mediumSize, mediumName,
					   userName, passwd);
  }
  catch(CTError xerr) {
    return CTError("CTDataBlockMedium::createMedium",xerr);
  }
}



