/* 
 * $Id: tinfo.c,v 1.11 2000/11/29 01:53:37 doviende Exp $
 *
 * libarr - a screen management toolkit
 *
 * Copyright (C) 2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Author: Chris Bond
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 * XXX It is possible that there are endianness issues with the code in this
 * XXX file particularly.  Perhaps a check should be performed at some point.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include "tinfo.h"
#include "types.h"
#include "printf.h"
#include "arr.h"
#include "cap_offset.h"
#include "tt_sprintf.h"

/* tinfo.c is the terminfo database interface.
 */

/* tc_entry is the terminfo file that is currently open.  It's used throughout
 * the code when retrieving data.
 */
struct tinfo_entry *tc_entry;

/* mbyte_pairs[] is a list of terminals that support multibyte character pairs.
 */
struct mbyte_pair mbyte_pairs[] = {
	{ "kon",		0xA1,	0xFE	},	/* Kanji console */
	{ (const char *)NULL,	0,	0	}
};
struct mbyte_pair *term_mbyte_pairs;

/* According to terminfo, if two consecutive bytes are read that contain 0376
 * and 0377 (octal) respectively, it is equal to -2 (decimal).  Conversely,
 * if we encounter two consecutive 0377's, it's equal to -1.
 */
#define TINFO_N2(buf) (((u_int8_t)((buf)[0]) == 0376) && \
		       ((u_int8_t)((buf)[1]) == 0377))
#define TINFO_N1(buf) (((u_int8_t)((buf)[0]) == 0377) && \
		       ((u_int8_t)((buf)[1]) == 0377))

/* TINFO_WORD() converts two int8_t's (8 bits, I hope) into a 16-bit int.
 */
#define TINFO_WORD(buf) (((u_int8_t)(buf)[0]) + (256 * ((u_int8_t)(buf)[1])))

/* tinfo_path[] is a list of paths for the terminfo directory to exist in.
 */
static char *tinfo_path[] = {
	"/usr/share/terminfo/",
	"/usr/local/share/terminfo/",
	"/usr/lib/terminfo/",
	"/usr/local/lib/terminfo/",
	"/usr/share/lib/terminfo/",
	"/usr/local/share/lib/terminfo/",
	"/etc/terminfo/",
	"/usr/local/etc/terminfo/",
	NULL
};
	
/* kill_tc_entry() frees tc_entry and all of its members, and makes sure to
 * nullify it so we know that it doesn't exist later.
 */
void
kill_tc_entry(void)
{
	if (tc_entry->fd >= 0)
		close(tc_entry->fd);

	free(tc_entry->names);
	free(tc_entry->caps.bool);
	free(tc_entry->caps.nums);
	free(tc_entry->caps.str_offsets);
	free(tc_entry->caps.str_table);
	free(tc_entry);

	tc_entry = (struct tinfo_entry *)NULL;
}

/* read_alloc() allocates enough space for a buffer and reads a chunk of data
 * into it from a file.
 */
static size_t
read_alloc(fd, buf, size)
	int fd;
	size_t size;
	char **buf;
{
	size_t read_size;

	if (((*buf = (char *)calloc(size, sizeof(char))) == NULL) ||
	    ((read_size = read(fd, *buf, size)) < size)) {
	    	if (*buf)
	    		free(*buf);
		*buf = NULL;
		return -1;
	}

	return read_size;
}

/* read_alloc_convert() is almost the same as read_alloc(), but it takes every
 * two bytes and forms a TINFO_WORD.
 */
static size_t
read_alloc_convert(fd, buf, count)
	int fd;
	size_t count;
	int16_t **buf;
{
	int8_t *buf_ptr, *pointer;
	int n_count;

	if ((*buf = (int16_t *)calloc(count, sizeof(int16_t))) == NULL)
		return -1;

	if (read_alloc(fd, (char **)&buf_ptr, (count * sizeof(int16_t))) <
					      (count * sizeof(int16_t)))
		return -1;

	/* Use `pointer' to advance through the buffer, two bytes at a time.
	 */
	pointer = buf_ptr;
	for (n_count = 0; n_count < count; ++n_count) {
		if (TINFO_N1(pointer) || TINFO_N2(pointer))
			(*buf)[n_count] = -1;
		else
			(*buf)[n_count] = TINFO_WORD(pointer);
		pointer += sizeof(int16_t);
	}

	/* buf_ptr was just a scratch buffer -- we can free it.
	 */
	free(buf_ptr);

	return n_count;
}

/* set_mbyte_pairs() checks to see if $TERM supports multibyte character pairs.
 */
static void
set_mbyte_pairs(term)
	char *term;
{
	struct mbyte_pair *mb_ptr;

	for (mb_ptr = mbyte_pairs; mb_ptr->term != NULL; ++mb_ptr) {
		if (strcmp(mb_ptr->term, term) == 0) {
			term_mbyte_pairs = mb_ptr;
			return;
		}
	}

	term_mbyte_pairs = (struct mbyte_pair *)NULL;
}

/* open_tinfo() opens a terminfo entry after finding it in one of the
 * directories in tinfo_path[].
 */
static int
open_tinfo(term)
	char *term;
{
	char **dir, *buf;
	int fd;

	for (dir = tinfo_path; *dir != NULL; ++dir) {
		/* Format `buf' into a terminfo-compatible path name for an
		 * entry.  That is, `linux' will become `<prefix>/l/linux'.
		 */
		if (malloc_printf(&buf, "%s%c/%s", *dir, *term, term) == 0) {
			perror("Unable to allocate memory chunk");
			return -1;
		}

		fd = open(buf, O_RDONLY);
		free(buf);

		if (fd >= 0)
			return fd;
	}

	/* Fuck.  No terminfo file.  We can soldier on, but we won't have any
	 * of the features that we want (e.g. cursor movement)...
	 */
	return -1;
}

/* tinfo_init() opens the terminfo entry and reads the contents of it.  If we
 * discover an inconsistency in the terminfo file at any point, we return -1.
 * Our caller is responsible for calling kill_tc_entry() to free everything.
 */
int
tinfo_init(term)
	char *term;
{
	char read_buf[12];

	tc_entry = (struct tinfo_entry *)calloc(1, sizeof(struct tinfo_entry));
	if (tc_entry == NULL) {
		perror("Unable to allocate memory chunk");
		return -1;
	}

	if ((tc_entry->fd = open_tinfo(term)) < 0)
		return -1;

	set_mbyte_pairs(term);

	/* Get the various statistics and flags of the entire file, and store
	 * them in ->sizes.  Also, terminfo requires that the file magic be
	 * 0432 (octal).
	 */
	if (read(tc_entry->fd, read_buf, sizeof(read_buf)) < sizeof(read_buf))
		return -1;

	/* Go through each two bytes converting them into one single 16-bit
	 * integer, and store them in appropriate places in the tc_entry struct.
	 */
	if ((tc_entry->hdr.magic = TINFO_WORD(read_buf)) != 0432)
		return -1;

	/* Allocate enough room for the list of terminal entries and read the
	 * list into our buffer.
	 */
	tc_entry->hdr.name_len = TINFO_WORD(&read_buf[2]);
	if (read_alloc(tc_entry->fd, &tc_entry->names, tc_entry->hdr.name_len)
			< 0)
		return -1;

	/* Grab all of the booleans out of the entry:
	 */
	tc_entry->hdr.bool_size = TINFO_WORD(&read_buf[4]);
	if (read_alloc(tc_entry->fd, &tc_entry->caps.bool,
		       tc_entry->hdr.bool_size) < tc_entry->hdr.bool_size)
		return -1;

	/* Make sure we're at an even offset in the file:
	 */
	if ((tc_entry->hdr.name_len + tc_entry->hdr.bool_size) & 1) {
		char dummy;

		if (read(tc_entry->fd, &dummy, sizeof(char)) < 0)
			return -1;
	}

	/* Read the numbers section (converting with TINFO_WORD()):
	 */
	tc_entry->hdr.num_size = TINFO_WORD(&read_buf[6]);
	if (read_alloc_convert(tc_entry->fd, &tc_entry->caps.nums,
			       tc_entry->hdr.num_size) < tc_entry->hdr.num_size)
		return -1;

	/* Finally, read in the string offsets, followed by the actual table:
	 */
	tc_entry->hdr.str_size = TINFO_WORD(&read_buf[8]);
	if (read_alloc_convert(tc_entry->fd, &tc_entry->caps.str_offsets,
			       tc_entry->hdr.str_size) < tc_entry->hdr.str_size)
		return -1;

	tc_entry->hdr.str_table_size = TINFO_WORD(&read_buf[10]);
	if (read_alloc(tc_entry->fd, &tc_entry->caps.str_table,
		  tc_entry->hdr.str_table_size) < tc_entry->hdr.str_table_size)
		return -1;

	/* Ok, at this point, we have the string offsets, and the string table.
	 * Now we can simply combine the two and have terminfo strings!
	 */

	close(tc_entry->fd);
	return 0;
}

char *
tinfo_getstr(en, offset)
	struct tinfo_entry *en;
	int offset;
{
	if ((en == NULL) || (en->hdr.str_size <= offset))
		return "";

	if (en->caps.str_offsets[offset] < 0)
		return "";

	return (en->caps.str_table + (en->caps.str_offsets[offset]));
}

int
tinfo_getflag(en, offset)
	struct tinfo_entry *en;
	int offset;
{
	if ((en == NULL) || (en->hdr.bool_size <= offset))
		return 0;

	return en->caps.bool[offset];
}

int
tinfo_getnum(en, offset)
	struct tinfo_entry *en;
	int offset;
{
	if ((en == NULL) || (en->hdr.num_size <= offset))
		return -1;

	return en->caps.nums[offset];
}
