# This module implements exporting to a web gallery.

import gtk
import gobject
import gc
import os
import sys

import Alert
import ThumbnailArea


def protect(str):
    str = u"&amp;".join(str.split(u"&"))
    str = u"&lt;".join(str.split(u"<"))
    str = u"&gt;".join(str.split(u">"))
    return str


class Template:

    def __init__(self, template_dir):
    	self.dir = template_dir
	self.templates = {}

    def get(self, basename):
    	if not self.templates.has_key(basename):
	    f = open(os.path.join(self.dir, basename + ".template"), "r")
	    self.templates[basename] = f.read()
	    f.close()
	return self.templates[basename]

    def other_filenames(self):
    	basenames = os.listdir(self.dir)
	basenames = filter(lambda n: n[-len(".template"):] != ".template",
			   basenames)
	filenames = map(lambda n: os.path.join(self.dir, n), basenames)
	filenames = filter(lambda n: not os.path.isdir(n), filenames)
	return filenames

    def write(self, target_filename, basename, dict):
    	f = open(target_filename, "w")
	f.write(self.get(basename) % dict)
	f.close()


class ProgressEstimate:

    def __init__(self):
    	self.total_work = 1
	self.work_done = 0

    def set_total(self, total_work):
    	self.total_work = total_work

    def add_done(self, amount):
    	self.work_done += amount

    def fraction(self):
    	return float(self.work_done) / self.total_work


class PhotoLoader:

    READ_AMOUNT = 4 * 1024

    def __init__(self, file, progress_estimate):
    	self.input = file
	self.loader = gtk.gdk.PixbufLoader("jpeg")
	self.progress_estimate = progress_estimate

    def work(self):
    	data = self.input.read(self.READ_AMOUNT)
	if data:
	    self.loader.write(data)
	    self.progress_estimate.add_done(len(data))
	    return gtk.TRUE
	else:
	    self.input.close()
	    self.loader.close()
	    self.pixbuf = self.loader.get_pixbuf()
	    return gtk.FALSE


class TemplateDict:

    def __init__(self, prefix=u"", metadata=None):
    	self.dict = {}
	if metadata:
	    for key in metadata.keys():
	    	self.dict[prefix + key] = protect(metadata[key])

    def normalize(self, str):
    	if type(str) != type(u""):
	    return unicode(str)
	else:
	    return str

    def __getitem__(self, key):
    	key = self.normalize(key)
    	if self.dict.has_key(key):
	    return unicode(self.dict[key])
    	else:
	    return u""

    def __setitem__(self, key, value):
    	key = self.normalize(key)
    	value = self.normalize(value)
	self.dict[key] = value

    def keys(self):
    	return self.dict.keys()


class ExportWeb:

    MAX_WIDTH = 500
    MAX_HEIGHT = 500
    
    THUMB_MAX = 128

    def __init__(self, view, dirname, doc, folders, template):
    	self.view = view
    	self.dirname = dirname
    	self.doc = doc
    	self.folders = folders

	self.photos = []
	self.photo_to_folder = {}
	for folder in self.folders:
	    for photo in folder.photos.get():
	    	self.photos.append(photo)
		self.photo_to_folder[photo[u"id"]] = folder

	self.progress_estimate = ProgressEstimate()
	self.template = Template(template)
	self.workers = []
	total_work = 0

	self.workers.append(self.create_target_dir)

	self.other_files = self.template.other_filenames()
	self.workers.append(self.copy_other_files)
	total_work += self.file_sizes(self.other_files)

    	self.photo_loader = None
	self.unloaded_index = 0
	self.workers.append(self.make_photo_files)
	total_work += self.photo_sizes(self.photos)
	
	self.workers.append(self.make_folder_pages)
	
	self.progress_estimate.set_total(total_work)
	
	gtk.idle_add(self.work)
	self.view.set_status_text("Exporting to web gallery...")

    def file_sizes(self, filenames):
    	sizes = 0
	for filename in filenames:
	    sizes += os.stat(filename).st_size
	return sizes

    def photo_sizes(self, photos):
    	filenames = \
	    map(lambda p: self.doc.storage.get_original_filename(p[u"id"]),
	    	photos)
    	return self.file_sizes(filenames)

    def work(self):
    	if self.workers:
	    try:
		ret = self.workers[0]()
		if ret == gtk.FALSE:
		    self.workers = self.workers[1:]
    	    	self.view.set_progress(self.progress_estimate.fraction())
		return gtk.TRUE
	    except IOError, (code, txt):
		Alert.Alert(gtk.STOCK_DIALOG_ERROR,
			    [(gtk.STOCK_OK,)],
			    u"Something went wrong.",
			    u"Error %d: %s" % (code, txt))
	    except OSError, (code, txt):
		Alert.Alert(gtk.STOCK_DIALOG_ERROR,
			    [(gtk.STOCK_OK,)],
			    u"Something went wrong.",
			    u"Error %d: %s" % (code, txt))
	    except gobject.GError, detail:
		Alert.Alert(gtk.STOCK_DIALOG_ERROR,
			    [(gtk.STOCK_OK,)],
			    u"Something went wrong.",
			    u"%s" % unicode(str(detail)))

	self.view.set_progress(0)
	self.view.set_status_text("Exported folder to %s" % self.dirname)
    	return gtk.FALSE

    def create_target_dir(self):
    	if not os.path.isdir(self.dirname):
	    os.makedirs(self.dirname)
	return gtk.FALSE

    def copy_other_files(self):
    	if not self.other_files:
	    return gtk.FALSE

    	filename = self.other_files[0]
	self.other_files = self.other_files[1:]
	
	f = open(filename, "r")
	data = f.read()
	f.close()
	
	f = open(os.path.join(self.dirname, os.path.basename(filename)), "w")
	f.write(data)
	f.close()
	
	self.progress_estimate.add_done(len(data))
	
    	return gtk.TRUE

    def make_photo_files(self):
    	if self.unloaded_index >= len(self.photos):
	    return gtk.FALSE

	if not self.photo_loader:
	    id = self.photos[self.unloaded_index][u"id"]
	    self.photo_loader = PhotoLoader(self.doc.storage.get_original(id),
	    	    	    	    	    self.progress_estimate)

    	if self.photo_loader.work() == gtk.TRUE:
	    return gtk.TRUE
	
    	photo = self.photos[self.unloaded_index]
    	pixbuf = self.photo_loader.pixbuf
	pixbuf = ThumbnailArea.rotate_pixbuf(pixbuf, photo.get_angle())

    	w, h = self.write_photo_file(photo, pixbuf, 
	    	    	    	     self.MAX_WIDTH, self.MAX_HEIGHT,
				     u"")
    	tw, th = self.write_photo_file(photo, pixbuf, 
	    	    	    	       self.THUMB_MAX, 
				       self.THUMB_MAX,
				       u"-thumb")

    	self.write_photo_html(self.unloaded_index, w, h, tw, th)

	self.photo_loader = None
	self.unloaded_index += 1
	gc.collect()
	return gtk.TRUE

    def write_photo_file(self, photo, pixbuf, max_width, max_height, suffix):
	w, h = self.scale(pixbuf.get_width(),
			  pixbuf.get_height(),
			  max_width,
			  max_height)
	scaled = pixbuf.scale_simple(w, h, gtk.gdk.INTERP_BILINEAR)
	name = os.path.join(self.dirname, photo[u"id"] + suffix + u".jpg")
	scaled.save(name, "jpeg")
	return w, h

    def scale(self, width, height, max_width, max_height):
    	if width <= max_width and height <= max_height:
	    w = width
	    h = height
	elif width <= max_width:
	    h = max_height
	    w = h * width / height
	elif height <= max_height:
	    w = max_width
	    h = w * height / width
	elif width < height:
	    h = max_height
	    w = h * width / height
	else:
	    w = max_width
	    h = w * height / width
	return w, h

    def write_photo_html(self, index, w, h, tw, th):
    	photo = self.photos[index]
	folder = self.photo_to_folder[photo[u"id"]]

    	dict = TemplateDict(u"photo.", photo)
	for key in folder.keys():
	    dict[u"folder." + key] = protect(folder[key])
    	dict[u"photo.width"] = u"%d" % w
    	dict[u"photo.height"] = u"%d" % h
    	dict[u"thumb.width"] = u"%d" % tw
    	dict[u"thumb.height"] = u"%d" % th
	if len(self.folders) == 1:
	    dict[u"photo.parent"] = u"index"
    	else:
	    dict[u"photo.parent"] = protect(folder[u"id"])

    	folder_photos = folder.photos.get()
	folder_index = folder_photos.index(photo)

	if folder_index == 0:
	    dict[u"photo.prev-link"] = self.template.get("photo-prev-empty")
	else:
	    dict[u"photo.prev-link"] = \
	    	self.template.get("photo-prev") % \
		    TemplateDict(u"photo.", folder_photos[folder_index - 1])

	if folder_index == len(folder_photos) - 1:
	    dict[u"photo.next-link"] = self.template.get("photo-next-empty")
	else:
	    dict[u"photo.next-link"] = \
	    	self.template.get("photo-next") % \
		    TemplateDict(u"photo.", folder_photos[folder_index + 1])

	destname = os.path.join(self.dirname, photo[u"id"] + u".html")

	self.template.write(destname, u"photo", dict)


    def make_folder_pages(self):
    	if len(self.folders) == 1:
	    self.write_folder_page(self.folders[0], u"single-folder-index", 
	    	    	    	   u"index")
    	else:
	    for folder in self.folders:
	    	self.write_folder_page(folder, u"folder", folder[u"id"])
	    self.write_multi_folder_index_page()
    	return gtk.FALSE

    def write_folder_page(self, folder, template_name, output_name):
    	dict = TemplateDict(u"folder.", folder)
	dict[u"thumbnails"] = self.make_thumbnail_index(folder.photos.get())

    	index = self.folders.index(folder)

	if index == 0:
	    dict[u"folder.prev-link"] = self.template.get("folder-prev-empty")
	else:
	    dict[u"folder.prev-link"] = \
	    	self.template.get("folder-prev") % \
		    TemplateDict(u"folder.", self.folders[index-1])

	if index == len(self.folders) - 1:
	    dict[u"folder.next-link"] = self.template.get("folder-next-empty")
	else:
	    dict[u"folder.next-link"] = \
	    	self.template.get("folder-next") % \
		    TemplateDict(u"folder.", self.folders[index+1])

	filename = os.path.join(self.dirname, output_name + u".html")
	self.template.write(filename, template_name, dict)

    def write_multi_folder_index_page(self):
    	pieces = []
	for folder in self.folders:
	    pieces.append(self.template.get(u"multi-folder-folder") %
	    	    	  TemplateDict(u"folder.", folder))
    	dict = TemplateDict()
	dict[u"folders"] = u"".join(pieces)

	filename = os.path.join(self.dirname, u"index.html")
	self.template.write(filename, u"multi-folder-index", dict)

    def make_thumbnail_index(self, photos):
    	pieces = []
	for photo in photos:
	    dict = TemplateDict(u"photo.", photo)
	    pieces.append(self.template.get(u"thumbnail") % dict)
    	return u"".join(pieces)
