package Lire::DlfResult;

use strict;

use Carp;

use Lire::I18N qw/ mark_unicode /;
use Lire::Utils qw/ check_param /;

=pod

=head1 NAME

Lire::DlfResult - interface with the database handler for returned data

=head1 SYNOPSIS

  use Lire::DlfResult;

=head1 DESCRIPTION

=head2 new( $query, $sql )

This is the constructor method for the Lire::DlfResult class. It takes a
Lire::DlfQuery object and an SQL statement string as parameters.

This method should never be called directly. Lire::DlfQuery will return
appropriate instances of Lire::DlfResult through its execute() and
execute_summary() methods.

=cut

sub new {
    my ( $class, $query, $sql, $store ) = @_;

    check_param( $query, 'query' );
    check_param( $sql, 'sql' );
    check_param( $store, 'store' );

    my $self = bless { '_query' => $query, }, $class;
    my $dbh = $store->_dbh();
    $self->{'_sth'} = $dbh->prepare_cached( $sql );
    $self->{'_sth'}->execute ( @{$query->sql_params()} );

    $self->{'_finished'} = 0;

    return $self;
}

=pod

=head2 dlf_query()

Returns the DlfQuery object from which this DlfResult was obtained.

=cut

sub dlf_query {
    return $_[0]{'_query'};
}

=pod

=head2 next_row()

Returns the "next row" of an executed query, that is, the first record down
the last one incrementally. Returns an undefined value if called after
all the records where returned and dies on subsequent calls.

The record is returned as a HASH reference.

=cut

sub next_row {
    my $self = $_[0];

    croak "next_row() called after undef was returned by previous call"
      if ($self->{'_finished'});

    my $row = $self->{'_sth'}->fetchrow_arrayref();

    unless ( defined $row ) {
        $self->{'_finished'} = 1;
        $self->{'_sth'}->finish();

        return undef;
    }

    unless ( defined $self->{'_field_names'} ) {
        $self->{'_field_names'} = [];
        foreach my $name (@{$self->{'_sth'}{'NAME'}}) {
            $name = substr( $name, 1, -1)
              if (substr( $name, 0, 1 ) eq '"');
            push @{$self->{'_field_names'}}, $name;
        }
    }

    my $new = {};
    my $i = 0;

    foreach my $field ( @{$self->{'_field_names'}} ) {
        $new->{$field} = mark_unicode( $row->[$i++] );
    }

    return $new;
}

=pod

=head2 next_row_aref()

Same as next_row() except that the record is returned as an ARRAY reference.

=cut

sub next_row_aref {
    my $self = $_[0];

    croak "next_row_aref() called after undef was returned by previous call"
      if ($self->{'_finished'});

    my $row = $self->{'_sth'}->fetchrow_arrayref();
    unless ( defined $row ) {
        $self->{'_finished'} = 1;
        $self->{'_sth'}->finish();
        return undef;
    }

    return [ map { mark_unicode( $_ ) } @$row ];
}

sub DESTROY {
    my $self = $_[0];

    unless ( $self->{'_finished'}) {
        $self->{'_sth'}->finish();
    }
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::DlfQuery(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>
  Wolfgang Sourdeau <wolfgang@logreport.org>

=head1 VERSION

$Id: DlfResult.pm,v 1.11 2004/05/17 17:05:19 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2003 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
