package Lire::Config::Index;

use strict;

use Lire::Utils qw/ check_object_param /;
use Carp;

our %REGISTRY = ();

=pod

=head1 NAME

Lire::Config::Index - Index registry and interface

=head1 SYNOPSIS

  use base qw/Lire::Config::Index/;

  sub has_entry { my ($self, $name) = @_; ... }

  sub entries { my $self = $_[0]; ... }

  sub get_ref { my ($self, $name) = @_; ... }

=head1 DESCRIPTION

The Lire::Config::Index interface is used by the
Lire::Config::ReferenceSpec(3pm) to validate and obtain referenced
object.

=head1 INDEX REGISTRY

These are the class methods used to register and obtain Index objects.

=head2 has_index( $name )

Returns true if an index was registered under $name.

=head2 get_index( $name )

Retuurns the Index named $name. Croaks if no such index was
registered. 

=head2 set_index( $name, $index )

Registers $index under $name. Replaces any previous index registered
under the same name.

=cut

sub has_index {
    my ( $self, $name ) = @_;

    return defined $REGISTRY{$name};
}

sub get_index {
    my ( $self, $name ) = @_;

    croak "no index registered under '$name'" 
      unless $self->has_index( $name );

    return $REGISTRY{$name};
}

sub set_index {
    my ( $self, $name, $index ) = @_;

    check_object_param( $index, 'index', 'Lire::Config::Index' )
      if defined $index;

    $REGISTRY{$name} = $index;

    return;
}

=pod

=head1 INDEX INTERFACE

=head2 has_entry( $name )

Should return true if $name is defined in this index.

=head2 entries()

Should return all the entries defined in the index.

=head2 get_ref( $name )

Should return the object indexed under $name. An error should be
thrown if there is no such entry.;.

=cut

sub has_entry {
    croak ref shift, "::has_entry is unimplemented";
}

sub entries {
    croak ref shift, "::entries is unimplemented";
}

sub get_ref {
    croak ref shift, "::get_ref is unimplemented";
}

1;

__END__

=pod

=head1 SEE ALSO

 Lire::Config::Reference(3pm) Lire::Config::ReferenceSpec(3pm)
 Lire:Config::ReportConfigIndex(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: Index.pm,v 1.1 2004/07/10 23:39:53 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
