/* @(#) .del-dbmemops.c 1.11 @(#) */
/***************************************************************\
*	Copyright (c) 1999 First Step Internet Services, Inc.
*		All Rights Reserved
*	Distributed under the BSD Licenese
*
*	Module: DB
\***************************************************************/

#define _KOALAMUD_DBMEMOPS_C "@(#) nitehawk@winghove.1ststep.net|BitKeeper/deleted/.del-dbmemops.c|20000307194803|50458 @(#)"

#include "autoconf.h"
#include <sys/stat.h>
#include <sys/mman.h>

#include "version.h"
#include "koalatypes.h"
#include "log.h"
#include "conf.h"
#include "memory.h"
#include "lib/dbinternal.h"

/* This file contains various memory handling routines specific to the
 * database code. */

/* Map a database fs block on a page alignment boundry.  This function adds
 * the mapping to a table for tracking.  We will page align the starting block
 * and map at least enough blocks to include blockstart + len in the mapping
 *
 * FIXME: No checking is done to make sure that the block already exists in the
 * file.  
 */
void *mmapblockaligned(int dbfd, kdbfs_blockid_t blockstart,
						kdbfs_blockid_t len)
{
	/* List iterator */
	dbmemstate_t *ms = dbstate.dbmap;
	off_t	foffset;
	size_t	offby;
	kdbfs_blockid_t numtomap = len;
	kdbfs_blockid_t blkstart = blockstart;

	/* First we need to figure out the offset into the file. */
	foffset = blockstart * BLOCKSIZE;

	/* Page align the starting point */
	if ((offby = foffset % dbstate.pagesize) != 0)
	{
		/* We are offset by one or more blocks.  Figure out the difference and
		 * add it to the maplength, then adjust the offset */
		numtomap += offby/BLOCKSIZE;
		foffset -= offby;
		blkstart -= offby/BLOCKSIZE;
	}

	/* Search for an existing mapping that meets our criteria */
	while (ms)
	{
		/* Does this block meet the request? */
		if ((blkstart == ms->mapstart && numtomap <= ms->maplen) ||
				(blkstart >= ms->mapstart &&
				 	((blkstart + numtomap) <= (ms->mapstart + ms->maplen))))
		{
			/* The block fits, increment the reference counter and return the
			 * pointer */
			ms->refcount++;
			return ms->chunkbase + (blockstart - ms->mapstart) * BLOCKSIZE;
		}

		ms = ms->next;
	}

	/* No existing map was found, create a new one */
	{
		/* Allocate a memstate block */
		if ((ms = kmalloc(sizeof(dbmemstate_t), ALLOC_DATABASE)) == NULL)
		{
			logerr("Unable to allocate memory for mapping");
			return NULL;
		}
		/* Link the block into the block map */
		ms->next = dbstate.dbmap;
		dbstate.dbmap = ms;
		/* Fill it in */
		ms->fd = dbfd;
		ms->mapstart = blkstart;
		ms->maplen = numtomap;
		ms->refcount = 1;

		/* map it into memory */
		if ((ms->chunkbase = mmap(NULL, ms->maplen * BLOCKSIZE,
			PROT_READ | PROT_WRITE, MAP_SHARED, ms->fd,
			ms->mapstart * BLOCKSIZE)) == MAP_FAILED)
		{
			dbstate.dbmap = ms->next;
			kmfree(ms, ALLOC_DATABASE);
			logerr("Unable to map database block");
			return NULL;
		}

		/* Keep track of the ammount mapped for reporting */
		dbstate.qmapped += ms->maplen * BLOCKSIZE;
	}

	return ms->chunkbase + offby;
}

/* Unmap all mapped blocks with a reference count of 0 */
void unmapfreeblocks(void)
{
	/* List iterator */
	dbmemstate_t *ms = dbstate.dbmap;
	dbmemstate_t *msb = ms;
	dbmemstate_t *tofree;
	kdbfs_blockid_t freecount;

	/* Loop through the list */
	while (ms)
	{
		if (ms->refcount == 0)
		{
			tofree = ms;

			/* Count the number of blocks we free for reporting */
			freecount += tofree->maplen;

			/* This block is the list head */
			if (tofree == dbstate.dbmap)
			{
				dbstate.dbmap = tofree->next;
				msb = dbstate.dbmap;
				ms = dbstate.dbmap;
			}
			else
			{
				msb->next = ms->next;
				ms = ms->next;
			}

			/* At this point, the iterator and back pointer have been
			 * advanced.  All that remains is to unmap 'tofree' and free the
			 * state node */
			munmap(tofree->chunkbase, tofree->maplen * BLOCKSIZE);
			dbstate.qmapped -= tofree->maplen * BLOCKSIZE;
			kmfree(tofree, ALLOC_DATABASE);
		}
		else
		{
			/* This block is still being referenced */
			msb = ms;
			ms = ms->next;
		}
	}
}

/* mapfree
 * 	Release a block mapping
 * 		This does not actually unmap the data, it only decrements the
 * 		reference count.
 */
void mapfree(void *blkaddr)
{
	/* List iterator */
	dbmemstate_t *ms = dbstate.dbmap;

	/* Loop through the linked list of mappings and locate the block */
	while (ms)
	{
		/* Is the address in the blocks range */
		if (blkaddr >= ms->chunkbase &&
				blkaddr <= (ms->chunkbase + BLOCKSIZE * ms->maplen))
		{
			ms->refcount -= ms->refcount ? 1 : 0;
			return;
		}

		ms = ms->next;
	}
}

/* closemapfile
 *	Unmap all mappings to the file and close its file handle
 */
void closemapfile(int dbfd)
{
	/* List iterator */
	dbmemstate_t *ms = dbstate.dbmap;

	/* Set the refcount on all mappings from this fd to 0 */
	while (ms)
	{
		if (ms->fd == dbfd)
		{
			ms->refcount = 0;
		}
		ms = ms->next;
	}

	/* Unmap all mapped blocks */
	unmapfreeblocks();

	/* Close the file */
	close(dbfd);
}
